import React, { useState, useEffect } from 'react';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';
import Button from '../../../components/ui/Button';
import Icon from '../../../components/AppIcon';
import Image from '../../../components/AppImage';
import StatusIndicatorBadges from '../../../components/ui/StatusIndicatorBadges';

const RentalContractTab = ({ contractData, onUpdate }) => {
  const [formData, setFormData] = useState(contractData);
  const [isEditing, setIsEditing] = useState(false);
  const [hasChanges, setHasChanges] = useState(false);
  const [isSaving, setIsSaving] = useState(false);
  const [ufRate] = useState(37245.67); // Mock UF rate

  const contractTypes = [
    { value: 'indefinite', label: 'Indefinido' },
    { value: 'fixed-term', label: 'Plazo Fijo' },
    { value: 'temporary', label: 'Temporal' }
  ];

  const paymentMethods = [
    { value: 'bank-transfer', label: 'Transferencia Bancaria' },
    { value: 'check', label: 'Cheque' },
    { value: 'cash', label: 'Efectivo' },
    { value: 'deposit', label: 'Depósito Bancario' }
  ];

  const adjustmentTypes = [
    { value: 'uf', label: 'UF (Unidad de Fomento)' },
    { value: 'ipc', label: 'IPC (Índice de Precios al Consumidor)' },
    { value: 'fixed', label: 'Porcentaje Fijo' },
    { value: 'none', label: 'Sin Reajuste' }
  ];

  useEffect(() => {
    const hasDataChanged = JSON.stringify(formData) !== JSON.stringify(contractData);
    setHasChanges(hasDataChanged);
  }, [formData, contractData]);

  const handleInputChange = (field, value) => {
    setFormData(prev => ({
      ...prev,
      [field]: value
    }));
  };

  const handleSave = async () => {
    setIsSaving(true);
    try {
      await new Promise(resolve => setTimeout(resolve, 1000));
      onUpdate(formData);
      setIsEditing(false);
      setHasChanges(false);
    } catch (error) {
      console.error('Error saving contract data:', error);
    } finally {
      setIsSaving(false);
    }
  };

  const handleCancel = () => {
    setFormData(contractData);
    setIsEditing(false);
    setHasChanges(false);
  };

  const formatCurrency = (amount) => {
    return new Intl.NumberFormat('es-CL', {
      style: 'currency',
      currency: 'CLP',
      minimumFractionDigits: 0
    })?.format(amount);
  };

  const formatUF = (amount) => {
    // Convert to number first and handle invalid values
    const numAmount = parseFloat(amount);
    if (isNaN(numAmount)) return '0.00 UF';
    return `${numAmount?.toFixed(2)} UF`;
  };

  const convertToUF = (clpAmount) => {
    // Convert to number and handle invalid values
    const numAmount = parseFloat(clpAmount);
    if (isNaN(numAmount) || numAmount === 0) return 0;
    return (numAmount / ufRate);
  };

  const convertToCLP = (ufAmount) => {
    // Convert to number and handle invalid values
    const numAmount = parseFloat(ufAmount);
    if (isNaN(numAmount) || numAmount === 0) return 0;
    return Math.round(numAmount * ufRate);
  };

  return (
    <div className="space-y-6">
      {/* Header with Edit Controls */}
      <div className="flex items-center justify-between">
        <div className="flex items-center space-x-3">
          <h3 className="text-lg font-semibold text-gray-900">Contrato de Arriendo</h3>
          <StatusIndicatorBadges status={formData?.status} />
        </div>
        
        <div className="flex items-center space-x-2">
          {isEditing ? (
            <>
              <Button
                variant="outline"
                size="sm"
                onClick={handleCancel}
                disabled={isSaving}
              >
                Cancelar
              </Button>
              <Button
                variant="default"
                size="sm"
                onClick={handleSave}
                loading={isSaving}
                disabled={!hasChanges}
                iconName="Save"
                iconPosition="left"
              >
                Guardar
              </Button>
            </>
          ) : (
            <Button
              variant="outline"
              size="sm"
              onClick={() => setIsEditing(true)}
              iconName="Edit"
              iconPosition="left"
            >
              Editar
            </Button>
          )}
        </div>
      </div>

      {/* Contract Configuration - Fixed section */}
      <div className="glass-card p-6 rounded-lg">
        <h4 className="text-lg font-medium text-gray-900 mb-4">
          <Icon name="Settings" size={20} className="inline mr-2" />
          Configuración del Contrato
        </h4>
        
        {/* Contract Details */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <Select
            label="Tipo de Contrato"
            options={contractTypes}
            value={formData?.contractType}
            onChange={(value) => handleInputChange('contractType', value)}
            disabled={!isEditing}
            required
          />

          <Input
            label="Fecha de Inicio"
            type="date"
            value={formData?.startDate}
            onChange={(e) => handleInputChange('startDate', e?.target?.value)}
            disabled={!isEditing}
            required
          />

          <Input
            label="Fecha de Término"
            type="date"
            value={formData?.endDate}
            onChange={(e) => handleInputChange('endDate', e?.target?.value)}
            disabled={!isEditing}
            required
          />

          <Input
            label="Fecha de Pago Mensual"
            type="number"
            value={formData?.paymentDueDay}
            onChange={(e) => handleInputChange('paymentDueDay', e?.target?.value)}
            disabled={!isEditing}
            min="1"
            max="31"
            placeholder="5"
            required
          />

          <Input
            label="Duración (meses)"
            type="number"
            value={formData?.duration}
            onChange={(e) => handleInputChange('duration', e?.target?.value)}
            disabled={!isEditing}
            min="1"
            placeholder="12"
          />
        </div>

        {/* Contract Status */}
        <div className="mt-6 p-4 bg-green-50 border border-green-200 rounded-lg">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-2">
              <Icon name="CheckCircle" size={20} className="text-green-600" />
              <span className="text-green-800 font-medium">
                Contrato correctamente configurado y asociado a la propiedad
              </span>
            </div>
            <div className="text-sm text-green-600">
              ID: {formData?.id || 'contract-001'}
            </div>
          </div>
        </div>
      </div>

      {/* Property-Contract Association */}
      <div className="glass-card p-6 rounded-lg">
        <h4 className="text-lg font-medium text-gray-900 mb-4">
          <Icon name="Link" size={20} className="inline mr-2" />
          Asociación Contrato-Propiedad
        </h4>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div className="flex items-center space-x-4 p-4 bg-blue-50 rounded-lg">
            <div className="w-12 h-12 bg-blue-500 text-white rounded-full flex items-center justify-center">
              <Icon name="Home" size={20} />
            </div>
            <div>
              <div className="font-medium text-blue-900">Propiedad Asociada</div>
              <div className="text-sm text-blue-700">Av. Providencia 1234, Depto 502</div>
              <div className="text-xs text-blue-600">ID: PROP-2024-001</div>
            </div>
          </div>
          
          <div className="flex items-center space-x-4 p-4 bg-green-50 rounded-lg">
            <div className="w-12 h-12 bg-green-500 text-white rounded-full flex items-center justify-center">
              <Icon name="UserCheck" size={20} />
            </div>
            <div>
              <div className="font-medium text-green-900">Propietario</div>
              <div className="text-sm text-green-700">Carlos Mendoza Ruiz</div>
              <div className="text-xs text-green-600">RUT: 12.345.678-9</div>
            </div>
          </div>
        </div>
        
        {/* Tenant Association in Contract */}
        {formData?.tenant && (
          <div className="mt-4 p-4 bg-purple-50 border border-purple-200 rounded-lg">
            <div className="flex items-center space-x-4">
              <div className="w-12 h-12 bg-purple-500 text-white rounded-full flex items-center justify-center">
                <Icon name="User" size={20} />
              </div>
              <div className="flex-1">
                <div className="font-medium text-purple-900">Inquilino Asociado</div>
                <div className="text-sm text-purple-700">{formData?.tenant?.fullName}</div>
                <div className="text-xs text-purple-600">
                  RUT: {formData?.tenant?.rut} | Teléfono: {formData?.tenant?.phone}
                </div>
              </div>
              <div className="text-right">
                <div className="text-sm font-medium text-purple-900">Desde</div>
                <div className="text-sm text-purple-700">{formData?.startDate}</div>
              </div>
            </div>
          </div>
        )}
      </div>

      {/* Current Tenant Information */}
      {formData?.tenant && (
        <div className="glass-card p-6 rounded-lg">
          <h4 className="text-md font-medium text-gray-900 mb-4">Inquilino Actual</h4>
          <div className="flex items-start space-x-4">
            <div className="flex-shrink-0">
              <div className="w-12 h-12 bg-gray-200 rounded-full overflow-hidden">
                {formData?.tenant?.avatar ? (
                  <Image 
                    src={formData?.tenant?.avatar} 
                    alt={formData?.tenant?.fullName}
                    className="w-full h-full object-cover"
                  />
                ) : (
                  <div className="w-full h-full bg-primary/10 flex items-center justify-center">
                    <Icon name="User" size={20} className="text-primary" />
                  </div>
                )}
              </div>
            </div>
            
            <div className="flex-1 min-w-0">
              <h5 className="text-md font-medium text-gray-900">{formData?.tenant?.fullName}</h5>
              <p className="text-sm text-gray-600">{formData?.tenant?.email}</p>
              <p className="text-sm text-gray-600">{formData?.tenant?.phone}</p>
              <div className="mt-2 flex items-center space-x-4 text-sm">
                <span className="text-gray-500">RUT: <span className="font-medium text-gray-900">{formData?.tenant?.rut}</span></span>
                <span className="text-gray-500">Desde: <span className="font-medium text-gray-900">{formData?.startDate}</span></span>
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Rental Amounts */}
      <div className="space-y-4">
        <h4 className="text-md font-medium text-gray-900">Montos de Arriendo</h4>
        
        {/* UF Rate Display */}
        <div className="glass-card p-3 rounded-lg bg-blue-50 border border-blue-200">
          <div className="flex items-center justify-between text-sm">
            <span className="text-blue-700 font-medium">Valor UF Actual:</span>
            <span className="text-blue-900 font-bold">{formatCurrency(ufRate)}</span>
          </div>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div className="space-y-4">
            <Input
              label="Arriendo Mensual (CLP)"
              type="number"
              value={formData?.monthlyRentCLP}
              onChange={(e) => handleInputChange('monthlyRentCLP', e?.target?.value)}
              disabled={!isEditing}
              min="0"
              placeholder="0"
            />
            
            {formData?.monthlyRentCLP && (
              <div className="text-sm text-gray-600">
                Equivalente: {formatUF(convertToUF(formData?.monthlyRentCLP))}
              </div>
            )}
          </div>

          <div className="space-y-4">
            <Input
              label="Arriendo Mensual (UF)"
              type="number"
              value={formData?.monthlyRentUF}
              onChange={(e) => handleInputChange('monthlyRentUF', e?.target?.value)}
              disabled={!isEditing}
              min="0"
              step="0.01"
              placeholder="0.00"
            />
            
            {formData?.monthlyRentUF && (
              <div className="text-sm text-gray-600">
                Equivalente: {formatCurrency(convertToCLP(formData?.monthlyRentUF))}
              </div>
            )}
          </div>

          <Input
            label="Garantía (CLP)"
            type="number"
            value={formData?.securityDeposit}
            onChange={(e) => handleInputChange('securityDeposit', e?.target?.value)}
            disabled={!isEditing}
            min="0"
            placeholder="0"
          />

          <Input
            label="Gastos Comunes (CLP)"
            type="number"
            value={formData?.commonExpenses}
            onChange={(e) => handleInputChange('commonExpenses', e?.target?.value)}
            disabled={!isEditing}
            min="0"
            placeholder="0"
          />
        </div>
      </div>

      {/* Enhanced Payment Configuration with Utility Integration */}
      <div className="space-y-4">
        <h4 className="text-md font-medium text-gray-900">Configuración de Pagos y Cuentas</h4>
        
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <Input
            label="Día de Vencimiento"
            type="number"
            value={formData?.paymentDueDay}
            onChange={(e) => handleInputChange('paymentDueDay', e?.target?.value)}
            disabled={!isEditing}
            min="1"
            max="31"
            placeholder="5"
            required
          />

          <Select
            label="Método de Pago"
            options={paymentMethods}
            value={formData?.paymentMethod}
            onChange={(value) => handleInputChange('paymentMethod', value)}
            disabled={!isEditing}
          />

          <Select
            label="Tipo de Reajuste"
            options={adjustmentTypes}
            value={formData?.adjustmentType}
            onChange={(value) => handleInputChange('adjustmentType', value)}
            disabled={!isEditing}
          />
        </div>

        {/* Utility Bills Integration */}
        <div className="glass-card p-4 rounded-lg bg-blue-50 border border-blue-200">
          <h5 className="font-medium text-blue-900 mb-3">
            <Icon name="Receipt" size={16} className="inline mr-2" />
            Integración con Cuentas Básicas
          </h5>
          
          <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-4">
            <div className="text-center">
              <div className="text-lg font-bold text-blue-700">{formatCurrency(25000)}</div>
              <div className="text-sm text-blue-600">Agua</div>
            </div>
            <div className="text-center">
              <div className="text-lg font-bold text-yellow-700">{formatCurrency(45000)}</div>
              <div className="text-sm text-yellow-600">Electricidad</div>
            </div>
            <div className="text-center">
              <div className="text-lg font-bold text-orange-700">{formatCurrency(18000)}</div>
              <div className="text-sm text-orange-600">Gas</div>
            </div>
            <div className="text-center">
              <div className="text-lg font-bold text-gray-700">{formatCurrency(45000)}</div>
              <div className="text-sm text-gray-600">Gastos Comunes</div>
            </div>
          </div>
          
          <div className="p-3 bg-white rounded border border-blue-300">
            <div className="flex items-center justify-between text-sm">
              <span className="text-blue-800 font-medium">
                ✓ Los montos de cuentas básicas se incluyen automáticamente en los comprobantes
              </span>
              <span className="text-blue-700 font-bold">
                Total: {formatCurrency(133000)}
              </span>
            </div>
          </div>
        </div>
      </div>

      {/* Contract Terms */}
      <div className="space-y-4">
        <h4 className="text-md font-medium text-gray-900">Términos del Contrato</h4>
        
        <div className="glass-card p-4 rounded-lg">
          <textarea
            value={formData?.specialTerms || ''}
            onChange={(e) => handleInputChange('specialTerms', e?.target?.value)}
            disabled={!isEditing}
            rows={4}
            className={`
              w-full px-3 py-2 border rounded-lg text-sm
              ${!isEditing 
                ? 'bg-gray-50 border-gray-200 text-gray-600' :'bg-white border-gray-300 text-gray-900 focus:ring-2 focus:ring-primary focus:border-transparent'
              }
              transition-colors duration-150
            `}
            placeholder="Términos especiales, cláusulas adicionales, restricciones, etc..."
          />
        </div>
      </div>

      {/* Contract Summary */}
      <div className="glass-card p-4 rounded-lg">
        <h4 className="text-md font-medium text-gray-900 mb-3">Resumen del Contrato</h4>
        <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-center">
          <div>
            <div className="text-2xl font-bold text-primary">
              {formatCurrency(formData?.monthlyRentCLP || 0)}
            </div>
            <div className="text-sm text-gray-600">Arriendo Mensual</div>
          </div>
          <div>
            <div className="text-2xl font-bold text-success">
              {formatCurrency(formData?.securityDeposit || 0)}
            </div>
            <div className="text-sm text-gray-600">Garantía</div>
          </div>
          <div>
            <div className="text-2xl font-bold text-secondary">
              {formData?.duration || 0}
            </div>
            <div className="text-sm text-gray-600">Meses</div>
          </div>
          <div>
            <div className="text-2xl font-bold text-warning">
              {formData?.paymentDueDay || '-'}
            </div>
            <div className="text-sm text-gray-600">Día Vencimiento</div>
          </div>
        </div>
      </div>

      {/* Auto-save Indicator */}
      {hasChanges && isEditing && (
        <div className="flex items-center space-x-2 text-sm text-warning">
          <Icon name="Clock" size={16} />
          <span>Cambios sin guardar</span>
        </div>
      )}
    </div>
  );
};

export default RentalContractTab;