import React, { useState, useEffect } from 'react';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';
import Button from '../../../components/ui/Button';
import Icon from '../../../components/AppIcon';
import { CHILEAN_UTILITY_COMPANIES, fetchUtilityBill, generateMockUtilityData } from '../../../utils/chileanUtilities';

const ServiceConnectionsTab = ({ servicesData, onUpdate }) => {
  const [formData, setFormData] = useState(servicesData);
  const [isEditing, setIsEditing] = useState(false);
  const [hasChanges, setHasChanges] = useState(false);
  const [isSaving, setIsSaving] = useState(false);
  const [apiConnections, setApiConnections] = useState({});
  const [billingData, setBillingData] = useState({});

  useEffect(() => {
    const hasDataChanged = JSON.stringify(formData) !== JSON.stringify(servicesData);
    setHasChanges(hasDataChanged);
  }, [formData, servicesData]);

  const handleInputChange = (service, field, value) => {
    setFormData(prev => ({
      ...prev,
      [service]: {
        ...prev?.[service],
        [field]: value
      }
    }));
  };

  const handleSave = async () => {
    setIsSaving(true);
    try {
      await new Promise(resolve => setTimeout(resolve, 1000));
      onUpdate(formData);
      setIsEditing(false);
      setHasChanges(false);
    } catch (error) {
      console.error('Error saving services data:', error);
    } finally {
      setIsSaving(false);
    }
  };

  const handleCancel = () => {
    setFormData(servicesData);
    setIsEditing(false);
    setHasChanges(false);
  };

  const formatCurrency = (amount) => {
    return new Intl.NumberFormat('es-CL', {
      style: 'currency',
      currency: 'CLP',
      minimumFractionDigits: 0
    })?.format(amount);
  };

  // Connect to utility API - REQUIRED BY USER
  const handleAPIConnection = async (serviceType, companyId, accountNumber) => {
    try {
      console.log(`Conectando API para ${serviceType}:`, { companyId, accountNumber });
      
      // Mock API connection for demonstration
      const mockConnection = await new Promise(resolve => {
        setTimeout(() => {
          resolve({
            connected: true,
            company: CHILEAN_UTILITY_COMPANIES?.[serviceType]?.find(c => c?.id === companyId)?.name,
            accountNumber,
            lastBill: generateMockUtilityData(serviceType, companyId)
          });
        }, 2000);
      });
      
      setApiConnections(prev => ({
        ...prev,
        [serviceType]: mockConnection
      }));
      
      // Update form data with API connection status
      handleInputChange(serviceType, 'apiConnected', true);
      handleInputChange(serviceType, 'lastSync', new Date()?.toISOString());
      
      console.log('API conectada exitosamente:', mockConnection);
      
    } catch (error) {
      console.error(`Error conectando API para ${serviceType}:`, error);
    }
  };

  // Fetch utility bill automatically - REQUIRED BY USER
  const handleFetchBill = async (serviceType, companyId, accountNumber) => {
    try {
      const currentDate = new Date();
      const bill = await fetchUtilityBill(
        companyId,
        accountNumber,
        serviceType,
        currentDate?.getMonth(),
        currentDate?.getFullYear()
      );
      
      setBillingData(prev => ({
        ...prev,
        [serviceType]: bill
      }));
      
      // Update monthly cost with fetched amount
      handleInputChange(serviceType, 'monthlyCost', bill?.amount);
      handleInputChange(serviceType, 'lastBillDate', bill?.dueDate);
      
      console.log(`Factura obtenida para ${serviceType}:`, bill);
      
    } catch (error) {
      console.error(`Error obteniendo factura para ${serviceType}:`, error);
    }
  };

  const serviceTypes = [
    {
      key: 'electricity',
      name: 'Electricidad',
      icon: 'Zap',
      color: 'yellow',
      companies: CHILEAN_UTILITY_COMPANIES?.electricity || []
    },
    {
      key: 'water',
      name: 'Agua',
      icon: 'Droplets',
      color: 'blue',
      companies: CHILEAN_UTILITY_COMPANIES?.water || []
    },
    {
      key: 'gas',
      name: 'Gas',
      icon: 'Flame',
      color: 'orange',
      companies: CHILEAN_UTILITY_COMPANIES?.gas || []
    },
    {
      key: 'internet',
      name: 'Internet',
      icon: 'Wifi',
      color: 'purple',
      companies: [
        { id: 'movistar', name: 'Movistar', apiSupported: false },
        { id: 'vtr', name: 'VTR', apiSupported: false },
        { id: 'entel', name: 'Entel', apiSupported: false },
        { id: 'claro', name: 'Claro', apiSupported: false }
      ]
    }
  ];

  const paymentResponsibilityOptions = [
    { value: 'tenant', label: 'Inquilino' },
    { value: 'owner', label: 'Propietario' },
    { value: 'shared', label: 'Compartido' }
  ];

  const statusOptions = [
    { value: 'active', label: 'Activo' },
    { value: 'inactive', label: 'Inactivo' },
    { value: 'suspended', label: 'Suspendido' }
  ];

  return (
    <div className="space-y-6">
      {/* Header with Edit Controls */}
      <div className="flex items-center justify-between">
        <div className="flex items-center space-x-3">
          <h3 className="text-lg font-semibold text-gray-900">Conexiones de Servicios</h3>
          <div className="text-sm text-gray-600">
            Gestiona las conexiones automáticas con compañías de servicios básicos
          </div>
        </div>
        
        <div className="flex items-center space-x-2">
          {isEditing ? (
            <>
              <Button
                variant="outline"
                size="sm"
                onClick={handleCancel}
                disabled={isSaving}
              >
                Cancelar
              </Button>
              <Button
                variant="default"
                size="sm"
                onClick={handleSave}
                loading={isSaving}
                disabled={!hasChanges}
                iconName="Save"
                iconPosition="left"
              >
                Guardar
              </Button>
            </>
          ) : (
            <Button
              variant="outline"
              size="sm"
              onClick={() => setIsEditing(true)}
              iconName="Edit"
              iconPosition="left"
            >
              Editar
            </Button>
          )}
        </div>
      </div>

      {/* API Integration Notice - REQUIRED BY USER */}
      <div className="glass-card p-4 rounded-lg bg-blue-50 border border-blue-200">
        <div className="flex items-start space-x-3">
          <Icon name="Info" size={20} className="text-blue-600 mt-0.5" />
          <div>
            <h4 className="font-medium text-blue-900 mb-1">
              Integración Automática con Servicios Básicos
            </h4>
            <p className="text-sm text-blue-700 mb-3">
              Conecta las cuentas de servicios básicos mediante API para actualizar automáticamente los saldos según el número de cliente. 
              Compatible con todas las compañías de agua, luz y gas de Chile.
            </p>
            <div className="text-xs text-blue-600">
              ✓ Actualización automática de montos | ✓ Sincronización mensual | ✓ Integración segura
            </div>
          </div>
        </div>
      </div>

      {/* Services Configuration */}
      <div className="space-y-6">
        {serviceTypes?.map((serviceType) => {
          const serviceData = formData?.[serviceType?.key] || {};
          const isConnected = apiConnections?.[serviceType?.key]?.connected;
          const billData = billingData?.[serviceType?.key];
          
          return (
            <div key={serviceType?.key} className="glass-card p-6 rounded-lg">
              <div className="flex items-center justify-between mb-4">
                <div className="flex items-center space-x-3">
                  <div className={`w-10 h-10 rounded-full flex items-center justify-center bg-${serviceType?.color}-100`}>
                    <Icon name={serviceType?.icon} size={20} className={`text-${serviceType?.color}-600`} />
                  </div>
                  <div>
                    <h4 className="text-lg font-medium text-gray-900">{serviceType?.name}</h4>
                    <div className="flex items-center space-x-2 mt-1">
                      <div className={`
                        px-2 py-1 rounded-full text-xs font-medium
                        ${serviceData?.status === 'active' ?'bg-green-100 text-green-800' 
                          : serviceData?.status === 'suspended' ?'bg-yellow-100 text-yellow-800' :'bg-gray-100 text-gray-800'
                        }
                      `}>
                        {statusOptions?.find(opt => opt?.value === serviceData?.status)?.label || 'No configurado'}
                      </div>
                      
                      {isConnected && (
                        <div className="px-2 py-1 bg-blue-100 text-blue-800 rounded-full text-xs font-medium">
                          API Conectada
                        </div>
                      )}
                    </div>
                  </div>
                </div>

                {/* API Connection Controls */}
                {serviceType?.companies?.length > 0 && (
                  <div className="flex items-center space-x-2">
                    <Button
                      variant="outline"
                      size="sm"
                      iconName="RefreshCw"
                      iconPosition="left"
                      onClick={() => handleFetchBill(
                        serviceType?.key,
                        serviceData?.provider,
                        serviceData?.accountNumber
                      )}
                      disabled={!serviceData?.provider || !serviceData?.accountNumber}
                    >
                      Actualizar Saldo
                    </Button>
                    
                    <Button
                      variant={isConnected ? "outline" : "default"}
                      size="sm"
                      iconName={isConnected ? "Unplug" : "Plug"}
                      iconPosition="left"
                      onClick={() => handleAPIConnection(
                        serviceType?.key,
                        serviceData?.provider,
                        serviceData?.accountNumber
                      )}
                      disabled={!serviceData?.provider || !serviceData?.accountNumber}
                    >
                      {isConnected ? 'Desconectar API' : 'Conectar API'}
                    </Button>
                  </div>
                )}
              </div>

              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                <Select
                  label="Estado"
                  options={statusOptions}
                  value={serviceData?.status}
                  onChange={(value) => handleInputChange(serviceType?.key, 'status', value)}
                  disabled={!isEditing}
                />

                {/* Provider Selection with ALL Chilean Companies */}
                {serviceType?.companies?.length > 0 && (
                  <Select
                    label="Compañía Proveedora"
                    options={serviceType?.companies?.map(company => ({
                      value: company?.id,
                      label: `${company?.name}${company?.apiSupported ? ' (API)' : ''}`
                    }))}
                    value={serviceData?.provider}
                    onChange={(value) => handleInputChange(serviceType?.key, 'provider', value)}
                    disabled={!isEditing}
                  />
                )}

                <Input
                  label="Número de Cliente"
                  value={serviceData?.accountNumber}
                  onChange={(e) => handleInputChange(serviceType?.key, 'accountNumber', e?.target?.value)}
                  disabled={!isEditing}
                  placeholder="Número de cuenta o cliente"
                />

                <Select
                  label="Responsable de Pago"
                  options={paymentResponsibilityOptions}
                  value={serviceData?.paymentResponsibility}
                  onChange={(value) => handleInputChange(serviceType?.key, 'paymentResponsibility', value)}
                  disabled={!isEditing}
                />

                <Input
                  label="Costo Mensual Promedio"
                  type="number"
                  value={serviceData?.monthlyCost}
                  onChange={(e) => handleInputChange(serviceType?.key, 'monthlyCost', e?.target?.value)}
                  disabled={!isEditing || isConnected}
                  placeholder="0"
                />

                <Input
                  label="Teléfono de Contacto"
                  value={serviceData?.contactPhone}
                  onChange={(e) => handleInputChange(serviceType?.key, 'contactPhone', e?.target?.value)}
                  disabled={!isEditing}
                  placeholder="+56 2 2XXX XXXX"
                />
              </div>

              {/* API Connection Status and Bill Data */}
              {isConnected && billData && (
                <div className="mt-4 p-4 bg-green-50 border border-green-200 rounded-lg">
                  <h5 className="font-medium text-green-900 mb-2">
                    <Icon name="CheckCircle" size={16} className="inline mr-2" />
                    Conexión API Activa
                  </h5>
                  <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm">
                    <div>
                      <div className="text-green-700 font-medium">Última Factura</div>
                      <div className="text-green-900">{formatCurrency(billData?.amount)}</div>
                    </div>
                    <div>
                      <div className="text-green-700 font-medium">Vencimiento</div>
                      <div className="text-green-900">
                        {billData?.dueDate ? new Date(billData?.dueDate)?.toLocaleDateString('es-CL') : 'N/A'}
                      </div>
                    </div>
                    <div>
                      <div className="text-green-700 font-medium">Consumo</div>
                      <div className="text-green-900">{billData?.consumption || 'N/A'}</div>
                    </div>
                    <div>
                      <div className="text-green-700 font-medium">Estado</div>
                      <div className="text-green-900">✓ Sincronizado</div>
                    </div>
                  </div>
                </div>
              )}

              {/* Service Notes */}
              <div className="mt-4">
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  Notas del Servicio
                </label>
                <textarea
                  value={serviceData?.notes || ''}
                  onChange={(e) => handleInputChange(serviceType?.key, 'notes', e?.target?.value)}
                  disabled={!isEditing}
                  rows={2}
                  className={`
                    w-full px-3 py-2 border rounded-lg text-sm
                    ${!isEditing 
                      ? 'bg-gray-50 border-gray-200 text-gray-600' :'bg-white border-gray-300 text-gray-900 focus:ring-2 focus:ring-primary focus:border-transparent'
                    }
                    transition-colors duration-150
                  `}
                  placeholder="Información adicional sobre el servicio..."
                />
              </div>
            </div>
          );
        })}
      </div>

      {/* Service Summary */}
      <div className="glass-card p-6 rounded-lg">
        <h4 className="text-lg font-medium text-gray-900 mb-4">Resumen de Servicios</h4>
        
        <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-center">
          <div>
            <div className="text-2xl font-bold text-primary">
              {Object.values(formData || {})?.filter(service => service?.status === 'active')?.length || 0}
            </div>
            <div className="text-sm text-gray-600">Servicios Activos</div>
          </div>
          <div>
            <div className="text-2xl font-bold text-success">
              {Object.keys(apiConnections || {})?.length || 0}
            </div>
            <div className="text-sm text-gray-600">APIs Conectadas</div>
          </div>
          <div>
            <div className="text-2xl font-bold text-secondary">
              {formatCurrency(
                Object.values(formData || {})
                  ?.reduce((total, service) => total + (parseInt(service?.monthlyCost) || 0), 0)
              )}
            </div>
            <div className="text-sm text-gray-600">Costo Mensual Total</div>
          </div>
          <div>
            <div className="text-2xl font-bold text-warning">
              {Object.values(formData || {})?.filter(service => service?.paymentResponsibility === 'owner')?.length || 0}
            </div>
            <div className="text-sm text-gray-600">Pagados por Propietario</div>
          </div>
        </div>
      </div>

      {/* Auto-save Indicator */}
      {hasChanges && isEditing && (
        <div className="flex items-center space-x-2 text-sm text-warning">
          <Icon name="Clock" size={16} />
          <span>Cambios sin guardar</span>
        </div>
      )}
    </div>
  );
};

export default ServiceConnectionsTab;