import React, { useState } from 'react';
import Button from '../../../components/ui/Button';
import Icon from '../../../components/AppIcon';
import { generateTenantVoucherPDF, generateOwnerVoucherPDF } from '../../../utils/voucherPDFGenerator';

const VoucherDownloadModal = ({ 
  isOpen, 
  onClose, 
  payment,
  propertyData = {},
  tenantData = {},
  ownerData = {}
}) => {
  const [selectedType, setSelectedType] = useState('');
  const [isGenerating, setIsGenerating] = useState(false);

  const handleDownload = async (voucherType) => {
    setIsGenerating(true);
    try {
      if (voucherType === 'tenant') {
        await generateTenantVoucherPDF(payment, propertyData, tenantData);
      } else if (voucherType === 'owner') {
        await generateOwnerVoucherPDF(payment, propertyData, ownerData);
      }
    } catch (error) {
      console.error('Error generating voucher:', error);
      alert('Error al generar el comprobante. Por favor, inténtelo de nuevo.');
    }
    setIsGenerating(false);
    onClose();
  };

  const handleQuickDownload = async (voucherType) => {
    await handleDownload(voucherType);
  };

  if (!isOpen || !payment) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4 z-50">
      <div className="bg-white rounded-xl max-w-md w-full shadow-2xl">
        {/* Header */}
        <div className="flex items-center justify-between p-6 border-b border-gray-200">
          <h3 className="text-lg font-semibold text-gray-900">
            Descargar Comprobante
          </h3>
          <button
            onClick={onClose}
            className="text-gray-400 hover:text-gray-600 transition-colors duration-150"
          >
            <Icon name="X" size={24} />
          </button>
        </div>

        {/* Content */}
        <div className="p-6 space-y-4">
          <div className="text-center mb-6">
            <div className="text-sm text-gray-600 mb-2">Período: {payment?.month}</div>
            <div className="text-sm text-gray-600">Comprobante: {payment?.voucherNumber}</div>
          </div>

          <div className="space-y-3">
            {/* Tenant Voucher Option */}
            <div className="border-2 border-blue-200 rounded-lg p-4 hover:border-blue-400 transition-colors duration-150">
              <div className="flex items-center justify-between">
                <div className="flex-1">
                  <h4 className="font-medium text-gray-900 mb-1">Comprobante para Arrendatario</h4>
                  <p className="text-sm text-gray-600">
                    Incluye detalles de pago del arrendatario con conceptos y montos pagados
                  </p>
                  <div className="mt-2 text-xs text-blue-600">
                    • Monto del arriendo y gastos
                    • Descuentos aplicados al arrendatario
                    • Fecha de pago y vencimiento
                    • Método de pago utilizado
                  </div>
                </div>
                <div className="ml-4">
                  <Icon name="User" size={32} className="text-blue-500" />
                </div>
              </div>
              <div className="mt-4">
                <Button
                  variant="default"
                  size="sm"
                  onClick={() => handleQuickDownload('tenant')}
                  disabled={isGenerating}
                  iconName="Download"
                  iconPosition="left"
                  className="w-full bg-blue-600 hover:bg-blue-700"
                >
                  {isGenerating ? 'Generando...' : 'Descargar para Arrendatario'}
                </Button>
              </div>
            </div>

            {/* Owner Voucher Option */}
            <div className="border-2 border-green-200 rounded-lg p-4 hover:border-green-400 transition-colors duration-150">
              <div className="flex items-center justify-between">
                <div className="flex-1">
                  <h4 className="font-medium text-gray-900 mb-1">Comprobante para Propietario</h4>
                  <p className="text-sm text-gray-600">
                    Incluye detalles del pago del arrendatario más descuentos y créditos del propietario
                  </p>
                  <div className="mt-2 text-xs text-green-600">
                    • Todos los conceptos del arrendatario
                    • Descuentos y créditos del propietario
                    • Comisiones de administración
                    • Monto neto a recibir
                  </div>
                </div>
                <div className="ml-4">
                  <Icon name="Building" size={32} className="text-green-500" />
                </div>
              </div>
              <div className="mt-4">
                <Button
                  variant="default"
                  size="sm"
                  onClick={() => handleQuickDownload('owner')}
                  disabled={isGenerating}
                  iconName="Download"
                  iconPosition="left"
                  className="w-full bg-green-600 hover:bg-green-700"
                >
                  {isGenerating ? 'Generando...' : 'Descargar para Propietario'}
                </Button>
              </div>
            </div>
          </div>

          <div className="text-center pt-4 border-t border-gray-200">
            <p className="text-xs text-gray-500">
              Los comprobantes se descargarán en formato PDF con diseño moderno y profesional
            </p>
          </div>
        </div>

        {/* Footer */}
        <div className="flex items-center justify-center p-6 border-t border-gray-200">
          <Button
            variant="outline"
            onClick={onClose}
            disabled={isGenerating}
          >
            Cancelar
          </Button>
        </div>
      </div>
    </div>
  );
};

export default VoucherDownloadModal;