import React, { useState, useEffect } from 'react';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';
import Button from '../../../components/ui/Button';
import Icon from '../../../components/AppIcon';

const VoucherEditModal = ({ 
  isOpen, 
  onClose, 
  voucher = null, 
  propertyData = {}, 
  onSave 
}) => {
  const [formData, setFormData] = useState({
    month: '',
    dueDate: '',
    rentAmount: 850000,
    gasAmount: 35000,
    commonExpenses: 45000,
    waterAmount: 25000,
    electricityAmount: 40000,
    internetAmount: 30000,
    tenantDiscounts: [{ description: '', amount: 0 }],
    ownerDiscounts: [{ description: '', amount: 0 }],
    otherDiscounts: [{ description: '', amount: 0 }],
    notes: '',
    paymentMethod: 'bank-transfer',
    status: 'pending'
  });

  const [errors, setErrors] = useState({});

  const paymentMethodOptions = [
    { value: 'bank-transfer', label: 'Transferencia Bancaria' },
    { value: 'cash', label: 'Efectivo' },
    { value: 'check', label: 'Cheque' },
    { value: 'deposit', label: 'Depósito' }
  ];

  const statusOptions = [
    { value: 'pending', label: 'Pendiente' },
    { value: 'paid', label: 'Pagado' },
    { value: 'overdue', label: 'Vencido' },
    { value: 'partial', label: 'Pago Parcial' }
  ];

  useEffect(() => {
    if (voucher) {
      // Edit mode - populate with existing voucher data
      setFormData({
        month: voucher?.month || '',
        dueDate: voucher?.dueDate || '',
        rentAmount: voucher?.amount || 850000,
        gasAmount: voucher?.gasAmount || 35000,
        commonExpenses: voucher?.commonExpenses || 45000,
        waterAmount: voucher?.waterAmount || 25000,
        electricityAmount: voucher?.electricityAmount || 40000,
        internetAmount: voucher?.internetAmount || 30000,
        tenantDiscounts: voucher?.tenantDiscounts || [{ description: '', amount: 0 }],
        ownerDiscounts: voucher?.ownerDiscounts || [{ description: '', amount: 0 }],
        otherDiscounts: voucher?.otherDiscounts || [{ description: '', amount: 0 }],
        notes: voucher?.notes || '',
        paymentMethod: voucher?.paymentMethod || 'bank-transfer',
        status: voucher?.status || 'pending'
      });
    } else {
      // Create mode - use defaults
      const currentDate = new Date();
      const currentMonth = currentDate?.toLocaleDateString('es-CL', { month: 'long', year: 'numeric' });
      const dueDate = new Date(currentDate.getFullYear(), currentDate.getMonth(), 5);
      
      setFormData({
        month: currentMonth,
        dueDate: dueDate?.toISOString()?.split('T')?.[0],
        rentAmount: propertyData?.monthlyRent || 850000,
        gasAmount: 35000,
        commonExpenses: propertyData?.commonExpenses || 45000,
        waterAmount: 25000,
        electricityAmount: 40000,
        internetAmount: 30000,
        tenantDiscounts: [{ description: '', amount: 0 }],
        ownerDiscounts: [{ description: '', amount: 0 }],
        otherDiscounts: [{ description: '', amount: 0 }],
        notes: '',
        paymentMethod: 'bank-transfer',
        status: 'pending'
      });
    }
  }, [voucher, propertyData, isOpen]);

  const handleInputChange = (field, value) => {
    setFormData(prev => ({
      ...prev,
      [field]: value
    }));
    
    // Clear error for this field
    if (errors?.[field]) {
      setErrors(prev => ({
        ...prev,
        [field]: null
      }));
    }
  };

  const handleDiscountChange = (discountType, index, field, value) => {
    setFormData(prev => ({
      ...prev,
      [discountType]: prev?.[discountType]?.map((discount, i) => 
        i === index ? { ...discount, [field]: value } : discount
      )
    }));
  };

  const addDiscount = (discountType) => {
    setFormData(prev => ({
      ...prev,
      [discountType]: [...prev?.[discountType], { description: '', amount: 0 }]
    }));
  };

  const removeDiscount = (discountType, index) => {
    if (formData?.[discountType]?.length > 1) {
      setFormData(prev => ({
        ...prev,
        [discountType]: prev?.[discountType]?.filter((_, i) => i !== index)
      }));
    }
  };

  const calculateTotalDiscounts = () => {
    const tenantTotal = formData?.tenantDiscounts?.reduce((sum, discount) => sum + (discount?.amount || 0), 0);
    const ownerTotal = formData?.ownerDiscounts?.reduce((sum, discount) => sum + (discount?.amount || 0), 0);
    const otherTotal = formData?.otherDiscounts?.reduce((sum, discount) => sum + (discount?.amount || 0), 0);
    return { tenantTotal, ownerTotal, otherTotal, total: tenantTotal + ownerTotal + otherTotal };
  };

  const calculateExpensesTotal = () => {
    return formData?.gasAmount + formData?.commonExpenses + formData?.waterAmount + 
           formData?.electricityAmount + formData?.internetAmount;
  };

  const calculateFinalTotal = () => {
    const expensesTotal = calculateExpensesTotal();
    const discounts = calculateTotalDiscounts();
    return formData?.rentAmount + expensesTotal - discounts?.total;
  };

  const validateForm = () => {
    const newErrors = {};
    
    if (!formData?.month?.trim()) newErrors.month = 'El período es obligatorio';
    if (!formData?.dueDate) newErrors.dueDate = 'La fecha de vencimiento es obligatoria';
    if (!formData?.rentAmount || formData?.rentAmount <= 0) newErrors.rentAmount = 'El monto del arriendo debe ser mayor a 0';
    
    setErrors(newErrors);
    return Object.keys(newErrors)?.length === 0;
  };

  const handleSave = () => {
    if (!validateForm()) return;

    const voucherData = {
      ...formData,
      id: voucher?.id || Date.now(),
      totalAmount: calculateFinalTotal(),
      expensesTotal: calculateExpensesTotal(),
      discountsTotal: calculateTotalDiscounts()?.total,
      voucherNumber: voucher?.voucherNumber || `VCH-${new Date()?.getFullYear()}-${String(new Date()?.getMonth() + 1)?.padStart(2, '0')}-${String(Math.floor(Math.random() * 1000))?.padStart(3, '0')}`
    };

    onSave?.(voucherData);
    onClose();
  };

  const formatCurrency = (amount) => {
    return new Intl.NumberFormat('es-CL', {
      style: 'currency',
      currency: 'CLP',
      minimumFractionDigits: 0
    })?.format(amount || 0);
  };

  const discounts = calculateTotalDiscounts();
  const expensesTotal = calculateExpensesTotal();
  const finalTotal = calculateFinalTotal();

  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4 z-50">
      <div className="bg-white rounded-lg max-w-4xl w-full max-h-[90vh] overflow-y-auto">
        {/* Header */}
        <div className="flex items-center justify-between p-6 border-b border-gray-200">
          <h3 className="text-lg font-semibold text-gray-900">
            {voucher ? 'Editar Comprobante' : 'Crear Nuevo Comprobante'}
          </h3>
          <button
            onClick={onClose}
            className="text-gray-400 hover:text-gray-600 transition-colors duration-150"
          >
            <Icon name="X" size={24} />
          </button>
        </div>

        {/* Content */}
        <div className="p-6 space-y-6">
          {/* Basic Information */}
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <Input
              label="Período *"
              value={formData?.month}
              onChange={(e) => handleInputChange('month', e?.target?.value)}
              placeholder="Enero 2024"
              error={errors?.month}
            />
            <Input
              label="Fecha de Vencimiento *"
              type="date"
              value={formData?.dueDate}
              onChange={(e) => handleInputChange('dueDate', e?.target?.value)}
              error={errors?.dueDate}
            />
          </div>

          {/* Amounts Section */}
          <div className="space-y-4">
            <h4 className="text-md font-medium text-gray-900 border-b pb-2">Montos</h4>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <Input
                label="Arriendo *"
                type="number"
                value={formData?.rentAmount}
                onChange={(e) => handleInputChange('rentAmount', parseInt(e?.target?.value) || 0)}
                placeholder="850000"
                error={errors?.rentAmount}
              />
              <div className="text-sm text-gray-600 flex items-center">
                <strong>Monto Base: {formatCurrency(formData?.rentAmount)}</strong>
              </div>
            </div>

            {/* Service Expenses - Gas first, then Common Expenses */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <Input
                label="Gas"
                type="number"
                value={formData?.gasAmount}
                onChange={(e) => handleInputChange('gasAmount', parseInt(e?.target?.value) || 0)}
                placeholder="35000"
              />
              <Input
                label="Gastos Comunes"
                type="number"
                value={formData?.commonExpenses}
                onChange={(e) => handleInputChange('commonExpenses', parseInt(e?.target?.value) || 0)}
                placeholder="45000"
              />
            </div>

            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              <Input
                label="Agua"
                type="number"
                value={formData?.waterAmount}
                onChange={(e) => handleInputChange('waterAmount', parseInt(e?.target?.value) || 0)}
                placeholder="25000"
              />
              <Input
                label="Electricidad"
                type="number"
                value={formData?.electricityAmount}
                onChange={(e) => handleInputChange('electricityAmount', parseInt(e?.target?.value) || 0)}
                placeholder="40000"
              />
              <Input
                label="Internet"
                type="number"
                value={formData?.internetAmount}
                onChange={(e) => handleInputChange('internetAmount', parseInt(e?.target?.value) || 0)}
                placeholder="30000"
              />
            </div>

            <div className="bg-blue-50 p-3 rounded-lg">
              <div className="text-sm text-blue-800">
                <strong>Total Gastos: {formatCurrency(expensesTotal)}</strong>
              </div>
            </div>
          </div>

          {/* Discounts Section */}
          <div className="space-y-4">
            <h4 className="text-md font-medium text-gray-900 border-b pb-2">Descuentos</h4>

            {/* Tenant Discounts */}
            <div className="space-y-2">
              <div className="flex items-center justify-between">
                <label className="text-sm font-medium text-gray-700">Descuentos al Arrendatario</label>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => addDiscount('tenantDiscounts')}
                  iconName="Plus"
                >
                  Agregar
                </Button>
              </div>
              {formData?.tenantDiscounts?.map((discount, index) => (
                <div key={index} className="flex items-center space-x-2">
                  <Input
                    placeholder="Descripción del descuento"
                    value={discount?.description}
                    onChange={(e) => handleDiscountChange('tenantDiscounts', index, 'description', e?.target?.value)}
                    className="flex-1"
                  />
                  <Input
                    type="number"
                    placeholder="Monto"
                    value={discount?.amount}
                    onChange={(e) => handleDiscountChange('tenantDiscounts', index, 'amount', parseInt(e?.target?.value) || 0)}
                    className="w-32"
                  />
                  {formData?.tenantDiscounts?.length > 1 && (
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => removeDiscount('tenantDiscounts', index)}
                      iconName="Trash2"
                      className="text-red-600 hover:text-red-800"
                    />
                  )}
                </div>
              ))}
            </div>

            {/* Owner Discounts */}
            <div className="space-y-2">
              <div className="flex items-center justify-between">
                <label className="text-sm font-medium text-gray-700">Descuentos al Propietario</label>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => addDiscount('ownerDiscounts')}
                  iconName="Plus"
                >
                  Agregar
                </Button>
              </div>
              {formData?.ownerDiscounts?.map((discount, index) => (
                <div key={index} className="flex items-center space-x-2">
                  <Input
                    placeholder="Descripción del descuento"
                    value={discount?.description}
                    onChange={(e) => handleDiscountChange('ownerDiscounts', index, 'description', e?.target?.value)}
                    className="flex-1"
                  />
                  <Input
                    type="number"
                    placeholder="Monto"
                    value={discount?.amount}
                    onChange={(e) => handleDiscountChange('ownerDiscounts', index, 'amount', parseInt(e?.target?.value) || 0)}
                    className="w-32"
                  />
                  {formData?.ownerDiscounts?.length > 1 && (
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => removeDiscount('ownerDiscounts', index)}
                      iconName="Trash2"
                      className="text-red-600 hover:text-red-800"
                    />
                  )}
                </div>
              ))}
            </div>

            {/* Other Discounts */}
            <div className="space-y-2">
              <div className="flex items-center justify-between">
                <label className="text-sm font-medium text-gray-700">Otros Descuentos</label>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => addDiscount('otherDiscounts')}
                  iconName="Plus"
                >
                  Agregar
                </Button>
              </div>
              {formData?.otherDiscounts?.map((discount, index) => (
                <div key={index} className="flex items-center space-x-2">
                  <Input
                    placeholder="Descripción del descuento"
                    value={discount?.description}
                    onChange={(e) => handleDiscountChange('otherDiscounts', index, 'description', e?.target?.value)}
                    className="flex-1"
                  />
                  <Input
                    type="number"
                    placeholder="Monto"
                    value={discount?.amount}
                    onChange={(e) => handleDiscountChange('otherDiscounts', index, 'amount', parseInt(e?.target?.value) || 0)}
                    className="w-32"
                  />
                  {formData?.otherDiscounts?.length > 1 && (
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={() => removeDiscount('otherDiscounts', index)}
                      iconName="Trash2"
                      className="text-red-600 hover:text-red-800"
                    />
                  )}
                </div>
              ))}
            </div>

            <div className="bg-yellow-50 p-3 rounded-lg">
              <div className="text-sm text-yellow-800 space-y-1">
                <div>Descuentos Arrendatario: {formatCurrency(discounts?.tenantTotal)}</div>
                <div>Descuentos Propietario: {formatCurrency(discounts?.ownerTotal)}</div>
                <div>Otros Descuentos: {formatCurrency(discounts?.otherTotal)}</div>
                <div><strong>Total Descuentos: {formatCurrency(discounts?.total)}</strong></div>
              </div>
            </div>
          </div>

          {/* Payment Details */}
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <Select
              label="Método de Pago"
              options={paymentMethodOptions}
              value={formData?.paymentMethod}
              onChange={(value) => handleInputChange('paymentMethod', value)}
            />
            <Select
              label="Estado"
              options={statusOptions}
              value={formData?.status}
              onChange={(value) => handleInputChange('status', value)}
            />
          </div>

          {/* Notes */}
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Notas Adicionales
            </label>
            <textarea
              rows={3}
              value={formData?.notes}
              onChange={(e) => handleInputChange('notes', e?.target?.value)}
              placeholder="Agregar notas o comentarios adicionales..."
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
            />
          </div>

          {/* Summary */}
          <div className="bg-green-50 border border-green-200 rounded-lg p-4">
            <h4 className="text-md font-medium text-green-800 mb-2">Resumen del Comprobante</h4>
            <div className="space-y-1 text-sm text-green-700">
              <div className="flex justify-between">
                <span>Arriendo:</span>
                <span>{formatCurrency(formData?.rentAmount)}</span>
              </div>
              <div className="flex justify-between">
                <span>Gas:</span>
                <span>{formatCurrency(formData?.gasAmount)}</span>
              </div>
              <div className="flex justify-between">
                <span>Gastos Comunes:</span>
                <span>{formatCurrency(formData?.commonExpenses)}</span>
              </div>
              <div className="flex justify-between">
                <span>Otros Gastos:</span>
                <span>{formatCurrency(formData?.waterAmount + formData?.electricityAmount + formData?.internetAmount)}</span>
              </div>
              <div className="flex justify-between border-t pt-1">
                <span>Subtotal:</span>
                <span>{formatCurrency(formData?.rentAmount + expensesTotal)}</span>
              </div>
              <div className="flex justify-between text-red-600">
                <span>Descuentos:</span>
                <span>-{formatCurrency(discounts?.total)}</span>
              </div>
              <div className="flex justify-between border-t pt-1 font-bold text-green-800">
                <span>Total Final:</span>
                <span>{formatCurrency(finalTotal)}</span>
              </div>
            </div>
          </div>
        </div>

        {/* Footer */}
        <div className="flex items-center justify-end space-x-4 p-6 border-t border-gray-200">
          <Button
            variant="outline"
            onClick={onClose}
          >
            Cancelar
          </Button>
          <Button
            variant="default"
            onClick={handleSave}
            iconName="Save"
          >
            {voucher ? 'Guardar Cambios' : 'Crear Comprobante'}
          </Button>
        </div>
      </div>
    </div>
  );
};

export default VoucherEditModal;