import React, { useState, useEffect } from 'react';
import { useLocation } from 'react-router-dom';
import Header from '../../components/ui/Header';
import Sidebar from '../../components/ui/Sidebar';
import QuickActionFloatingButton from '../../components/ui/QuickActionFloatingButton';
import PropertyDetailsHeader from './components/PropertyDetailsHeader';
import BasicDataTab from './components/BasicDataTab';
import OwnerInformationTab from './components/OwnerInformationTab';
import PropertyCharacteristicsTab from './components/PropertyCharacteristicsTab';
import RentalContractTab from './components/RentalContractTab';
import ServiceConnectionsTab from './components/ServiceConnectionsTab';
import PaymentHistoryTab from './components/PaymentHistoryTab';
import Icon from '../../components/AppIcon';

const PropertyDetails = () => {
  const [isSidebarOpen, setIsSidebarOpen] = useState(false);
  const [activeTab, setActiveTab] = useState('basic-data');
  const [propertyData, setPropertyData] = useState({});
  const [ownerData, setOwnerData] = useState({});
  const [characteristicsData, setCharacteristicsData] = useState({});
  const [contractData, setContractData] = useState({});
  const [servicesData, setServicesData] = useState({});
  const [paymentData, setPaymentData] = useState({});
  const location = useLocation();

  // Mock data initialization
  useEffect(() => {
    // Initialize mock property data
    setPropertyData({
      id: 1,
      address: "Av. Providencia 1234, Depto 502",
      commune: "Providencia",
      city: "Santiago",
      type: "apartment",
      status: "occupied",
      propertyCode: "PROP-2024-001",
      builtArea: 85,
      totalArea: 85,
      bedrooms: 2,
      bathrooms: 2,
      parkingSpaces: 1,
      floor: "5",
      yearBuilt: 2018,
      description: `Hermoso departamento de 2 dormitorios y 2 baños en el corazón de Providencia.\n\nUbicado en un moderno edificio con excelentes terminaciones y amenidades.\n\nCerca del metro y centros comerciales.`,
      monthlyRent: 850000,
      occupancyRate: 100,
      totalPayments: 12,
      pendingIssues: 1
    });

    setOwnerData({
      id: 1,
      fullName: "Carlos Mendoza Ruiz",
      rut: "12.345.678-9",
      email: "carlos.mendoza@email.com",
      phone: "+56 9 8765 4321",
      alternativePhone: "+56 2 2234 5678",
      occupation: "Ingeniero Civil",
      address: "Las Condes 567, Las Condes",
      city: "Santiago",
      commune: "Las Condes",
      commissionRate: 8.5,
      minimumCommission: 50000,
      maximumCommission: 200000,
      bankName: "Banco de Chile",
      accountType: "Cuenta Corriente",
      accountNumber: "12345678",
      transferEmail: "carlos.mendoza@email.com",
      totalProperties: 3,
      avatar: null
    });

    setCharacteristicsData({
      condition: "very-good",
      orientation: "north",
      amenities: {
        furnished: false,
        airConditioning: true,
        heating: true,
        fireplace: false,
        balcony: true,
        terrace: false,
        walkinCloset: true,
        laundryRoom: true,
        elevator: true,
        doorman: true,
        security: true,
        gym: true,
        pool: true,
        playground: false,
        eventRoom: true,
        rooftop: true,
        nearMetro: true,
        nearBus: true,
        nearSchools: true,
        nearShopping: true,
        nearParks: true,
        nearHospitals: false,
        quietArea: true,
        commercialArea: false
      },
      additionalFeatures: `Departamento completamente remodelado en 2023 con terminaciones de primera calidad.\n\nCocina equipada con electrodomésticos nuevos.\n\nVista panorámica hacia la cordillera.\n\nEstacionamiento techado y bodega incluida.`
    });

    setContractData({
      status: "active",
      contractType: "fixed-term",
      startDate: "2024-01-15",
      endDate: "2025-01-14",
      duration: 12,
      monthlyRentCLP: 850000,
      monthlyRentUF: 22.83,
      securityDeposit: 850000,
      commonExpenses: 45000,
      paymentDueDay: 5,
      paymentMethod: "bank-transfer",
      adjustmentType: "uf",
      specialTerms: `El arrendatario se compromete a mantener la propiedad en perfecto estado.\n\nNo se permiten mascotas sin autorización previa del propietario.\n\nEl pago debe realizarse dentro de los primeros 5 días de cada mes.\n\nLa garantía será devuelta al término del contrato, previa inspección de la propiedad.`,
      tenant: {
        id: 1,
        fullName: "María José González Silva",
        email: "maria.gonzalez@email.com",
        phone: "+56 9 1234 5678",
        rut: "18.765.432-1",
        avatar: null
      }
    });

    setServicesData({
      electricity: {
        status: "active",
        provider: "Enel Distribución Chile",
        accountNumber: "1234567890",
        paymentResponsibility: "tenant",
        monthlyCost: 0,
        installationDate: "2024-01-15",
        contactPhone: "+56 2 2696 9696",
        notes: "Medidor individual, lectura mensual automática"
      },
      water: {
        status: "active",
        provider: "Aguas Andinas",
        accountNumber: "WTR-789456123",
        paymentResponsibility: "owner",
        monthlyCost: 25000,
        installationDate: "2024-01-15",
        contactPhone: "+56 2 2426 9000",
        notes: "Incluye agua potable y alcantarillado"
      },
      gas: {
        status: "active",
        provider: "Metrogas",
        accountNumber: "GAS-456789012",
        paymentResponsibility: "tenant",
        monthlyCost: 0,
        installationDate: "2024-01-15",
        contactPhone: "+56 2 2337 8000",
        notes: "Gas natural por red, medidor individual"
      },
      internet: {
        status: "active",
        provider: "Movistar Fibra",
        accountNumber: "INT-321654987",
        paymentResponsibility: "tenant",
        monthlyCost: 0,
        installationDate: "2024-01-20",
        contactPhone: "+56 2 2800 9000",
        notes: "Fibra óptica 300 Mbps, instalación en living"
      },
      cable: {
        status: "inactive",
        provider: "",
        accountNumber: "",
        paymentResponsibility: "tenant",
        monthlyCost: 0,
        installationDate: "",
        contactPhone: "",
        notes: "Servicio no contratado"
      },
      building: {
        status: "active",
        provider: "Administración Edificio Providencia",
        accountNumber: "ADM-502",
        paymentResponsibility: "owner",
        monthlyCost: 45000,
        installationDate: "2024-01-15",
        contactPhone: "+56 2 2234 5678",
        notes: "Incluye mantención áreas comunes, seguridad y conserje"
      }
    });

    setPaymentData({
      totalPaid: 10750000,
      totalPending: 895000,
      paidCount: 12,
      overdueCount: 1
    });
  }, []);

  const tabs = [
    {
      key: 'basic-data',
      label: 'Datos Básicos',
      icon: 'Home',
      component: BasicDataTab
    },
    {
      key: 'owner-info',
      label: 'Propietario',
      icon: 'UserCheck',
      component: OwnerInformationTab
    },
    {
      key: 'characteristics',
      label: 'Características',
      icon: 'Settings',
      component: PropertyCharacteristicsTab
    },
    {
      key: 'contract',
      label: 'Contrato',
      icon: 'FileText',
      component: RentalContractTab
    },
    {
      key: 'services',
      label: 'Servicios',
      icon: 'Zap',
      component: ServiceConnectionsTab
    },
    {
      key: 'payments',
      label: 'Historial Pagos',
      icon: 'CreditCard',
      component: PaymentHistoryTab
    }
  ];

  const handleMenuToggle = () => {
    setIsSidebarOpen(!isSidebarOpen);
  };

  const handleSidebarClose = () => {
    setIsSidebarOpen(false);
  };

  const handleDataUpdate = (section, data) => {
    switch (section) {
      case 'property':
        setPropertyData(data);
        break;
      case 'owner':
        setOwnerData(data);
        break;
      case 'characteristics':
        setCharacteristicsData(data);
        break;
      case 'contract':
        setContractData(data);
        break;
      case 'services':
        setServicesData(data);
        break;
      default:
        console.log('Unknown section:', section);
    }
  };

  const renderTabContent = () => {
    const activeTabData = tabs?.find(tab => tab?.key === activeTab);
    if (!activeTabData) return null;

    const TabComponent = activeTabData?.component;
    
    switch (activeTab) {
      case 'basic-data':
        return <TabComponent propertyData={propertyData} onUpdate={(data) => handleDataUpdate('property', data)} />;
      case 'owner-info':
        return <TabComponent ownerData={ownerData} onUpdate={(data) => handleDataUpdate('owner', data)} />;
      case 'characteristics':
        return <TabComponent characteristicsData={characteristicsData} onUpdate={(data) => handleDataUpdate('characteristics', data)} />;
      case 'contract':
        return <TabComponent contractData={contractData} onUpdate={(data) => handleDataUpdate('contract', data)} />;
      case 'services':
        return <TabComponent servicesData={servicesData} onUpdate={(data) => handleDataUpdate('services', data)} />;
      case 'payments':
        return <TabComponent paymentData={paymentData} propertyData={propertyData} />;
      default:
        return null;
    }
  };

  return (
    <div className="min-h-screen bg-background">
      <Header onMenuToggle={handleMenuToggle} isMenuOpen={isSidebarOpen} />
      <Sidebar isOpen={isSidebarOpen} onClose={handleSidebarClose} />
      <main className="lg:ml-60 pt-16">
        <div className="p-6 max-w-7xl mx-auto">
          {/* Property Header */}
          <PropertyDetailsHeader propertyData={propertyData} />
          
          {/* Tab Navigation */}
          <div className="mt-8">
            {/* Desktop Tabs */}
            <div className="hidden md:block">
              <div className="border-b border-gray-200">
                <nav className="-mb-px flex space-x-8 overflow-x-auto">
                  {tabs?.map((tab) => (
                    <button
                      key={tab?.key}
                      onClick={() => setActiveTab(tab?.key)}
                      className={`
                        flex items-center space-x-2 py-4 px-1 border-b-2 font-medium text-sm whitespace-nowrap
                        transition-colors duration-150
                        ${activeTab === tab?.key
                          ? 'border-primary text-primary' :'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                        }
                      `}
                    >
                      <Icon name={tab?.icon} size={18} />
                      <span>{tab?.label}</span>
                    </button>
                  ))}
                </nav>
              </div>
            </div>

            {/* Mobile Tab Selector */}
            <div className="md:hidden mb-6">
              <select
                value={activeTab}
                onChange={(e) => setActiveTab(e?.target?.value)}
                className="w-full px-3 py-2 border border-gray-300 rounded-lg bg-white text-gray-900 focus:ring-2 focus:ring-primary focus:border-transparent"
              >
                {tabs?.map((tab) => (
                  <option key={tab?.key} value={tab?.key}>
                    {tab?.label}
                  </option>
                ))}
              </select>
            </div>

            {/* Tab Content */}
            <div className="mt-6">
              {renderTabContent()}
            </div>
          </div>
        </div>
      </main>
      <QuickActionFloatingButton />
    </div>
  );
};

export default PropertyDetails;