import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import Input from '../../../components/ui/Input';
import Button from '../../../components/ui/Button';
import Icon from '../../../components/AppIcon';
import PasswordStrengthMeter from './PasswordStrengthMeter';
import PhoneNumberInput from './PhoneNumberInput';
import TermsCheckbox from './TermsCheckbox';

const RegistrationForm = () => {
  const navigate = useNavigate();
  const [loading, setLoading] = useState(false);
  const [showPassword, setShowPassword] = useState(false);
  const [showConfirmPassword, setShowConfirmPassword] = useState(false);
  const [formData, setFormData] = useState({
    fullName: '',
    email: '',
    password: '',
    confirmPassword: '',
    phone: '',
    acceptTerms: false
  });
  const [errors, setErrors] = useState({});

  const validateForm = () => {
    const newErrors = {};

    // Full name validation
    if (!formData.fullName.trim()) {
      newErrors.fullName = 'El nombre completo es requerido';
    } else if (formData.fullName.trim().length < 2) {
      newErrors.fullName = 'El nombre debe tener al menos 2 caracteres';
    }

    // Email validation
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!formData.email) {
      newErrors.email = 'El email es requerido';
    } else if (!emailRegex.test(formData.email)) {
      newErrors.email = 'Ingresa un email válido';
    }

    // Password validation
    if (!formData.password) {
      newErrors.password = 'La contraseña es requerida';
    } else if (formData.password.length < 8) {
      newErrors.password = 'La contraseña debe tener al menos 8 caracteres';
    }

    // Confirm password validation
    if (!formData.confirmPassword) {
      newErrors.confirmPassword = 'Confirma tu contraseña';
    } else if (formData.password !== formData.confirmPassword) {
      newErrors.confirmPassword = 'Las contraseñas no coinciden';
    }

    // Phone validation
    if (!formData.phone) {
      newErrors.phone = 'El número de teléfono es requerido';
    } else if (formData.phone.length !== 9) {
      newErrors.phone = 'Ingresa un número de teléfono válido';
    }

    // Terms validation
    if (!formData.acceptTerms) {
      newErrors.acceptTerms = 'Debes aceptar los términos y condiciones';
    }

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleInputChange = (field, value) => {
    setFormData(prev => ({ ...prev, [field]: value }));
    
    // Clear error when user starts typing
    if (errors[field]) {
      setErrors(prev => ({ ...prev, [field]: '' }));
    }
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    
    if (!validateForm()) return;

    setLoading(true);
    
    try {
      // Simulate API call
      await new Promise(resolve => setTimeout(resolve, 2000));
      
      // Mock successful registration
      console.log('Registration successful:', formData);
      
      // In a real app, you would handle the API response here
      // For now, we'll navigate to login with a success message
      navigate('/login', { 
        state: { 
          message: 'Cuenta creada exitosamente. Revisa tu email para verificar tu cuenta.',
          type: 'success'
        }
      });
      
    } catch (error) {
      console.error('Registration error:', error);
      setErrors({ submit: 'Error al crear la cuenta. Inténtalo nuevamente.' });
    } finally {
      setLoading(false);
    }
  };

  return (
    <form onSubmit={handleSubmit} className="space-y-6">
      {/* Full Name */}
      <Input
        type="text"
        label="Nombre Completo"
        placeholder="Ingresa tu nombre completo"
        value={formData.fullName}
        onChange={(e) => handleInputChange('fullName', e.target.value)}
        error={errors.fullName}
        required
        disabled={loading}
      />

      {/* Email */}
      <Input
        type="email"
        label="Correo Electrónico"
        placeholder="tu@email.com"
        value={formData.email}
        onChange={(e) => handleInputChange('email', e.target.value)}
        error={errors.email}
        required
        disabled={loading}
      />

      {/* Password */}
      <div className="relative">
        <Input
          type={showPassword ? "text" : "password"}
          label="Contraseña"
          placeholder="Crea una contraseña segura"
          value={formData.password}
          onChange={(e) => handleInputChange('password', e.target.value)}
          error={errors.password}
          required
          disabled={loading}
        />
        <button
          type="button"
          onClick={() => setShowPassword(!showPassword)}
          className="absolute right-3 top-9 text-gray-500 hover:text-gray-700"
          disabled={loading}
        >
          <Icon name={showPassword ? "EyeOff" : "Eye"} size={20} />
        </button>
        
        <PasswordStrengthMeter password={formData.password} />
      </div>

      {/* Confirm Password */}
      <div className="relative">
        <Input
          type={showConfirmPassword ? "text" : "password"}
          label="Confirmar Contraseña"
          placeholder="Repite tu contraseña"
          value={formData.confirmPassword}
          onChange={(e) => handleInputChange('confirmPassword', e.target.value)}
          error={errors.confirmPassword}
          required
          disabled={loading}
        />
        <button
          type="button"
          onClick={() => setShowConfirmPassword(!showConfirmPassword)}
          className="absolute right-3 top-9 text-gray-500 hover:text-gray-700"
          disabled={loading}
        >
          <Icon name={showConfirmPassword ? "EyeOff" : "Eye"} size={20} />
        </button>
      </div>

      {/* Phone Number */}
      <PhoneNumberInput
        value={formData.phone}
        onChange={(value) => handleInputChange('phone', value)}
        error={errors.phone}
        required
      />

      {/* Terms and Conditions */}
      <TermsCheckbox
        checked={formData.acceptTerms}
        onChange={(checked) => handleInputChange('acceptTerms', checked)}
        error={errors.acceptTerms}
      />

      {/* Submit Error */}
      {errors.submit && (
        <div className="p-4 bg-error/10 border border-error/20 rounded-lg">
          <div className="flex items-center space-x-2">
            <Icon name="AlertCircle" size={16} className="text-error flex-shrink-0" />
            <p className="text-sm text-error">{errors.submit}</p>
          </div>
        </div>
      )}

      {/* Submit Button */}
      <Button
        type="submit"
        variant="default"
        fullWidth
        loading={loading}
        disabled={loading}
        iconName="UserPlus"
        iconPosition="left"
      >
        {loading ? 'Creando cuenta...' : 'Crear Cuenta'}
      </Button>

      {/* Login Link */}
      <div className="text-center">
        <p className="text-sm text-gray-600">
          ¿Ya tienes una cuenta?{' '}
          <button
            type="button"
            onClick={() => navigate('/login')}
            className="text-primary hover:text-primary/80 font-medium underline"
            disabled={loading}
          >
            Inicia sesión aquí
          </button>
        </p>
      </div>
    </form>
  );
};

export default RegistrationForm;