import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';
import StatusIndicatorBadges from '../../../components/ui/StatusIndicatorBadges';
import Select from '../../../components/ui/Select';

const ActiveRemindersTable = ({ reminders, onReminderAction, onBulkAction }) => {
  const [selectedReminders, setSelectedReminders] = useState([]);
  const [filterStatus, setFilterStatus] = useState('all');
  const [sortBy, setSortBy] = useState('scheduledDate');
  const [sortOrder, setSortOrder] = useState('asc');

  const statusOptions = [
    { value: 'all', label: 'Todos los Estados' },
    { value: 'scheduled', label: 'Programados' },
    { value: 'sent', label: 'Enviados' },
    { value: 'delivered', label: 'Entregados' },
    { value: 'failed', label: 'Fallidos' },
    { value: 'cancelled', label: 'Cancelados' }
  ];

  const sortOptions = [
    { value: 'scheduledDate', label: 'Fecha Programada' },
    { value: 'tenantName', label: 'Nombre Inquilino' },
    { value: 'propertyAddress', label: 'Propiedad' },
    { value: 'status', label: 'Estado' },
    { value: 'retryCount', label: 'Intentos' }
  ];

  const getStatusBadge = (status) => {
    const statusMap = {
      scheduled: 'pending',
      sent: 'review',
      delivered: 'paid',
      failed: 'overdue',
      cancelled: 'inactive'
    };
    return statusMap?.[status] || 'inactive';
  };

  const getChannelIcon = (channel) => {
    switch (channel) {
      case 'email': return 'Mail';
      case 'sms': return 'MessageSquare';
      case 'whatsapp': return 'MessageCircle';
      default: return 'Bell';
    }
  };

  const getPriorityColor = (priority) => {
    switch (priority) {
      case 'high': return 'text-error';
      case 'medium': return 'text-warning';
      case 'low': return 'text-success';
      default: return 'text-gray-600';
    }
  };

  const filteredReminders = reminders?.filter(reminder => filterStatus === 'all' || reminder?.status === filterStatus)?.sort((a, b) => {
      let aValue = a?.[sortBy];
      let bValue = b?.[sortBy];
      
      if (sortBy === 'scheduledDate') {
        aValue = new Date(aValue);
        bValue = new Date(bValue);
      }
      
      if (sortOrder === 'asc') {
        return aValue > bValue ? 1 : -1;
      } else {
        return aValue < bValue ? 1 : -1;
      }
    });

  const handleSelectAll = (checked) => {
    if (checked) {
      setSelectedReminders(filteredReminders?.map(r => r?.id));
    } else {
      setSelectedReminders([]);
    }
  };

  const handleSelectReminder = (reminderId, checked) => {
    if (checked) {
      setSelectedReminders(prev => [...prev, reminderId]);
    } else {
      setSelectedReminders(prev => prev?.filter(id => id !== reminderId));
    }
  };

  const handleBulkAction = (action) => {
    if (selectedReminders?.length > 0) {
      onBulkAction(action, selectedReminders);
      setSelectedReminders([]);
    }
  };

  const formatDate = (date) => {
    return new Date(date)?.toLocaleDateString('es-CL', {
      day: '2-digit',
      month: '2-digit',
      year: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };

  return (
    <div className="bg-white rounded-lg border border-gray-200 overflow-hidden">
      {/* Header */}
      <div className="p-6 border-b border-gray-200">
        <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between space-y-4 lg:space-y-0">
          <div>
            <h3 className="text-lg font-semibold text-gray-900">Recordatorios Activos</h3>
            <p className="text-sm text-gray-600 mt-1">
              {filteredReminders?.length} recordatorios {filterStatus !== 'all' ? `(${statusOptions?.find(s => s?.value === filterStatus)?.label?.toLowerCase()})` : ''}
            </p>
          </div>
          
          <div className="flex flex-col sm:flex-row items-stretch sm:items-center space-y-2 sm:space-y-0 sm:space-x-3">
            <Select
              options={statusOptions}
              value={filterStatus}
              onChange={setFilterStatus}
              className="w-full sm:w-48"
            />
            <Select
              options={sortOptions}
              value={sortBy}
              onChange={setSortBy}
              className="w-full sm:w-48"
            />
            <Button
              variant="outline"
              size="sm"
              iconName={sortOrder === 'asc' ? 'ArrowUp' : 'ArrowDown'}
              onClick={() => setSortOrder(sortOrder === 'asc' ? 'desc' : 'asc')}
            />
          </div>
        </div>

        {/* Bulk Actions */}
        {selectedReminders?.length > 0 && (
          <div className="mt-4 p-3 bg-primary/5 rounded-lg border border-primary/20">
            <div className="flex flex-col sm:flex-row items-start sm:items-center justify-between space-y-2 sm:space-y-0">
              <span className="text-sm font-medium text-primary">
                {selectedReminders?.length} recordatorios seleccionados
              </span>
              <div className="flex items-center space-x-2">
                <Button
                  variant="outline"
                  size="sm"
                  iconName="Send"
                  onClick={() => handleBulkAction('send')}
                >
                  Enviar Ahora
                </Button>
                <Button
                  variant="outline"
                  size="sm"
                  iconName="Calendar"
                  onClick={() => handleBulkAction('reschedule')}
                >
                  Reprogramar
                </Button>
                <Button
                  variant="outline"
                  size="sm"
                  iconName="X"
                  onClick={() => handleBulkAction('cancel')}
                >
                  Cancelar
                </Button>
              </div>
            </div>
          </div>
        )}
      </div>
      {/* Table */}
      <div className="overflow-x-auto">
        <table className="w-full">
          <thead className="bg-gray-50 border-b border-gray-200">
            <tr>
              <th className="w-12 px-6 py-3 text-left">
                <input
                  type="checkbox"
                  className="rounded border-gray-300 text-primary focus:ring-primary"
                  checked={selectedReminders?.length === filteredReminders?.length && filteredReminders?.length > 0}
                  onChange={(e) => handleSelectAll(e?.target?.checked)}
                />
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Inquilino
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Propiedad
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Plantilla
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Canal
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Programado
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Estado
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Intentos
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Acciones
              </th>
            </tr>
          </thead>
          <tbody className="bg-white divide-y divide-gray-200">
            {filteredReminders?.map((reminder) => (
              <tr key={reminder?.id} className="hover:bg-gray-50 transition-colors duration-150">
                <td className="px-6 py-4">
                  <input
                    type="checkbox"
                    className="rounded border-gray-300 text-primary focus:ring-primary"
                    checked={selectedReminders?.includes(reminder?.id)}
                    onChange={(e) => handleSelectReminder(reminder?.id, e?.target?.checked)}
                  />
                </td>
                <td className="px-6 py-4">
                  <div className="flex items-center space-x-3">
                    <div className="w-8 h-8 bg-primary/10 text-primary rounded-full flex items-center justify-center text-sm font-medium">
                      {reminder?.tenantName?.charAt(0)}
                    </div>
                    <div>
                      <p className="text-sm font-medium text-gray-900">{reminder?.tenantName}</p>
                      <p className="text-xs text-gray-500">{reminder?.tenantEmail}</p>
                    </div>
                  </div>
                </td>
                <td className="px-6 py-4">
                  <div>
                    <p className="text-sm text-gray-900">{reminder?.propertyAddress}</p>
                    <p className="text-xs text-gray-500">
                      {reminder?.priority && (
                        <span className={`font-medium ${getPriorityColor(reminder?.priority)}`}>
                          {reminder?.priority?.toUpperCase()}
                        </span>
                      )}
                    </p>
                  </div>
                </td>
                <td className="px-6 py-4">
                  <p className="text-sm text-gray-900">{reminder?.templateName}</p>
                  <p className="text-xs text-gray-500">{reminder?.templateType}</p>
                </td>
                <td className="px-6 py-4">
                  <div className="flex items-center space-x-2">
                    <Icon name={getChannelIcon(reminder?.channel)} size={16} className="text-gray-500" />
                    <span className="text-sm text-gray-900 capitalize">{reminder?.channel}</span>
                  </div>
                </td>
                <td className="px-6 py-4">
                  <p className="text-sm text-gray-900">{formatDate(reminder?.scheduledDate)}</p>
                  {reminder?.sentDate && (
                    <p className="text-xs text-gray-500">Enviado: {formatDate(reminder?.sentDate)}</p>
                  )}
                </td>
                <td className="px-6 py-4">
                  <StatusIndicatorBadges status={getStatusBadge(reminder?.status)} size="sm" />
                </td>
                <td className="px-6 py-4">
                  <div className="text-center">
                    <span className="text-sm font-medium text-gray-900">{reminder?.retryCount}</span>
                    <span className="text-xs text-gray-500">/{reminder?.maxRetries}</span>
                  </div>
                </td>
                <td className="px-6 py-4">
                  <div className="flex items-center space-x-1">
                    <Button
                      variant="ghost"
                      size="sm"
                      iconName="Eye"
                      onClick={() => onReminderAction('preview', reminder)}
                      title="Vista previa"
                    />
                    <Button
                      variant="ghost"
                      size="sm"
                      iconName="Edit"
                      onClick={() => onReminderAction('edit', reminder)}
                      title="Editar"
                    />
                    <Button
                      variant="ghost"
                      size="sm"
                      iconName="Send"
                      onClick={() => onReminderAction('send', reminder)}
                      disabled={reminder?.status === 'sent' || reminder?.status === 'delivered'}
                      title="Enviar ahora"
                    />
                    <Button
                      variant="ghost"
                      size="sm"
                      iconName="X"
                      onClick={() => onReminderAction('cancel', reminder)}
                      disabled={reminder?.status === 'delivered' || reminder?.status === 'cancelled'}
                      title="Cancelar"
                    />
                  </div>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
      {filteredReminders?.length === 0 && (
        <div className="p-12 text-center">
          <Icon name="Clock" size={48} className="text-gray-300 mx-auto mb-4" />
          <h3 className="text-lg font-medium text-gray-900 mb-2">
            {filterStatus === 'all' ? 'No hay recordatorios activos' : 'No hay recordatorios con este estado'}
          </h3>
          <p className="text-gray-600">
            {filterStatus === 'all' ?'Los recordatorios aparecerán aquí cuando sean programados' :'Cambia el filtro para ver recordatorios con otros estados'
            }
          </p>
        </div>
      )}
    </div>
  );
};

export default ActiveRemindersTable;