import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';
import StatusIndicatorBadges from '../../../components/ui/StatusIndicatorBadges';
import Select from '../../../components/ui/Select';

const DeliveryTrackingPanel = ({ deliveryHistory, onRetryDelivery, onViewDetails }) => {
  const [filterPeriod, setFilterPeriod] = useState('7days');
  const [filterChannel, setFilterChannel] = useState('all');
  const [filterStatus, setFilterStatus] = useState('all');

  const periodOptions = [
    { value: '24hours', label: 'Últimas 24 horas' },
    { value: '7days', label: 'Últimos 7 días' },
    { value: '30days', label: 'Últimos 30 días' },
    { value: '90days', label: 'Últimos 90 días' }
  ];

  const channelOptions = [
    { value: 'all', label: 'Todos los Canales' },
    { value: 'email', label: 'Email' },
    { value: 'sms', label: 'SMS' },
    { value: 'whatsapp', label: 'WhatsApp' }
  ];

  const statusOptions = [
    { value: 'all', label: 'Todos los Estados' },
    { value: 'delivered', label: 'Entregados' },
    { value: 'failed', label: 'Fallidos' },
    { value: 'pending', label: 'Pendientes' },
    { value: 'bounced', label: 'Rebotados' }
  ];

  const getFilteredHistory = () => {
    const now = new Date();
    let startDate = new Date();
    
    switch (filterPeriod) {
      case '24hours':
        startDate?.setHours(now?.getHours() - 24);
        break;
      case '7days':
        startDate?.setDate(now?.getDate() - 7);
        break;
      case '30days':
        startDate?.setDate(now?.getDate() - 30);
        break;
      case '90days':
        startDate?.setDate(now?.getDate() - 90);
        break;
    }

    return deliveryHistory?.filter(item => {
      const itemDate = new Date(item.sentAt);
      const matchesPeriod = itemDate >= startDate;
      const matchesChannel = filterChannel === 'all' || item?.channel === filterChannel;
      const matchesStatus = filterStatus === 'all' || item?.status === filterStatus;
      
      return matchesPeriod && matchesChannel && matchesStatus;
    });
  };

  const getDeliveryStats = () => {
    const filtered = getFilteredHistory();
    const total = filtered?.length;
    const delivered = filtered?.filter(item => item?.status === 'delivered')?.length;
    const failed = filtered?.filter(item => item?.status === 'failed')?.length;
    const pending = filtered?.filter(item => item?.status === 'pending')?.length;
    
    return {
      total,
      delivered,
      failed,
      pending,
      deliveryRate: total > 0 ? ((delivered / total) * 100)?.toFixed(1) : 0
    };
  };

  const getChannelIcon = (channel) => {
    switch (channel) {
      case 'email': return 'Mail';
      case 'sms': return 'MessageSquare';
      case 'whatsapp': return 'MessageCircle';
      default: return 'Bell';
    }
  };

  const getStatusBadge = (status) => {
    switch (status) {
      case 'delivered': return 'paid';
      case 'failed': return 'overdue';
      case 'pending': return 'pending';
      case 'bounced': return 'inactive';
      default: return 'inactive';
    }
  };

  const formatDate = (date) => {
    return new Date(date)?.toLocaleDateString('es-CL', {
      day: '2-digit',
      month: '2-digit',
      year: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };

  const stats = getDeliveryStats();
  const filteredHistory = getFilteredHistory();

  return (
    <div className="bg-white rounded-lg border border-gray-200 overflow-hidden">
      {/* Header */}
      <div className="p-6 border-b border-gray-200">
        <div className="flex items-center space-x-3 mb-4">
          <div className="w-10 h-10 bg-secondary/10 text-secondary rounded-lg flex items-center justify-center">
            <Icon name="Activity" size={20} />
          </div>
          <div>
            <h3 className="text-lg font-semibold text-gray-900">Seguimiento de Entregas</h3>
            <p className="text-sm text-gray-600">Monitorea el estado de entrega de todos los recordatorios</p>
          </div>
        </div>

        {/* Filters */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <Select
            label="Período"
            options={periodOptions}
            value={filterPeriod}
            onChange={setFilterPeriod}
          />
          <Select
            label="Canal"
            options={channelOptions}
            value={filterChannel}
            onChange={setFilterChannel}
          />
          <Select
            label="Estado"
            options={statusOptions}
            value={filterStatus}
            onChange={setFilterStatus}
          />
        </div>
      </div>
      {/* Stats Cards */}
      <div className="p-6 bg-gray-50 border-b border-gray-200">
        <div className="grid grid-cols-2 md:grid-cols-5 gap-4">
          <div className="text-center">
            <p className="text-2xl font-bold text-gray-900">{stats?.total}</p>
            <p className="text-xs text-gray-600">Total Enviados</p>
          </div>
          <div className="text-center">
            <p className="text-2xl font-bold text-success">{stats?.delivered}</p>
            <p className="text-xs text-gray-600">Entregados</p>
          </div>
          <div className="text-center">
            <p className="text-2xl font-bold text-error">{stats?.failed}</p>
            <p className="text-xs text-gray-600">Fallidos</p>
          </div>
          <div className="text-center">
            <p className="text-2xl font-bold text-warning">{stats?.pending}</p>
            <p className="text-xs text-gray-600">Pendientes</p>
          </div>
          <div className="text-center">
            <p className="text-2xl font-bold text-primary">{stats?.deliveryRate}%</p>
            <p className="text-xs text-gray-600">Tasa de Entrega</p>
          </div>
        </div>
      </div>
      {/* Delivery History Table */}
      <div className="overflow-x-auto">
        <table className="w-full">
          <thead className="bg-gray-50 border-b border-gray-200">
            <tr>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Destinatario
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Plantilla
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Canal
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Enviado
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Estado
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Detalles
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Acciones
              </th>
            </tr>
          </thead>
          <tbody className="bg-white divide-y divide-gray-200">
            {filteredHistory?.map((item) => (
              <tr key={item?.id} className="hover:bg-gray-50 transition-colors duration-150">
                <td className="px-6 py-4">
                  <div className="flex items-center space-x-3">
                    <div className="w-8 h-8 bg-primary/10 text-primary rounded-full flex items-center justify-center text-sm font-medium">
                      {item?.recipientName?.charAt(0)}
                    </div>
                    <div>
                      <p className="text-sm font-medium text-gray-900">{item?.recipientName}</p>
                      <p className="text-xs text-gray-500">{item?.recipientContact}</p>
                    </div>
                  </div>
                </td>
                <td className="px-6 py-4">
                  <div>
                    <p className="text-sm text-gray-900">{item?.templateName}</p>
                    <p className="text-xs text-gray-500">{item?.subject}</p>
                  </div>
                </td>
                <td className="px-6 py-4">
                  <div className="flex items-center space-x-2">
                    <Icon name={getChannelIcon(item?.channel)} size={16} className="text-gray-500" />
                    <span className="text-sm text-gray-900 capitalize">{item?.channel}</span>
                  </div>
                </td>
                <td className="px-6 py-4">
                  <p className="text-sm text-gray-900">{formatDate(item?.sentAt)}</p>
                  {item?.deliveredAt && (
                    <p className="text-xs text-success">Entregado: {formatDate(item?.deliveredAt)}</p>
                  )}
                </td>
                <td className="px-6 py-4">
                  <StatusIndicatorBadges status={getStatusBadge(item?.status)} size="sm" />
                </td>
                <td className="px-6 py-4">
                  <div className="text-sm text-gray-600">
                    {item?.status === 'failed' && item?.errorMessage && (
                      <p className="text-error text-xs">{item?.errorMessage}</p>
                    )}
                    {item?.retryCount > 0 && (
                      <p className="text-xs text-gray-500">Reintentos: {item?.retryCount}</p>
                    )}
                    {item?.openedAt && (
                      <p className="text-xs text-success">Abierto: {formatDate(item?.openedAt)}</p>
                    )}
                  </div>
                </td>
                <td className="px-6 py-4">
                  <div className="flex items-center space-x-1">
                    <Button
                      variant="ghost"
                      size="sm"
                      iconName="Eye"
                      onClick={() => onViewDetails(item)}
                      title="Ver detalles"
                    />
                    {item?.status === 'failed' && (
                      <Button
                        variant="ghost"
                        size="sm"
                        iconName="RotateCcw"
                        onClick={() => onRetryDelivery(item)}
                        title="Reintentar envío"
                      />
                    )}
                    {item?.channel === 'email' && item?.status === 'delivered' && (
                      <Button
                        variant="ghost"
                        size="sm"
                        iconName="BarChart3"
                        onClick={() => onViewDetails(item)}
                        title="Ver métricas"
                      />
                    )}
                  </div>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
      {filteredHistory?.length === 0 && (
        <div className="p-12 text-center">
          <Icon name="Activity" size={48} className="text-gray-300 mx-auto mb-4" />
          <h3 className="text-lg font-medium text-gray-900 mb-2">No hay entregas registradas</h3>
          <p className="text-gray-600">
            {filterPeriod === '24hours' ?'No se han enviado recordatorios en las últimas 24 horas' :'Ajusta los filtros para ver el historial de entregas'
            }
          </p>
        </div>
      )}
    </div>
  );
};

export default DeliveryTrackingPanel;