import React, { useState, useEffect } from 'react';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';

const MessagePreviewModal = ({ 
  isOpen, 
  onClose, 
  template, 
  recipients, 
  onConfirmSend,
  scheduledDate,
  channel 
}) => {
  const [previewData, setPreviewData] = useState({
    subject: '',
    content: '',
    recipientCount: 0,
    estimatedCost: 0
  });

  useEffect(() => {
    if (isOpen && template && recipients?.length > 0) {
      generatePreview();
    }
  }, [isOpen, template, recipients]);

  const generatePreview = () => {
    const sampleRecipient = recipients?.[0];
    const variables = {
      '{TENANT_NAME}': sampleRecipient?.tenantName || 'Juan Pérez',
      '{PROPERTY_ADDRESS}': sampleRecipient?.propertyAddress || 'Av. Providencia 1234, Providencia',
      '{AMOUNT_DUE}': '$450.000',
      '{DUE_DATE}': '15/09/2025',
      '{CURRENT_DATE}': new Date()?.toLocaleDateString('es-CL'),
      '{OWNER_NAME}': 'María González',
      '{COMPANY_NAME}': 'PropManager'
    };

    let processedSubject = template?.subject || '';
    let processedContent = template?.content || '';

    Object.entries(variables)?.forEach(([variable, value]) => {
      processedSubject = processedSubject?.replace(new RegExp(variable, 'g'), value);
      processedContent = processedContent?.replace(new RegExp(variable, 'g'), value);
    });

    const estimatedCost = calculateEstimatedCost(channel, recipients?.length);

    setPreviewData({
      subject: processedSubject,
      content: processedContent,
      recipientCount: recipients?.length,
      estimatedCost
    });
  };

  const calculateEstimatedCost = (channel, recipientCount) => {
    const costs = {
      email: 0, // Free
      sms: 50, // $50 per SMS
      whatsapp: 30 // $30 per WhatsApp message
    };

    return (costs?.[channel] || 0) * recipientCount;
  };

  const getChannelIcon = (channel) => {
    switch (channel) {
      case 'email': return 'Mail';
      case 'sms': return 'MessageSquare';
      case 'whatsapp': return 'MessageCircle';
      default: return 'Bell';
    }
  };

  const getChannelName = (channel) => {
    switch (channel) {
      case 'email': return 'Email';
      case 'sms': return 'SMS';
      case 'whatsapp': return 'WhatsApp';
      default: return 'Desconocido';
    }
  };

  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 z-1100 flex items-center justify-center p-4">
      <div className="bg-white rounded-lg shadow-xl max-w-4xl w-full max-h-[90vh] overflow-hidden">
        {/* Header */}
        <div className="p-6 border-b border-gray-200">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-3">
              <div className="w-10 h-10 bg-primary/10 text-primary rounded-lg flex items-center justify-center">
                <Icon name="Eye" size={20} />
              </div>
              <div>
                <h3 className="text-lg font-semibold text-gray-900">Vista Previa del Mensaje</h3>
                <p className="text-sm text-gray-600">Revisa el contenido antes de enviar</p>
              </div>
            </div>
            <Button
              variant="ghost"
              size="sm"
              iconName="X"
              onClick={onClose}
            />
          </div>
        </div>

        {/* Content */}
        <div className="p-6 overflow-y-auto max-h-[calc(90vh-200px)]">
          {/* Summary */}
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6 p-4 bg-gray-50 rounded-lg">
            <div className="text-center">
              <div className="flex items-center justify-center space-x-2 mb-1">
                <Icon name={getChannelIcon(channel)} size={16} className="text-primary" />
                <span className="text-sm font-medium text-gray-900">{getChannelName(channel)}</span>
              </div>
              <p className="text-xs text-gray-600">Canal</p>
            </div>
            <div className="text-center">
              <p className="text-lg font-bold text-gray-900">{previewData?.recipientCount}</p>
              <p className="text-xs text-gray-600">Destinatarios</p>
            </div>
            <div className="text-center">
              <p className="text-lg font-bold text-gray-900">
                {scheduledDate ? new Date(scheduledDate)?.toLocaleDateString('es-CL') : 'Inmediato'}
              </p>
              <p className="text-xs text-gray-600">Fecha de Envío</p>
            </div>
            <div className="text-center">
              <p className="text-lg font-bold text-gray-900">
                {previewData?.estimatedCost > 0 ? `$${previewData?.estimatedCost?.toLocaleString('es-CL')}` : 'Gratis'}
              </p>
              <p className="text-xs text-gray-600">Costo Estimado</p>
            </div>
          </div>

          {/* Message Preview */}
          <div className="space-y-6">
            {/* Subject (for email) */}
            {channel === 'email' && previewData?.subject && (
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Asunto</label>
                <div className="p-3 bg-gray-50 rounded-lg border">
                  <p className="text-sm text-gray-900">{previewData?.subject}</p>
                </div>
              </div>
            )}

            {/* Message Content */}
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Contenido del Mensaje</label>
              <div className="p-4 bg-gray-50 rounded-lg border min-h-32">
                <div className="whitespace-pre-wrap text-sm text-gray-900">
                  {previewData?.content}
                </div>
              </div>
            </div>

            {/* Variables Used */}
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">Variables Utilizadas</label>
              <div className="flex flex-wrap gap-2">
                {['{TENANT_NAME}', '{PROPERTY_ADDRESS}', '{AMOUNT_DUE}', '{DUE_DATE}', '{CURRENT_DATE}']?.map((variable) => {
                  const isUsed = template?.content?.includes(variable) || template?.subject?.includes(variable);
                  return (
                    <span
                      key={variable}
                      className={`px-2 py-1 text-xs rounded-full ${
                        isUsed 
                          ? 'bg-primary/10 text-primary border border-primary/20' :'bg-gray-100 text-gray-500'
                      }`}
                    >
                      {variable}
                    </span>
                  );
                })}
              </div>
            </div>

            {/* Recipient Sample */}
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Muestra de Destinatarios ({Math.min(3, recipients?.length)} de {recipients?.length})
              </label>
              <div className="space-y-2">
                {recipients?.slice(0, 3)?.map((recipient, index) => (
                  <div key={index} className="flex items-center space-x-3 p-3 bg-gray-50 rounded-lg">
                    <div className="w-8 h-8 bg-primary/10 text-primary rounded-full flex items-center justify-center text-sm font-medium">
                      {recipient?.tenantName?.charAt(0) || 'T'}
                    </div>
                    <div className="flex-1">
                      <p className="text-sm font-medium text-gray-900">{recipient?.tenantName}</p>
                      <p className="text-xs text-gray-500">{recipient?.propertyAddress}</p>
                    </div>
                    <div className="text-right">
                      <p className="text-xs text-gray-500">
                        {channel === 'email' ? recipient?.tenantEmail : recipient?.tenantPhone}
                      </p>
                    </div>
                  </div>
                ))}
                {recipients?.length > 3 && (
                  <div className="text-center py-2">
                    <span className="text-xs text-gray-500">
                      y {recipients?.length - 3} destinatarios más...
                    </span>
                  </div>
                )}
              </div>
            </div>

            {/* Cost Breakdown */}
            {previewData?.estimatedCost > 0 && (
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">Desglose de Costos</label>
                <div className="p-3 bg-yellow-50 rounded-lg border border-yellow-200">
                  <div className="flex items-center justify-between text-sm">
                    <span className="text-gray-700">
                      {previewData?.recipientCount} mensajes × ${calculateEstimatedCost(channel, 1)?.toLocaleString('es-CL')}
                    </span>
                    <span className="font-medium text-gray-900">
                      ${previewData?.estimatedCost?.toLocaleString('es-CL')}
                    </span>
                  </div>
                </div>
              </div>
            )}
          </div>
        </div>

        {/* Footer */}
        <div className="p-6 border-t border-gray-200 bg-gray-50">
          <div className="flex items-center justify-between">
            <div className="text-sm text-gray-600">
              <Icon name="Info" size={16} className="inline mr-1" />
              Los mensajes se enviarán según la configuración programada
            </div>
            <div className="flex items-center space-x-3">
              <Button
                variant="outline"
                onClick={onClose}
              >
                Cancelar
              </Button>
              <Button
                variant="default"
                iconName="Send"
                iconPosition="left"
                onClick={() => {
                  onConfirmSend();
                  onClose();
                }}
              >
                Confirmar y Enviar
              </Button>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default MessagePreviewModal;