import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';
import { Checkbox } from '../../../components/ui/Checkbox';

const ReminderConfigurationPanel = ({ config, onConfigUpdate }) => {
  const [localConfig, setLocalConfig] = useState({
    globalSettings: {
      defaultRetryLimit: 3,
      defaultRetryInterval: 24, // hours
      deliveryTimeWindow: {
        start: '09:00',
        end: '18:00'
      },
      enableWeekendDelivery: false,
      timezone: 'America/Santiago'
    },
    emailSettings: {
      enabled: true,
      fromName: 'PropManager',
      fromEmail: 'noreply@propmanager.cl',
      replyToEmail: 'soporte@propmanager.cl',
      trackOpens: true,
      trackClicks: true
    },
    smsSettings: {
      enabled: true,
      provider: 'twilio',
      senderId: 'PropManager',
      enableDeliveryReports: true
    },
    whatsappSettings: {
      enabled: false,
      businessNumber: '+56912345678',
      enableTemplateMessages: true
    },
    automationRules: {
      paymentReminders: {
        enabled: true,
        daysBeforeDue: [7, 3, 1],
        escalationEnabled: true
      },
      maintenanceReminders: {
        enabled: true,
        daysBeforeScheduled: [3, 1]
      },
      contractRenewalReminders: {
        enabled: true,
        daysBeforeExpiry: [60, 30, 15, 7]
      }
    },
    ...config
  });

  const timezoneOptions = [
    { value: 'America/Santiago', label: 'Santiago (Chile)' },
    { value: 'America/Argentina/Buenos_Aires', label: 'Buenos Aires' },
    { value: 'America/Lima', label: 'Lima' },
    { value: 'America/Bogota', label: 'Bogotá' }
  ];

  const smsProviderOptions = [
    { value: 'twilio', label: 'Twilio' },
    { value: 'aws-sns', label: 'AWS SNS' },
    { value: 'messagebird', label: 'MessageBird' }
  ];

  const retryIntervalOptions = [
    { value: 1, label: '1 hora' },
    { value: 6, label: '6 horas' },
    { value: 12, label: '12 horas' },
    { value: 24, label: '24 horas' },
    { value: 48, label: '48 horas' }
  ];

  const handleConfigChange = (section, field, value) => {
    setLocalConfig(prev => ({
      ...prev,
      [section]: {
        ...prev?.[section],
        [field]: value
      }
    }));
  };

  const handleNestedConfigChange = (section, subsection, field, value) => {
    setLocalConfig(prev => ({
      ...prev,
      [section]: {
        ...prev?.[section],
        [subsection]: {
          ...prev?.[section]?.[subsection],
          [field]: value
        }
      }
    }));
  };

  const handleSaveConfig = () => {
    onConfigUpdate(localConfig);
  };

  const testConnection = (channel) => {
    console.log(`Testing ${channel} connection...`);
    // Simulate connection test
    setTimeout(() => {
      alert(`Conexión ${channel} exitosa`);
    }, 1000);
  };

  return (
    <div className="bg-white rounded-lg border border-gray-200 overflow-hidden">
      <div className="p-6 border-b border-gray-200">
        <div className="flex items-center space-x-3">
          <div className="w-10 h-10 bg-secondary/10 text-secondary rounded-lg flex items-center justify-center">
            <Icon name="Settings" size={20} />
          </div>
          <div>
            <h3 className="text-lg font-semibold text-gray-900">Configuración del Sistema</h3>
            <p className="text-sm text-gray-600">Gestiona las configuraciones globales de recordatorios</p>
          </div>
        </div>
      </div>
      <div className="p-6 space-y-8">
        {/* Global Settings */}
        <div>
          <h4 className="text-md font-medium text-gray-900 mb-4 flex items-center space-x-2">
            <Icon name="Globe" size={18} className="text-gray-600" />
            <span>Configuración Global</span>
          </h4>
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
            <Input
              label="Límite de Reintentos por Defecto"
              type="number"
              min="0"
              max="10"
              value={localConfig?.globalSettings?.defaultRetryLimit}
              onChange={(e) => handleConfigChange('globalSettings', 'defaultRetryLimit', parseInt(e?.target?.value))}
            />
            <Select
              label="Intervalo de Reintento por Defecto"
              options={retryIntervalOptions}
              value={localConfig?.globalSettings?.defaultRetryInterval}
              onChange={(value) => handleConfigChange('globalSettings', 'defaultRetryInterval', value)}
            />
          </div>

          <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
            <Input
              label="Ventana de Entrega - Inicio"
              type="time"
              value={localConfig?.globalSettings?.deliveryTimeWindow?.start}
              onChange={(e) => handleNestedConfigChange('globalSettings', 'deliveryTimeWindow', 'start', e?.target?.value)}
            />
            <Input
              label="Ventana de Entrega - Fin"
              type="time"
              value={localConfig?.globalSettings?.deliveryTimeWindow?.end}
              onChange={(e) => handleNestedConfigChange('globalSettings', 'deliveryTimeWindow', 'end', e?.target?.value)}
            />
            <Select
              label="Zona Horaria"
              options={timezoneOptions}
              value={localConfig?.globalSettings?.timezone}
              onChange={(value) => handleConfigChange('globalSettings', 'timezone', value)}
            />
          </div>

          <Checkbox
            label="Habilitar Entrega en Fines de Semana"
            description="Los recordatorios se pueden enviar sábados y domingos"
            checked={localConfig?.globalSettings?.enableWeekendDelivery}
            onChange={(e) => handleConfigChange('globalSettings', 'enableWeekendDelivery', e?.target?.checked)}
          />
        </div>

        {/* Email Settings */}
        <div className="border-t border-gray-200 pt-6">
          <h4 className="text-md font-medium text-gray-900 mb-4 flex items-center space-x-2">
            <Icon name="Mail" size={18} className="text-gray-600" />
            <span>Configuración de Email</span>
            <div className="flex items-center space-x-2">
              <Checkbox
                checked={localConfig?.emailSettings?.enabled}
                onChange={(e) => handleConfigChange('emailSettings', 'enabled', e?.target?.checked)}
              />
              <Button
                variant="outline"
                size="sm"
                iconName="TestTube"
                onClick={() => testConnection('email')}
                disabled={!localConfig?.emailSettings?.enabled}
              >
                Probar
              </Button>
            </div>
          </h4>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
            <Input
              label="Nombre del Remitente"
              value={localConfig?.emailSettings?.fromName}
              onChange={(e) => handleConfigChange('emailSettings', 'fromName', e?.target?.value)}
              disabled={!localConfig?.emailSettings?.enabled}
            />
            <Input
              label="Email del Remitente"
              type="email"
              value={localConfig?.emailSettings?.fromEmail}
              onChange={(e) => handleConfigChange('emailSettings', 'fromEmail', e?.target?.value)}
              disabled={!localConfig?.emailSettings?.enabled}
            />
          </div>

          <div className="mb-4">
            <Input
              label="Email de Respuesta"
              type="email"
              value={localConfig?.emailSettings?.replyToEmail}
              onChange={(e) => handleConfigChange('emailSettings', 'replyToEmail', e?.target?.value)}
              disabled={!localConfig?.emailSettings?.enabled}
            />
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <Checkbox
              label="Rastrear Aperturas"
              description="Registra cuando los emails son abiertos"
              checked={localConfig?.emailSettings?.trackOpens}
              onChange={(e) => handleConfigChange('emailSettings', 'trackOpens', e?.target?.checked)}
              disabled={!localConfig?.emailSettings?.enabled}
            />
            <Checkbox
              label="Rastrear Clics"
              description="Registra cuando se hace clic en enlaces"
              checked={localConfig?.emailSettings?.trackClicks}
              onChange={(e) => handleConfigChange('emailSettings', 'trackClicks', e?.target?.checked)}
              disabled={!localConfig?.emailSettings?.enabled}
            />
          </div>
        </div>

        {/* SMS Settings */}
        <div className="border-t border-gray-200 pt-6">
          <h4 className="text-md font-medium text-gray-900 mb-4 flex items-center space-x-2">
            <Icon name="MessageSquare" size={18} className="text-gray-600" />
            <span>Configuración de SMS</span>
            <div className="flex items-center space-x-2">
              <Checkbox
                checked={localConfig?.smsSettings?.enabled}
                onChange={(e) => handleConfigChange('smsSettings', 'enabled', e?.target?.checked)}
              />
              <Button
                variant="outline"
                size="sm"
                iconName="TestTube"
                onClick={() => testConnection('sms')}
                disabled={!localConfig?.smsSettings?.enabled}
              >
                Probar
              </Button>
            </div>
          </h4>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
            <Select
              label="Proveedor SMS"
              options={smsProviderOptions}
              value={localConfig?.smsSettings?.provider}
              onChange={(value) => handleConfigChange('smsSettings', 'provider', value)}
              disabled={!localConfig?.smsSettings?.enabled}
            />
            <Input
              label="ID del Remitente"
              value={localConfig?.smsSettings?.senderId}
              onChange={(e) => handleConfigChange('smsSettings', 'senderId', e?.target?.value)}
              disabled={!localConfig?.smsSettings?.enabled}
            />
          </div>

          <Checkbox
            label="Habilitar Reportes de Entrega"
            description="Recibe confirmaciones de entrega de SMS"
            checked={localConfig?.smsSettings?.enableDeliveryReports}
            onChange={(e) => handleConfigChange('smsSettings', 'enableDeliveryReports', e?.target?.checked)}
            disabled={!localConfig?.smsSettings?.enabled}
          />
        </div>

        {/* WhatsApp Settings */}
        <div className="border-t border-gray-200 pt-6">
          <h4 className="text-md font-medium text-gray-900 mb-4 flex items-center space-x-2">
            <Icon name="MessageCircle" size={18} className="text-gray-600" />
            <span>Configuración de WhatsApp</span>
            <div className="flex items-center space-x-2">
              <Checkbox
                checked={localConfig?.whatsappSettings?.enabled}
                onChange={(e) => handleConfigChange('whatsappSettings', 'enabled', e?.target?.checked)}
              />
              <Button
                variant="outline"
                size="sm"
                iconName="TestTube"
                onClick={() => testConnection('whatsapp')}
                disabled={!localConfig?.whatsappSettings?.enabled}
              >
                Probar
              </Button>
            </div>
          </h4>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
            <Input
              label="Número de Negocio"
              type="tel"
              value={localConfig?.whatsappSettings?.businessNumber}
              onChange={(e) => handleConfigChange('whatsappSettings', 'businessNumber', e?.target?.value)}
              disabled={!localConfig?.whatsappSettings?.enabled}
            />
          </div>

          <Checkbox
            label="Habilitar Mensajes de Plantilla"
            description="Usa plantillas pre-aprobadas de WhatsApp Business"
            checked={localConfig?.whatsappSettings?.enableTemplateMessages}
            onChange={(e) => handleConfigChange('whatsappSettings', 'enableTemplateMessages', e?.target?.checked)}
            disabled={!localConfig?.whatsappSettings?.enabled}
          />
        </div>

        {/* Automation Rules */}
        <div className="border-t border-gray-200 pt-6">
          <h4 className="text-md font-medium text-gray-900 mb-4 flex items-center space-x-2">
            <Icon name="Zap" size={18} className="text-gray-600" />
            <span>Reglas de Automatización</span>
          </h4>

          <div className="space-y-4">
            <div className="p-4 border border-gray-200 rounded-lg">
              <div className="flex items-center justify-between mb-3">
                <h5 className="font-medium text-gray-900">Recordatorios de Pago</h5>
                <Checkbox
                  checked={localConfig?.automationRules?.paymentReminders?.enabled}
                  onChange={(e) => handleNestedConfigChange('automationRules', 'paymentReminders', 'enabled', e?.target?.checked)}
                />
              </div>
              <p className="text-sm text-gray-600 mb-3">
                Envía recordatorios automáticos antes de la fecha de vencimiento del pago
              </p>
              <div className="text-sm text-gray-700">
                <span className="font-medium">Días antes:</span> 7, 3, 1
              </div>
            </div>

            <div className="p-4 border border-gray-200 rounded-lg">
              <div className="flex items-center justify-between mb-3">
                <h5 className="font-medium text-gray-900">Recordatorios de Mantenimiento</h5>
                <Checkbox
                  checked={localConfig?.automationRules?.maintenanceReminders?.enabled}
                  onChange={(e) => handleNestedConfigChange('automationRules', 'maintenanceReminders', 'enabled', e?.target?.checked)}
                />
              </div>
              <p className="text-sm text-gray-600 mb-3">
                Notifica sobre mantenimientos programados
              </p>
              <div className="text-sm text-gray-700">
                <span className="font-medium">Días antes:</span> 3, 1
              </div>
            </div>

            <div className="p-4 border border-gray-200 rounded-lg">
              <div className="flex items-center justify-between mb-3">
                <h5 className="font-medium text-gray-900">Renovación de Contratos</h5>
                <Checkbox
                  checked={localConfig?.automationRules?.contractRenewalReminders?.enabled}
                  onChange={(e) => handleNestedConfigChange('automationRules', 'contractRenewalReminders', 'enabled', e?.target?.checked)}
                />
              </div>
              <p className="text-sm text-gray-600 mb-3">
                Alerta sobre contratos próximos a vencer
              </p>
              <div className="text-sm text-gray-700">
                <span className="font-medium">Días antes:</span> 60, 30, 15, 7
              </div>
            </div>
          </div>
        </div>

        {/* Save Button */}
        <div className="flex items-center justify-end pt-6 border-t border-gray-200">
          <Button
            variant="default"
            iconName="Save"
            iconPosition="left"
            onClick={handleSaveConfig}
          >
            Guardar Configuración
          </Button>
        </div>
      </div>
    </div>
  );
};

export default ReminderConfigurationPanel;