import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';
import { Checkbox } from '../../../components/ui/Checkbox';

const ReminderScheduler = ({ templates, properties, tenants, onScheduleReminder }) => {
  const [schedulerData, setSchedulerData] = useState({
    templateId: '',
    recipients: [],
    schedulingType: 'manual', // manual, automatic
    scheduledDate: '',
    scheduledTime: '09:00',
    daysBeforeDue: 3,
    channel: 'email',
    priority: 'medium',
    maxRetries: 3,
    retryInterval: 24, // hours
    enableAutoRetry: true
  });

  const [selectedProperties, setSelectedProperties] = useState([]);
  const [selectedTenants, setSelectedTenants] = useState([]);
  const [recipientType, setRecipientType] = useState('property'); // property, tenant, custom

  const schedulingTypes = [
    { value: 'manual', label: 'Programación Manual' },
    { value: 'automatic', label: 'Programación Automática' }
  ];

  const channelOptions = [
    { value: 'email', label: 'Email' },
    { value: 'sms', label: 'SMS' },
    { value: 'whatsapp', label: 'WhatsApp' }
  ];

  const priorityOptions = [
    { value: 'low', label: 'Baja' },
    { value: 'medium', label: 'Media' },
    { value: 'high', label: 'Alta' }
  ];

  const recipientTypeOptions = [
    { value: 'property', label: 'Por Propiedad' },
    { value: 'tenant', label: 'Por Inquilino' },
    { value: 'custom', label: 'Selección Personalizada' }
  ];

  const retryIntervalOptions = [
    { value: 1, label: '1 hora' },
    { value: 6, label: '6 horas' },
    { value: 12, label: '12 horas' },
    { value: 24, label: '24 horas' },
    { value: 48, label: '48 horas' }
  ];

  const handlePropertySelection = (propertyId, checked) => {
    if (checked) {
      setSelectedProperties(prev => [...prev, propertyId]);
    } else {
      setSelectedProperties(prev => prev?.filter(id => id !== propertyId));
    }
  };

  const handleTenantSelection = (tenantId, checked) => {
    if (checked) {
      setSelectedTenants(prev => [...prev, tenantId]);
    } else {
      setSelectedTenants(prev => prev?.filter(id => id !== tenantId));
    }
  };

  const getRecipientCount = () => {
    if (recipientType === 'property') {
      return selectedProperties?.reduce((count, propId) => {
        const property = properties?.find(p => p?.id === propId);
        return count + (property?.tenants?.length || 0);
      }, 0);
    } else if (recipientType === 'tenant') {
      return selectedTenants?.length;
    }
    return 0;
  };

  const handleScheduleReminder = () => {
    const recipients = [];
    
    if (recipientType === 'property') {
      selectedProperties?.forEach(propId => {
        const property = properties?.find(p => p?.id === propId);
        if (property?.tenants) {
          property?.tenants?.forEach(tenant => {
            recipients?.push({
              tenantId: tenant?.id,
              propertyId: propId,
              tenantName: tenant?.name,
              tenantEmail: tenant?.email,
              tenantPhone: tenant?.phone,
              propertyAddress: property?.address
            });
          });
        }
      });
    } else if (recipientType === 'tenant') {
      selectedTenants?.forEach(tenantId => {
        const tenant = tenants?.find(t => t?.id === tenantId);
        if (tenant) {
          recipients?.push({
            tenantId: tenant?.id,
            propertyId: tenant?.propertyId,
            tenantName: tenant?.name,
            tenantEmail: tenant?.email,
            tenantPhone: tenant?.phone,
            propertyAddress: tenant?.propertyAddress
          });
        }
      });
    }

    const reminderData = {
      ...schedulerData,
      recipients,
      id: Date.now()?.toString(),
      createdAt: new Date(),
      status: 'scheduled'
    };

    onScheduleReminder(reminderData);
    
    // Reset form
    setSchedulerData({
      templateId: '',
      recipients: [],
      schedulingType: 'manual',
      scheduledDate: '',
      scheduledTime: '09:00',
      daysBeforeDue: 3,
      channel: 'email',
      priority: 'medium',
      maxRetries: 3,
      retryInterval: 24,
      enableAutoRetry: true
    });
    setSelectedProperties([]);
    setSelectedTenants([]);
  };

  const isFormValid = () => {
    return schedulerData?.templateId && 
           getRecipientCount() > 0 && 
           (schedulerData?.schedulingType === 'automatic' || schedulerData?.scheduledDate);
  };

  return (
    <div className="bg-white rounded-lg border border-gray-200 overflow-hidden">
      <div className="p-6 border-b border-gray-200">
        <div className="flex items-center space-x-3">
          <div className="w-10 h-10 bg-primary/10 text-primary rounded-lg flex items-center justify-center">
            <Icon name="Calendar" size={20} />
          </div>
          <div>
            <h3 className="text-lg font-semibold text-gray-900">Programar Recordatorios</h3>
            <p className="text-sm text-gray-600">Configura y programa recordatorios automáticos o manuales</p>
          </div>
        </div>
      </div>
      <div className="p-6 space-y-6">
        {/* Template Selection */}
        <div>
          <Select
            label="Plantilla de Recordatorio"
            description="Selecciona la plantilla que se utilizará para el recordatorio"
            options={templates?.map(t => ({ value: t?.id, label: t?.name, description: t?.type }))}
            value={schedulerData?.templateId}
            onChange={(value) => setSchedulerData(prev => ({ ...prev, templateId: value }))}
            required
          />
        </div>

        {/* Scheduling Type */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <Select
            label="Tipo de Programación"
            options={schedulingTypes}
            value={schedulerData?.schedulingType}
            onChange={(value) => setSchedulerData(prev => ({ ...prev, schedulingType: value }))}
          />
          <Select
            label="Canal de Comunicación"
            options={channelOptions}
            value={schedulerData?.channel}
            onChange={(value) => setSchedulerData(prev => ({ ...prev, channel: value }))}
          />
        </div>

        {/* Scheduling Configuration */}
        {schedulerData?.schedulingType === 'manual' ? (
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <Input
              label="Fecha de Envío"
              type="date"
              value={schedulerData?.scheduledDate}
              onChange={(e) => setSchedulerData(prev => ({ ...prev, scheduledDate: e?.target?.value }))}
              required
            />
            <Input
              label="Hora de Envío"
              type="time"
              value={schedulerData?.scheduledTime}
              onChange={(e) => setSchedulerData(prev => ({ ...prev, scheduledTime: e?.target?.value }))}
            />
          </div>
        ) : (
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <Input
              label="Días Antes del Vencimiento"
              type="number"
              min="1"
              max="30"
              value={schedulerData?.daysBeforeDue}
              onChange={(e) => setSchedulerData(prev => ({ ...prev, daysBeforeDue: parseInt(e?.target?.value) }))}
              description="Los recordatorios se enviarán automáticamente X días antes del vencimiento"
            />
            <Input
              label="Hora de Envío"
              type="time"
              value={schedulerData?.scheduledTime}
              onChange={(e) => setSchedulerData(prev => ({ ...prev, scheduledTime: e?.target?.value }))}
            />
          </div>
        )}

        {/* Priority and Retry Settings */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <Select
            label="Prioridad"
            options={priorityOptions}
            value={schedulerData?.priority}
            onChange={(value) => setSchedulerData(prev => ({ ...prev, priority: value }))}
          />
          <Input
            label="Máximo de Reintentos"
            type="number"
            min="0"
            max="10"
            value={schedulerData?.maxRetries}
            onChange={(e) => setSchedulerData(prev => ({ ...prev, maxRetries: parseInt(e?.target?.value) }))}
          />
          <Select
            label="Intervalo de Reintento"
            options={retryIntervalOptions}
            value={schedulerData?.retryInterval}
            onChange={(value) => setSchedulerData(prev => ({ ...prev, retryInterval: value }))}
          />
        </div>

        {/* Auto Retry Toggle */}
        <div>
          <Checkbox
            label="Habilitar Reintentos Automáticos"
            description="Los recordatorios fallidos se reenviarán automáticamente según la configuración"
            checked={schedulerData?.enableAutoRetry}
            onChange={(e) => setSchedulerData(prev => ({ ...prev, enableAutoRetry: e?.target?.checked }))}
          />
        </div>

        {/* Recipient Selection */}
        <div className="border-t border-gray-200 pt-6">
          <h4 className="text-md font-medium text-gray-900 mb-4">Selección de Destinatarios</h4>
          
          <div className="mb-4">
            <Select
              label="Tipo de Selección"
              options={recipientTypeOptions}
              value={recipientType}
              onChange={setRecipientType}
            />
          </div>

          {recipientType === 'property' && (
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-3">
                Seleccionar Propiedades ({selectedProperties?.length} seleccionadas)
              </label>
              <div className="max-h-64 overflow-y-auto border border-gray-200 rounded-lg">
                {properties?.map((property) => (
                  <div key={property?.id} className="p-3 border-b border-gray-100 last:border-b-0">
                    <Checkbox
                      label={
                        <div className="flex items-center justify-between w-full">
                          <div>
                            <p className="text-sm font-medium text-gray-900">{property?.address}</p>
                            <p className="text-xs text-gray-500">
                              {property?.tenants?.length || 0} inquilino(s)
                            </p>
                          </div>
                          <span className="text-xs text-gray-400">{property?.type}</span>
                        </div>
                      }
                      checked={selectedProperties?.includes(property?.id)}
                      onChange={(e) => handlePropertySelection(property?.id, e?.target?.checked)}
                    />
                  </div>
                ))}
              </div>
            </div>
          )}

          {recipientType === 'tenant' && (
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-3">
                Seleccionar Inquilinos ({selectedTenants?.length} seleccionados)
              </label>
              <div className="max-h-64 overflow-y-auto border border-gray-200 rounded-lg">
                {tenants?.map((tenant) => (
                  <div key={tenant?.id} className="p-3 border-b border-gray-100 last:border-b-0">
                    <Checkbox
                      label={
                        <div className="flex items-center justify-between w-full">
                          <div>
                            <p className="text-sm font-medium text-gray-900">{tenant?.name}</p>
                            <p className="text-xs text-gray-500">{tenant?.propertyAddress}</p>
                          </div>
                          <span className="text-xs text-gray-400">{tenant?.email}</span>
                        </div>
                      }
                      checked={selectedTenants?.includes(tenant?.id)}
                      onChange={(e) => handleTenantSelection(tenant?.id, e?.target?.checked)}
                    />
                  </div>
                ))}
              </div>
            </div>
          )}

          {getRecipientCount() > 0 && (
            <div className="mt-4 p-3 bg-primary/5 rounded-lg border border-primary/20">
              <p className="text-sm text-primary font-medium">
                Se enviarán recordatorios a {getRecipientCount()} destinatario(s)
              </p>
            </div>
          )}
        </div>

        {/* Action Buttons */}
        <div className="flex items-center justify-end space-x-3 pt-6 border-t border-gray-200">
          <Button
            variant="outline"
            onClick={() => {
              setSchedulerData({
                templateId: '',
                recipients: [],
                schedulingType: 'manual',
                scheduledDate: '',
                scheduledTime: '09:00',
                daysBeforeDue: 3,
                channel: 'email',
                priority: 'medium',
                maxRetries: 3,
                retryInterval: 24,
                enableAutoRetry: true
              });
              setSelectedProperties([]);
              setSelectedTenants([]);
            }}
          >
            Limpiar
          </Button>
          <Button
            variant="default"
            iconName="Calendar"
            iconPosition="left"
            onClick={handleScheduleReminder}
            disabled={!isFormValid()}
          >
            Programar Recordatorio
          </Button>
        </div>
      </div>
    </div>
  );
};

export default ReminderScheduler;