import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';

const ReminderTemplateManager = ({ templates, onTemplateCreate, onTemplateUpdate, onTemplateDelete }) => {
  const [isCreating, setIsCreating] = useState(false);
  const [editingTemplate, setEditingTemplate] = useState(null);
  const [newTemplate, setNewTemplate] = useState({
    name: '',
    subject: '',
    content: '',
    type: 'payment',
    channel: 'email'
  });

  const templateTypes = [
    { value: 'payment', label: 'Recordatorio de Pago' },
    { value: 'maintenance', label: 'Mantenimiento' },
    { value: 'contract', label: 'Contrato' },
    { value: 'inspection', label: 'Inspección' },
    { value: 'general', label: 'General' }
  ];

  const channelOptions = [
    { value: 'email', label: 'Email' },
    { value: 'sms', label: 'SMS' },
    { value: 'whatsapp', label: 'WhatsApp' }
  ];

  const availableVariables = [
    { key: '{TENANT_NAME}', description: 'Nombre del inquilino' },
    { key: '{PROPERTY_ADDRESS}', description: 'Dirección de la propiedad' },
    { key: '{AMOUNT_DUE}', description: 'Monto adeudado' },
    { key: '{DUE_DATE}', description: 'Fecha de vencimiento' },
    { key: '{CURRENT_DATE}', description: 'Fecha actual' },
    { key: '{OWNER_NAME}', description: 'Nombre del propietario' },
    { key: '{COMPANY_NAME}', description: 'Nombre de la empresa' }
  ];

  const handleCreateTemplate = () => {
    if (newTemplate?.name && newTemplate?.content) {
      onTemplateCreate({
        ...newTemplate,
        id: Date.now()?.toString(),
        createdAt: new Date(),
        lastUsed: null,
        usageCount: 0
      });
      setNewTemplate({ name: '', subject: '', content: '', type: 'payment', channel: 'email' });
      setIsCreating(false);
    }
  };

  const handleEditTemplate = (template) => {
    setEditingTemplate({ ...template });
  };

  const handleUpdateTemplate = () => {
    if (editingTemplate?.name && editingTemplate?.content) {
      onTemplateUpdate(editingTemplate);
      setEditingTemplate(null);
    }
  };

  const insertVariable = (variable, isEditing = false) => {
    if (isEditing) {
      setEditingTemplate(prev => ({
        ...prev,
        content: prev?.content + variable
      }));
    } else {
      setNewTemplate(prev => ({
        ...prev,
        content: prev?.content + variable
      }));
    }
  };

  const getChannelIcon = (channel) => {
    switch (channel) {
      case 'email': return 'Mail';
      case 'sms': return 'MessageSquare';
      case 'whatsapp': return 'MessageCircle';
      default: return 'Bell';
    }
  };

  const getTypeColor = (type) => {
    switch (type) {
      case 'payment': return 'text-error';
      case 'maintenance': return 'text-warning';
      case 'contract': return 'text-primary';
      case 'inspection': return 'text-secondary';
      default: return 'text-gray-600';
    }
  };

  return (
    <div className="bg-white rounded-lg border border-gray-200 overflow-hidden">
      <div className="p-6 border-b border-gray-200">
        <div className="flex items-center justify-between">
          <div>
            <h3 className="text-lg font-semibold text-gray-900">Plantillas de Recordatorios</h3>
            <p className="text-sm text-gray-600 mt-1">Gestiona plantillas personalizables para diferentes tipos de comunicación</p>
          </div>
          <Button
            variant="default"
            iconName="Plus"
            iconPosition="left"
            onClick={() => setIsCreating(true)}
          >
            Nueva Plantilla
          </Button>
        </div>
      </div>
      {/* Template Creation Form */}
      {isCreating && (
        <div className="p-6 bg-gray-50 border-b border-gray-200">
          <h4 className="text-md font-medium text-gray-900 mb-4">Crear Nueva Plantilla</h4>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-4">
            <Input
              label="Nombre de la Plantilla"
              placeholder="Ej: Recordatorio Pago Mensual"
              value={newTemplate?.name}
              onChange={(e) => setNewTemplate(prev => ({ ...prev, name: e?.target?.value }))}
              required
            />
            <Select
              label="Tipo de Recordatorio"
              options={templateTypes}
              value={newTemplate?.type}
              onChange={(value) => setNewTemplate(prev => ({ ...prev, type: value }))}
            />
            <Input
              label="Asunto (Email/SMS)"
              placeholder="Recordatorio de pago - {PROPERTY_ADDRESS}"
              value={newTemplate?.subject}
              onChange={(e) => setNewTemplate(prev => ({ ...prev, subject: e?.target?.value }))}
            />
            <Select
              label="Canal de Comunicación"
              options={channelOptions}
              value={newTemplate?.channel}
              onChange={(value) => setNewTemplate(prev => ({ ...prev, channel: value }))}
            />
          </div>

          <div className="mb-4">
            <label className="block text-sm font-medium text-gray-700 mb-2">Contenido del Mensaje</label>
            <textarea
              className="w-full h-32 px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-primary focus:border-transparent resize-none"
              placeholder="Estimado/a {TENANT_NAME}, le recordamos que el pago de..."
              value={newTemplate?.content}
              onChange={(e) => setNewTemplate(prev => ({ ...prev, content: e?.target?.value }))}
            />
          </div>

          <div className="mb-4">
            <label className="block text-sm font-medium text-gray-700 mb-2">Variables Disponibles</label>
            <div className="flex flex-wrap gap-2">
              {availableVariables?.map((variable) => (
                <button
                  key={variable?.key}
                  onClick={() => insertVariable(variable?.key)}
                  className="px-3 py-1 bg-primary/10 text-primary text-xs rounded-full hover:bg-primary/20 transition-colors duration-150"
                  title={variable?.description}
                >
                  {variable?.key}
                </button>
              ))}
            </div>
          </div>

          <div className="flex items-center space-x-3">
            <Button
              variant="default"
              onClick={handleCreateTemplate}
              disabled={!newTemplate?.name || !newTemplate?.content}
            >
              Crear Plantilla
            </Button>
            <Button
              variant="outline"
              onClick={() => setIsCreating(false)}
            >
              Cancelar
            </Button>
          </div>
        </div>
      )}
      {/* Template List */}
      <div className="divide-y divide-gray-200">
        {templates?.map((template) => (
          <div key={template?.id} className="p-6">
            {editingTemplate?.id === template?.id ? (
              <div className="space-y-4">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <Input
                    label="Nombre de la Plantilla"
                    value={editingTemplate?.name}
                    onChange={(e) => setEditingTemplate(prev => ({ ...prev, name: e?.target?.value }))}
                  />
                  <Select
                    label="Tipo de Recordatorio"
                    options={templateTypes}
                    value={editingTemplate?.type}
                    onChange={(value) => setEditingTemplate(prev => ({ ...prev, type: value }))}
                  />
                  <Input
                    label="Asunto"
                    value={editingTemplate?.subject}
                    onChange={(e) => setEditingTemplate(prev => ({ ...prev, subject: e?.target?.value }))}
                  />
                  <Select
                    label="Canal"
                    options={channelOptions}
                    value={editingTemplate?.channel}
                    onChange={(value) => setEditingTemplate(prev => ({ ...prev, channel: value }))}
                  />
                </div>
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">Contenido</label>
                  <textarea
                    className="w-full h-32 px-3 py-2 border border-gray-300 rounded-md focus:ring-2 focus:ring-primary focus:border-transparent resize-none"
                    value={editingTemplate?.content}
                    onChange={(e) => setEditingTemplate(prev => ({ ...prev, content: e?.target?.value }))}
                  />
                </div>
                <div className="mb-4">
                  <div className="flex flex-wrap gap-2">
                    {availableVariables?.map((variable) => (
                      <button
                        key={variable?.key}
                        onClick={() => insertVariable(variable?.key, true)}
                        className="px-3 py-1 bg-primary/10 text-primary text-xs rounded-full hover:bg-primary/20 transition-colors duration-150"
                        title={variable?.description}
                      >
                        {variable?.key}
                      </button>
                    ))}
                  </div>
                </div>
                <div className="flex items-center space-x-3">
                  <Button variant="default" onClick={handleUpdateTemplate}>
                    Guardar Cambios
                  </Button>
                  <Button variant="outline" onClick={() => setEditingTemplate(null)}>
                    Cancelar
                  </Button>
                </div>
              </div>
            ) : (
              <div className="flex items-start justify-between">
                <div className="flex-1">
                  <div className="flex items-center space-x-3 mb-2">
                    <h4 className="text-md font-medium text-gray-900">{template?.name}</h4>
                    <div className="flex items-center space-x-2">
                      <span className={`text-xs font-medium ${getTypeColor(template?.type)}`}>
                        {templateTypes?.find(t => t?.value === template?.type)?.label}
                      </span>
                      <div className="flex items-center space-x-1 text-xs text-gray-500">
                        <Icon name={getChannelIcon(template?.channel)} size={12} />
                        <span>{channelOptions?.find(c => c?.value === template?.channel)?.label}</span>
                      </div>
                    </div>
                  </div>
                  {template?.subject && (
                    <p className="text-sm text-gray-600 mb-2">
                      <span className="font-medium">Asunto:</span> {template?.subject}
                    </p>
                  )}
                  <p className="text-sm text-gray-600 mb-3 line-clamp-2">{template?.content}</p>
                  <div className="flex items-center space-x-4 text-xs text-gray-500">
                    <span>Usado {template?.usageCount} veces</span>
                    {template?.lastUsed && (
                      <span>Último uso: {new Date(template.lastUsed)?.toLocaleDateString('es-CL')}</span>
                    )}
                  </div>
                </div>
                <div className="flex items-center space-x-2 ml-4">
                  <Button
                    variant="ghost"
                    size="sm"
                    iconName="Edit"
                    onClick={() => handleEditTemplate(template)}
                  />
                  <Button
                    variant="ghost"
                    size="sm"
                    iconName="Trash2"
                    onClick={() => onTemplateDelete(template?.id)}
                  />
                </div>
              </div>
            )}
          </div>
        ))}
      </div>
      {templates?.length === 0 && !isCreating && (
        <div className="p-12 text-center">
          <Icon name="FileText" size={48} className="text-gray-300 mx-auto mb-4" />
          <h3 className="text-lg font-medium text-gray-900 mb-2">No hay plantillas creadas</h3>
          <p className="text-gray-600 mb-4">Crea tu primera plantilla para automatizar tus recordatorios</p>
          <Button
            variant="default"
            iconName="Plus"
            iconPosition="left"
            onClick={() => setIsCreating(true)}
          >
            Crear Primera Plantilla
          </Button>
        </div>
      )}
    </div>
  );
};

export default ReminderTemplateManager;