import React, { useState, useEffect } from 'react';
import Header from '../../components/ui/Header';
import Sidebar from '../../components/ui/Sidebar';
import NavigationBreadcrumbs from '../../components/ui/NavigationBreadcrumbs';
import QuickActionFloatingButton from '../../components/ui/QuickActionFloatingButton';
import Icon from '../../components/AppIcon';


// Import all components
import ReminderTemplateManager from './components/ReminderTemplateManager';
import ActiveRemindersTable from './components/ActiveRemindersTable';
import ReminderScheduler from './components/ReminderScheduler';
import DeliveryTrackingPanel from './components/DeliveryTrackingPanel';
import MessagePreviewModal from './components/MessagePreviewModal';
import ReminderConfigurationPanel from './components/ReminderConfigurationPanel';

const RemindersSystem = () => {
  const [isSidebarOpen, setIsSidebarOpen] = useState(false);
  const [activeTab, setActiveTab] = useState('active');
  const [showPreviewModal, setShowPreviewModal] = useState(false);
  const [previewData, setPreviewData] = useState(null);

  // Mock data
  const [templates, setTemplates] = useState([
    {
      id: '1',
      name: 'Recordatorio Pago Mensual',
      subject: 'Recordatorio de Pago - {PROPERTY_ADDRESS}',
      content: `Estimado/a {TENANT_NAME},\n\nEsperamos que se encuentre bien. Le recordamos que el pago correspondiente a la propiedad ubicada en {PROPERTY_ADDRESS} tiene fecha de vencimiento el {DUE_DATE}.\n\nMonto a pagar: {AMOUNT_DUE}\n\nPor favor, realice el pago antes de la fecha indicada para evitar recargos.\n\nSaludos cordiales,\n{COMPANY_NAME}`,
      type: 'payment',
      channel: 'email',
      createdAt: new Date('2025-08-20'),
      lastUsed: new Date('2025-08-25'),
      usageCount: 15
    },
    {
      id: '2',
      name: 'Recordatorio Mantenimiento',
      subject: 'Mantenimiento Programado - {PROPERTY_ADDRESS}',
      content: `Hola {TENANT_NAME},\n\nLe informamos que se ha programado un mantenimiento en {PROPERTY_ADDRESS} para el {DUE_DATE}.\n\nPor favor, asegúrese de estar disponible en el horario acordado.\n\nGracias por su colaboración.\n\n{COMPANY_NAME}`,
      type: 'maintenance',
      channel: 'sms',
      createdAt: new Date('2025-08-18'),
      lastUsed: new Date('2025-08-24'),
      usageCount: 8
    },
    {
      id: '3',
      name: 'Recordatorio Vencimiento Contrato',
      subject: 'Renovación de Contrato - {PROPERTY_ADDRESS}',
      content: `Estimado/a {TENANT_NAME},\n\nSu contrato de arriendo para la propiedad {PROPERTY_ADDRESS} vence el {DUE_DATE}.\n\nSi desea renovar, por favor contáctenos con anticipación.\n\nAtentamente,\n{OWNER_NAME}\n{COMPANY_NAME}`,
      type: 'contract',
      channel: 'email',
      createdAt: new Date('2025-08-15'),
      lastUsed: null,
      usageCount: 0
    }
  ]);

  const [activeReminders, setActiveReminders] = useState([
    {
      id: '1',
      tenantName: 'Carlos Mendoza',
      tenantEmail: 'carlos.mendoza@email.com',
      tenantPhone: '+56987654321',
      propertyAddress: 'Av. Providencia 1234, Providencia',
      templateName: 'Recordatorio Pago Mensual',
      templateType: 'payment',
      channel: 'email',
      scheduledDate: new Date('2025-08-27T09:00:00'),
      sentDate: null,
      status: 'scheduled',
      priority: 'high',
      retryCount: 0,
      maxRetries: 3
    },
    {
      id: '2',
      tenantName: 'Ana Rodríguez',
      tenantEmail: 'ana.rodriguez@email.com',
      tenantPhone: '+56912345678',
      propertyAddress: 'Las Condes 567, Las Condes',
      templateName: 'Recordatorio Pago Mensual',
      templateType: 'payment',
      channel: 'sms',
      scheduledDate: new Date('2025-08-26T10:30:00'),
      sentDate: new Date('2025-08-26T10:30:00'),
      status: 'sent',
      priority: 'medium',
      retryCount: 0,
      maxRetries: 3
    },
    {
      id: '3',
      tenantName: 'Roberto Silva',
      tenantEmail: 'roberto.silva@email.com',
      tenantPhone: '+56998765432',
      propertyAddress: 'Ñuñoa 890, Ñuñoa',
      templateName: 'Recordatorio Mantenimiento',
      templateType: 'maintenance',
      channel: 'email',
      scheduledDate: new Date('2025-08-25T14:00:00'),
      sentDate: new Date('2025-08-25T14:00:00'),
      status: 'delivered',
      priority: 'low',
      retryCount: 0,
      maxRetries: 2
    },
    {
      id: '4',
      tenantName: 'María González',
      tenantEmail: 'maria.gonzalez@email.com',
      tenantPhone: '+56987123456',
      propertyAddress: 'Vitacura 123, Vitacura',
      templateName: 'Recordatorio Pago Mensual',
      templateType: 'payment',
      channel: 'whatsapp',
      scheduledDate: new Date('2025-08-24T11:00:00'),
      sentDate: new Date('2025-08-24T11:00:00'),
      status: 'failed',
      priority: 'high',
      retryCount: 2,
      maxRetries: 3
    }
  ]);

  const [deliveryHistory] = useState([
    {
      id: '1',
      recipientName: 'Carlos Mendoza',
      recipientContact: 'carlos.mendoza@email.com',
      templateName: 'Recordatorio Pago Mensual',
      subject: 'Recordatorio de Pago - Av. Providencia 1234',
      channel: 'email',
      sentAt: new Date('2025-08-25T09:00:00'),
      deliveredAt: new Date('2025-08-25T09:02:00'),
      openedAt: new Date('2025-08-25T14:30:00'),
      status: 'delivered',
      retryCount: 0,
      errorMessage: null
    },
    {
      id: '2',
      recipientName: 'Ana Rodríguez',
      recipientContact: '+56912345678',
      templateName: 'Recordatorio Pago Mensual',
      subject: 'Recordatorio de Pago',
      channel: 'sms',
      sentAt: new Date('2025-08-24T10:30:00'),
      deliveredAt: new Date('2025-08-24T10:31:00'),
      status: 'delivered',
      retryCount: 0,
      errorMessage: null
    },
    {
      id: '3',
      recipientName: 'Roberto Silva',
      recipientContact: 'roberto.silva@email.com',
      templateName: 'Recordatorio Mantenimiento',
      subject: 'Mantenimiento Programado',
      channel: 'email',
      sentAt: new Date('2025-08-23T14:00:00'),
      deliveredAt: null,
      status: 'failed',
      retryCount: 1,
      errorMessage: 'Dirección de email inválida'
    }
  ]);

  const mockProperties = [
    {
      id: '1',
      address: 'Av. Providencia 1234, Providencia',
      type: 'Departamento',
      tenants: [
        { id: '1', name: 'Carlos Mendoza', email: 'carlos.mendoza@email.com', phone: '+56987654321' }
      ]
    },
    {
      id: '2',
      address: 'Las Condes 567, Las Condes',
      type: 'Casa',
      tenants: [
        { id: '2', name: 'Ana Rodríguez', email: 'ana.rodriguez@email.com', phone: '+56912345678' }
      ]
    },
    {
      id: '3',
      address: 'Ñuñoa 890, Ñuñoa',
      type: 'Departamento',
      tenants: [
        { id: '3', name: 'Roberto Silva', email: 'roberto.silva@email.com', phone: '+56998765432' }
      ]
    }
  ];

  const mockTenants = [
    {
      id: '1',
      name: 'Carlos Mendoza',
      email: 'carlos.mendoza@email.com',
      phone: '+56987654321',
      propertyId: '1',
      propertyAddress: 'Av. Providencia 1234, Providencia'
    },
    {
      id: '2',
      name: 'Ana Rodríguez',
      email: 'ana.rodriguez@email.com',
      phone: '+56912345678',
      propertyId: '2',
      propertyAddress: 'Las Condes 567, Las Condes'
    },
    {
      id: '3',
      name: 'Roberto Silva',
      email: 'roberto.silva@email.com',
      phone: '+56998765432',
      propertyId: '3',
      propertyAddress: 'Ñuñoa 890, Ñuñoa'
    }
  ];

  const [systemConfig, setSystemConfig] = useState({
    globalSettings: {
      defaultRetryLimit: 3,
      defaultRetryInterval: 24,
      deliveryTimeWindow: { start: '09:00', end: '18:00' },
      enableWeekendDelivery: false,
      timezone: 'America/Santiago'
    },
    emailSettings: {
      enabled: true,
      fromName: 'PropManager',
      fromEmail: 'noreply@propmanager.cl',
      replyToEmail: 'soporte@propmanager.cl',
      trackOpens: true,
      trackClicks: true
    },
    smsSettings: {
      enabled: true,
      provider: 'twilio',
      senderId: 'PropManager',
      enableDeliveryReports: true
    },
    whatsappSettings: {
      enabled: false,
      businessNumber: '+56912345678',
      enableTemplateMessages: true
    }
  });

  const tabs = [
    { id: 'active', label: 'Recordatorios Activos', icon: 'Clock', count: activeReminders?.length },
    { id: 'scheduler', label: 'Programar', icon: 'Calendar', count: null },
    { id: 'templates', label: 'Plantillas', icon: 'FileText', count: templates?.length },
    { id: 'tracking', label: 'Seguimiento', icon: 'Activity', count: null },
    { id: 'config', label: 'Configuración', icon: 'Settings', count: null }
  ];

  // Event handlers
  const handleTemplateCreate = (template) => {
    setTemplates(prev => [...prev, template]);
  };

  const handleTemplateUpdate = (updatedTemplate) => {
    setTemplates(prev => prev?.map(t => t?.id === updatedTemplate?.id ? updatedTemplate : t));
  };

  const handleTemplateDelete = (templateId) => {
    if (window.confirm('¿Estás seguro de que deseas eliminar esta plantilla?')) {
      setTemplates(prev => prev?.filter(t => t?.id !== templateId));
    }
  };

  const handleScheduleReminder = (reminderData) => {
    const newReminders = reminderData?.recipients?.map((recipient, index) => ({
      id: `${Date.now()}_${index}`,
      tenantName: recipient?.tenantName,
      tenantEmail: recipient?.tenantEmail,
      tenantPhone: recipient?.tenantPhone,
      propertyAddress: recipient?.propertyAddress,
      templateName: templates?.find(t => t?.id === reminderData?.templateId)?.name || 'Plantilla Desconocida',
      templateType: templates?.find(t => t?.id === reminderData?.templateId)?.type || 'general',
      channel: reminderData?.channel,
      scheduledDate: reminderData?.schedulingType === 'manual' 
        ? new Date(`${reminderData.scheduledDate}T${reminderData.scheduledTime}:00`)
        : new Date(Date.now() + (reminderData.daysBeforeDue * 24 * 60 * 60 * 1000)),
      sentDate: null,
      status: 'scheduled',
      priority: reminderData?.priority,
      retryCount: 0,
      maxRetries: reminderData?.maxRetries
    }));

    setActiveReminders(prev => [...prev, ...newReminders]);
    alert(`${newReminders?.length} recordatorios programados exitosamente`);
  };

  const handleReminderAction = (action, reminder) => {
    switch (action) {
      case 'preview':
        setPreviewData({
          template: templates?.find(t => t?.name === reminder?.templateName),
          recipients: [reminder],
          scheduledDate: reminder?.scheduledDate,
          channel: reminder?.channel
        });
        setShowPreviewModal(true);
        break;
      case 'edit': console.log('Editing reminder:', reminder);
        break;
      case 'send':
        setActiveReminders(prev => prev?.map(r => 
          r?.id === reminder?.id 
            ? { ...r, status: 'sent', sentDate: new Date() }
            : r
        ));
        alert('Recordatorio enviado exitosamente');
        break;
      case 'cancel':
        setActiveReminders(prev => prev?.map(r => 
          r?.id === reminder?.id 
            ? { ...r, status: 'cancelled' }
            : r
        ));
        break;
      default:
        console.log('Unknown action:', action);
    }
  };

  const handleBulkAction = (action, selectedIds) => {
    switch (action) {
      case 'send':
        setActiveReminders(prev => prev?.map(r => 
          selectedIds?.includes(r?.id) && r?.status === 'scheduled'
            ? { ...r, status: 'sent', sentDate: new Date() }
            : r
        ));
        alert(`${selectedIds?.length} recordatorios enviados`);
        break;
      case 'reschedule': console.log('Rescheduling reminders:', selectedIds);
        break;
      case 'cancel':
        setActiveReminders(prev => prev?.map(r => 
          selectedIds?.includes(r?.id)
            ? { ...r, status: 'cancelled' }
            : r
        ));
        alert(`${selectedIds?.length} recordatorios cancelados`);
        break;
      default:
        console.log('Unknown bulk action:', action);
    }
  };

  const handleRetryDelivery = (deliveryItem) => {
    console.log('Retrying delivery for:', deliveryItem);
    alert('Reintentando envío...');
  };

  const handleViewDetails = (deliveryItem) => {
    console.log('Viewing details for:', deliveryItem);
  };

  const handleConfigUpdate = (newConfig) => {
    setSystemConfig(newConfig);
    alert('Configuración guardada exitosamente');
  };

  const handlePreviewConfirm = () => {
    alert('Recordatorios enviados exitosamente');
  };

  const renderTabContent = () => {
    switch (activeTab) {
      case 'active':
        return (
          <ActiveRemindersTable
            reminders={activeReminders}
            onReminderAction={handleReminderAction}
            onBulkAction={handleBulkAction}
          />
        );
      case 'scheduler':
        return (
          <ReminderScheduler
            templates={templates}
            properties={mockProperties}
            tenants={mockTenants}
            onScheduleReminder={handleScheduleReminder}
          />
        );
      case 'templates':
        return (
          <ReminderTemplateManager
            templates={templates}
            onTemplateCreate={handleTemplateCreate}
            onTemplateUpdate={handleTemplateUpdate}
            onTemplateDelete={handleTemplateDelete}
          />
        );
      case 'tracking':
        return (
          <DeliveryTrackingPanel
            deliveryHistory={deliveryHistory}
            onRetryDelivery={handleRetryDelivery}
            onViewDetails={handleViewDetails}
          />
        );
      case 'config':
        return (
          <ReminderConfigurationPanel
            config={systemConfig}
            onConfigUpdate={handleConfigUpdate}
          />
        );
      default:
        return null;
    }
  };

  return (
    <div className="min-h-screen bg-background">
      <Header onMenuToggle={() => setIsSidebarOpen(!isSidebarOpen)} isMenuOpen={isSidebarOpen} />
      <Sidebar isOpen={isSidebarOpen} onClose={() => setIsSidebarOpen(false)} />
      <main className="lg:ml-60 pt-16">
        <div className="p-6">
          <NavigationBreadcrumbs />
          
          {/* Page Header */}
          <div className="mb-8">
            <div className="flex items-center space-x-4 mb-4">
              <div className="w-12 h-12 bg-primary/10 text-primary rounded-xl flex items-center justify-center">
                <Icon name="Bell" size={24} />
              </div>
              <div>
                <h1 className="text-2xl font-bold text-gray-900">Sistema de Recordatorios</h1>
                <p className="text-gray-600">Gestiona comunicaciones automáticas y manuales con inquilinos</p>
              </div>
            </div>

            {/* Quick Stats */}
            <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-6">
              <div className="bg-white p-4 rounded-lg border border-gray-200">
                <div className="flex items-center space-x-3">
                  <div className="w-10 h-10 bg-primary/10 text-primary rounded-lg flex items-center justify-center">
                    <Icon name="Clock" size={20} />
                  </div>
                  <div>
                    <p className="text-2xl font-bold text-gray-900">{activeReminders?.filter(r => r?.status === 'scheduled')?.length}</p>
                    <p className="text-sm text-gray-600">Programados</p>
                  </div>
                </div>
              </div>
              <div className="bg-white p-4 rounded-lg border border-gray-200">
                <div className="flex items-center space-x-3">
                  <div className="w-10 h-10 bg-success/10 text-success rounded-lg flex items-center justify-center">
                    <Icon name="CheckCircle" size={20} />
                  </div>
                  <div>
                    <p className="text-2xl font-bold text-gray-900">{activeReminders?.filter(r => r?.status === 'delivered')?.length}</p>
                    <p className="text-sm text-gray-600">Entregados</p>
                  </div>
                </div>
              </div>
              <div className="bg-white p-4 rounded-lg border border-gray-200">
                <div className="flex items-center space-x-3">
                  <div className="w-10 h-10 bg-error/10 text-error rounded-lg flex items-center justify-center">
                    <Icon name="AlertCircle" size={20} />
                  </div>
                  <div>
                    <p className="text-2xl font-bold text-gray-900">{activeReminders?.filter(r => r?.status === 'failed')?.length}</p>
                    <p className="text-sm text-gray-600">Fallidos</p>
                  </div>
                </div>
              </div>
              <div className="bg-white p-4 rounded-lg border border-gray-200">
                <div className="flex items-center space-x-3">
                  <div className="w-10 h-10 bg-secondary/10 text-secondary rounded-lg flex items-center justify-center">
                    <Icon name="FileText" size={20} />
                  </div>
                  <div>
                    <p className="text-2xl font-bold text-gray-900">{templates?.length}</p>
                    <p className="text-sm text-gray-600">Plantillas</p>
                  </div>
                </div>
              </div>
            </div>

            {/* Tab Navigation */}
            <div className="border-b border-gray-200">
              <nav className="flex space-x-8 overflow-x-auto">
                {tabs?.map((tab) => (
                  <button
                    key={tab?.id}
                    onClick={() => setActiveTab(tab?.id)}
                    className={`
                      flex items-center space-x-2 py-4 px-1 border-b-2 font-medium text-sm whitespace-nowrap
                      transition-colors duration-150
                      ${activeTab === tab?.id
                        ? 'border-primary text-primary' :'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                      }
                    `}
                  >
                    <Icon name={tab?.icon} size={16} />
                    <span>{tab?.label}</span>
                    {tab?.count !== null && (
                      <span className={`
                        px-2 py-0.5 text-xs rounded-full
                        ${activeTab === tab?.id
                          ? 'bg-primary/10 text-primary' :'bg-gray-100 text-gray-600'
                        }
                      `}>
                        {tab?.count}
                      </span>
                    )}
                  </button>
                ))}
              </nav>
            </div>
          </div>

          {/* Tab Content */}
          <div className="space-y-6">
            {renderTabContent()}
          </div>
        </div>
      </main>
      {/* Message Preview Modal */}
      <MessagePreviewModal
        isOpen={showPreviewModal}
        onClose={() => setShowPreviewModal(false)}
        template={previewData?.template}
        recipients={previewData?.recipients || []}
        scheduledDate={previewData?.scheduledDate}
        channel={previewData?.channel}
        onConfirmSend={handlePreviewConfirm}
      />
      <QuickActionFloatingButton />
    </div>
  );
};

export default RemindersSystem;