import React, { useState } from 'react';
import { LineChart, Line, BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, Legend, ResponsiveContainer, ComposedChart } from 'recharts';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';

const AdvancedAnalyticsSection = ({ analyticsData }) => {
  const [activeAnalysis, setActiveAnalysis] = useState('trends');
  const [timeframe, setTimeframe] = useState('12months');

  const analysisTypes = [
    { key: 'trends', label: 'Análisis de Tendencias', icon: 'TrendingUp' },
    { key: 'comparative', label: 'Comparativo Anual', icon: 'BarChart3' },
    { key: 'forecasting', label: 'Proyecciones', icon: 'Target' },
    { key: 'performance', label: 'Rendimiento por Propiedad', icon: 'Building2' }
  ];

  const timeframes = [
    { key: '6months', label: '6 Meses' },
    { key: '12months', label: '12 Meses' },
    { key: '24months', label: '2 Años' },
    { key: '36months', label: '3 Años' }
  ];

  const formatCurrency = (value) => {
    return new Intl.NumberFormat('es-CL', {
      style: 'currency',
      currency: 'CLP',
      minimumFractionDigits: 0
    })?.format(value);
  };

  const CustomTooltip = ({ active, payload, label }) => {
    if (active && payload && payload?.length) {
      return (
        <div className="bg-white p-4 border border-gray-200 rounded-lg shadow-lg">
          <p className="text-sm font-medium text-gray-900 mb-2">{label}</p>
          {payload?.map((entry, index) => (
            <div key={index} className="flex items-center space-x-2 text-sm">
              <div 
                className="w-3 h-3 rounded-full" 
                style={{ backgroundColor: entry?.color }}
              />
              <span className="text-gray-600">{entry?.name}:</span>
              <span className="font-medium" style={{ color: entry?.color }}>
                {entry?.name?.includes('Ingreso') || entry?.name?.includes('Costo') 
                  ? formatCurrency(entry?.value) 
                  : `${entry?.value}%`
                }
              </span>
            </div>
          ))}
        </div>
      );
    }
    return null;
  };

  const renderAnalysisChart = () => {
    const data = analyticsData?.[activeAnalysis]?.[timeframe] || [];
    
    switch (activeAnalysis) {
      case 'trends':
        return (
          <ResponsiveContainer width="100%" height={400}>
            <ComposedChart data={data}>
              <CartesianGrid strokeDasharray="3 3" stroke="#f0f0f0" />
              <XAxis dataKey="period" stroke="#6b7280" />
              <YAxis yAxisId="left" stroke="#6b7280" tickFormatter={formatCurrency} />
              <YAxis yAxisId="right" orientation="right" stroke="#6b7280" />
              <Tooltip content={<CustomTooltip />} />
              <Legend />
              <Bar yAxisId="left" dataKey="income" fill="#3a86ff" name="Ingresos" radius={[2, 2, 0, 0]} />
              <Line yAxisId="right" type="monotone" dataKey="growth" stroke="#38b000" strokeWidth={3} name="Crecimiento (%)" />
            </ComposedChart>
          </ResponsiveContainer>
        );
      
      case 'comparative':
        return (
          <ResponsiveContainer width="100%" height={400}>
            <BarChart data={data}>
              <CartesianGrid strokeDasharray="3 3" stroke="#f0f0f0" />
              <XAxis dataKey="month" stroke="#6b7280" />
              <YAxis stroke="#6b7280" tickFormatter={formatCurrency} />
              <Tooltip content={<CustomTooltip />} />
              <Legend />
              <Bar dataKey="currentYear" fill="#3a86ff" name="2025" radius={[2, 2, 0, 0]} />
              <Bar dataKey="previousYear" fill="#8338ec" name="2024" radius={[2, 2, 0, 0]} />
            </BarChart>
          </ResponsiveContainer>
        );
      
      case 'forecasting':
        return (
          <ResponsiveContainer width="100%" height={400}>
            <LineChart data={data}>
              <CartesianGrid strokeDasharray="3 3" stroke="#f0f0f0" />
              <XAxis dataKey="period" stroke="#6b7280" />
              <YAxis stroke="#6b7280" tickFormatter={formatCurrency} />
              <Tooltip content={<CustomTooltip />} />
              <Legend />
              <Line 
                type="monotone" 
                dataKey="actual" 
                stroke="#3a86ff" 
                strokeWidth={3}
                name="Datos Reales"
                dot={{ fill: '#3a86ff', strokeWidth: 2, r: 4 }}
              />
              <Line 
                type="monotone" 
                dataKey="forecast" 
                stroke="#ff9e00" 
                strokeWidth={3}
                strokeDasharray="5 5"
                name="Proyección"
                dot={{ fill: '#ff9e00', strokeWidth: 2, r: 4 }}
              />
              <Line 
                type="monotone" 
                dataKey="confidence" 
                stroke="#38b000" 
                strokeWidth={2}
                name="Intervalo de Confianza"
                dot={false}
              />
            </LineChart>
          </ResponsiveContainer>
        );
      
      case 'performance':
        return (
          <ResponsiveContainer width="100%" height={400}>
            <BarChart data={data} layout="horizontal">
              <CartesianGrid strokeDasharray="3 3" stroke="#f0f0f0" />
              <XAxis type="number" stroke="#6b7280" tickFormatter={formatCurrency} />
              <YAxis dataKey="property" type="category" stroke="#6b7280" width={120} />
              <Tooltip content={<CustomTooltip />} />
              <Legend />
              <Bar dataKey="revenue" fill="#3a86ff" name="Ingresos" radius={[0, 2, 2, 0]} />
              <Bar dataKey="costs" fill="#ff006e" name="Costos" radius={[0, 2, 2, 0]} />
            </BarChart>
          </ResponsiveContainer>
        );
      
      default:
        return null;
    }
  };

  const getInsights = () => {
    const insights = {
      trends: [
        "Los ingresos han mostrado un crecimiento constante del 8.5% en los últimos 6 meses",
        "La temporada alta (diciembre-febrero) genera 23% más ingresos que el promedio",
        "Se observa una correlación positiva entre ocupación y rentabilidad"
      ],
      comparative: [
        "Los ingresos de 2025 superan en 12.3% a los del mismo período de 2024",
        "Marzo y abril muestran el mayor crecimiento interanual (18% y 15%)",
        "La estabilidad de precios ha mejorado significativamente este año"
      ],
      forecasting: [
        "Se proyecta un crecimiento del 15% para el próximo trimestre",
        "La demanda estacional sugiere un pico en diciembre-enero",
        "El modelo predice una estabilización de precios en el segundo semestre"
      ],
      performance: [
        "Las Condes 1234 es la propiedad más rentable con $2.1M mensuales",
        "Ñuñoa 9012 tiene el mejor ratio costo-beneficio (ROI 18.5%)",
        "Se recomienda revisar los costos de mantenimiento en Santiago Centro"
      ]
    };

    return insights?.[activeAnalysis] || [];
  };

  return (
    <div className="glass-card p-6 rounded-lg mb-8">
      <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between mb-6">
        <div className="flex items-center space-x-3 mb-4 lg:mb-0">
          <div className="w-10 h-10 bg-accent/10 text-gray-700 rounded-lg flex items-center justify-center">
            <Icon name="Brain" size={20} />
          </div>
          <div>
            <h3 className="text-xl font-semibold text-gray-900">Análisis Avanzado</h3>
            <p className="text-sm text-gray-600">Inteligencia de negocio y proyecciones</p>
          </div>
        </div>
        
        <div className="flex flex-wrap gap-2">
          {timeframes?.map((tf) => (
            <button
              key={tf?.key}
              onClick={() => setTimeframe(tf?.key)}
              className={`
                px-3 py-1.5 rounded-md text-sm font-medium transition-all duration-150
                ${timeframe === tf?.key
                  ? 'bg-primary text-white' :'bg-gray-100 text-gray-700 hover:bg-gray-200'
                }
              `}
            >
              {tf?.label}
            </button>
          ))}
        </div>
      </div>
      <div className="flex flex-wrap gap-2 mb-6">
        {analysisTypes?.map((analysis) => (
          <button
            key={analysis?.key}
            onClick={() => setActiveAnalysis(analysis?.key)}
            className={`
              flex items-center space-x-2 px-4 py-2 rounded-lg text-sm font-medium
              transition-all duration-150 hover-lift
              ${activeAnalysis === analysis?.key
                ? 'bg-primary text-white shadow-sm'
                : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
              }
            `}
          >
            <Icon name={analysis?.icon} size={16} />
            <span>{analysis?.label}</span>
          </button>
        ))}
      </div>
      <div className="grid grid-cols-1 xl:grid-cols-4 gap-6">
        {/* Chart Area */}
        <div className="xl:col-span-3">
          <div className="bg-gray-50 rounded-lg p-4">
            {renderAnalysisChart()}
          </div>
        </div>

        {/* Insights Panel */}
        <div className="xl:col-span-1">
          <div className="bg-white border border-gray-200 rounded-lg p-4">
            <div className="flex items-center space-x-2 mb-4">
              <Icon name="Lightbulb" size={18} className="text-warning" />
              <h4 className="text-sm font-semibold text-gray-900">Insights Clave</h4>
            </div>
            
            <div className="space-y-3">
              {getInsights()?.map((insight, index) => (
                <div key={index} className="flex items-start space-x-2">
                  <div className="w-1.5 h-1.5 bg-primary rounded-full mt-2 flex-shrink-0" />
                  <p className="text-xs text-gray-600 leading-relaxed">{insight}</p>
                </div>
              ))}
            </div>

            <div className="mt-6 pt-4 border-t border-gray-100">
              <Button
                variant="outline"
                size="sm"
                iconName="Download"
                iconPosition="left"
                fullWidth
              >
                Exportar Análisis
              </Button>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default AdvancedAnalyticsSection;