import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';

const AutomatedReportScheduling = ({ onSaveSchedule }) => {
  const [schedules, setSchedules] = useState([
    {
      id: 1,
      name: 'Reporte Mensual de Ingresos',
      type: 'income_statement',
      frequency: 'monthly',
      dayOfMonth: 1,
      time: '09:00',
      recipients: ['maria.gonzalez@propmanager.cl', 'admin@propmanager.cl'],
      format: 'pdf',
      isActive: true,
      lastSent: '2025-08-01',
      nextSend: '2025-09-01'
    },
    {
      id: 2,
      name: 'Resumen Semanal de Comprobantes',
      type: 'voucher_summary',
      frequency: 'weekly',
      dayOfWeek: 'monday',
      time: '08:30',
      recipients: ['contabilidad@propmanager.cl'],
      format: 'excel',
      isActive: true,
      lastSent: '2025-08-19',
      nextSend: '2025-08-26'
    },
    {
      id: 3,
      name: 'Análisis Trimestral de Ocupación',
      type: 'occupancy_analysis',
      frequency: 'quarterly',
      dayOfMonth: 15,
      time: '10:00',
      recipients: ['gerencia@propmanager.cl', 'inversores@propmanager.cl'],
      format: 'pdf',
      isActive: false,
      lastSent: '2025-05-15',
      nextSend: '2025-11-15'
    }
  ]);

  const [showNewScheduleForm, setShowNewScheduleForm] = useState(false);
  const [newSchedule, setNewSchedule] = useState({
    name: '',
    type: '',
    frequency: 'monthly',
    dayOfMonth: 1,
    dayOfWeek: 'monday',
    time: '09:00',
    recipients: [],
    format: 'pdf',
    isActive: true
  });

  const reportTypes = [
    { value: 'income_statement', label: 'Estado de Ingresos' },
    { value: 'voucher_summary', label: 'Resumen de Comprobantes' },
    { value: 'tenant_payments', label: 'Pagos de Inquilinos' },
    { value: 'owner_commissions', label: 'Comisiones de Propietarios' },
    { value: 'occupancy_analysis', label: 'Análisis de Ocupación' },
    { value: 'financial_summary', label: 'Resumen Financiero' }
  ];

  const frequencyOptions = [
    { value: 'daily', label: 'Diario' },
    { value: 'weekly', label: 'Semanal' },
    { value: 'monthly', label: 'Mensual' },
    { value: 'quarterly', label: 'Trimestral' }
  ];

  const dayOfWeekOptions = [
    { value: 'monday', label: 'Lunes' },
    { value: 'tuesday', label: 'Martes' },
    { value: 'wednesday', label: 'Miércoles' },
    { value: 'thursday', label: 'Jueves' },
    { value: 'friday', label: 'Viernes' },
    { value: 'saturday', label: 'Sábado' },
    { value: 'sunday', label: 'Domingo' }
  ];

  const formatOptions = [
    { value: 'pdf', label: 'PDF' },
    { value: 'excel', label: 'Excel' },
    { value: 'both', label: 'PDF + Excel' }
  ];

  const toggleScheduleStatus = (scheduleId) => {
    setSchedules(prev => prev?.map(schedule => 
      schedule?.id === scheduleId 
        ? { ...schedule, isActive: !schedule?.isActive }
        : schedule
    ));
  };

  const deleteSchedule = (scheduleId) => {
    if (confirm('¿Estás seguro de que deseas eliminar esta programación?')) {
      setSchedules(prev => prev?.filter(schedule => schedule?.id !== scheduleId));
    }
  };

  const handleNewScheduleSubmit = (e) => {
    e?.preventDefault();
    
    if (!newSchedule?.name || !newSchedule?.type || newSchedule?.recipients?.length === 0) {
      alert('Por favor completa todos los campos requeridos');
      return;
    }

    const schedule = {
      ...newSchedule,
      id: Date.now(),
      lastSent: null,
      nextSend: calculateNextSend(newSchedule)
    };

    setSchedules(prev => [...prev, schedule]);
    setNewSchedule({
      name: '',
      type: '',
      frequency: 'monthly',
      dayOfMonth: 1,
      dayOfWeek: 'monday',
      time: '09:00',
      recipients: [],
      format: 'pdf',
      isActive: true
    });
    setShowNewScheduleForm(false);

    if (onSaveSchedule) {
      onSaveSchedule(schedule);
    }
  };

  const calculateNextSend = (schedule) => {
    const now = new Date();
    const nextDate = new Date();
    
    switch (schedule?.frequency) {
      case 'daily':
        nextDate?.setDate(now?.getDate() + 1);
        break;
      case 'weekly':
        const daysUntilTarget = (getDayNumber(schedule?.dayOfWeek) - now?.getDay() + 7) % 7;
        nextDate?.setDate(now?.getDate() + (daysUntilTarget || 7));
        break;
      case 'monthly':
        nextDate?.setMonth(now?.getMonth() + 1);
        nextDate?.setDate(schedule?.dayOfMonth);
        break;
      case 'quarterly':
        nextDate?.setMonth(now?.getMonth() + 3);
        nextDate?.setDate(schedule?.dayOfMonth);
        break;
    }
    
    return nextDate?.toISOString()?.split('T')?.[0];
  };

  const getDayNumber = (dayName) => {
    const days = { sunday: 0, monday: 1, tuesday: 2, wednesday: 3, thursday: 4, friday: 5, saturday: 6 };
    return days?.[dayName];
  };

  const formatDate = (dateString) => {
    if (!dateString) return 'Nunca';
    return new Date(dateString)?.toLocaleDateString('es-CL', {
      day: '2-digit',
      month: '2-digit',
      year: 'numeric'
    });
  };

  const getFrequencyText = (schedule) => {
    switch (schedule?.frequency) {
      case 'daily':
        return `Diario a las ${schedule?.time}`;
      case 'weekly':
        return `${dayOfWeekOptions?.find(d => d?.value === schedule?.dayOfWeek)?.label}s a las ${schedule?.time}`;
      case 'monthly':
        return `Día ${schedule?.dayOfMonth} de cada mes a las ${schedule?.time}`;
      case 'quarterly':
        return `Día ${schedule?.dayOfMonth} cada trimestre a las ${schedule?.time}`;
      default:
        return 'No configurado';
    }
  };

  return (
    <div className="glass-card p-6 rounded-lg">
      <div className="flex items-center justify-between mb-6">
        <div className="flex items-center space-x-3">
          <div className="w-10 h-10 bg-warning/10 text-warning rounded-lg flex items-center justify-center">
            <Icon name="Calendar" size={20} />
          </div>
          <div>
            <h3 className="text-xl font-semibold text-gray-900">Reportes Programados</h3>
            <p className="text-sm text-gray-600">Automatiza la entrega de reportes a stakeholders</p>
          </div>
        </div>
        
        <Button
          onClick={() => setShowNewScheduleForm(true)}
          iconName="Plus"
          iconPosition="left"
        >
          Nueva Programación
        </Button>
      </div>
      {/* New Schedule Form */}
      {showNewScheduleForm && (
        <div className="bg-gray-50 rounded-lg p-6 mb-6">
          <div className="flex items-center justify-between mb-4">
            <h4 className="text-lg font-semibold text-gray-900">Nueva Programación</h4>
            <button
              onClick={() => setShowNewScheduleForm(false)}
              className="p-2 rounded-md hover:bg-gray-200 transition-colors duration-150"
            >
              <Icon name="X" size={16} className="text-gray-500" />
            </button>
          </div>

          <form onSubmit={handleNewScheduleSubmit} className="space-y-4">
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-4">
              <Input
                label="Nombre de la Programación"
                placeholder="Ej: Reporte Mensual de Ingresos"
                value={newSchedule?.name}
                onChange={(e) => setNewSchedule(prev => ({ ...prev, name: e?.target?.value }))}
                required
              />

              <Select
                label="Tipo de Reporte"
                options={reportTypes}
                value={newSchedule?.type}
                onChange={(value) => setNewSchedule(prev => ({ ...prev, type: value }))}
                required
              />

              <Select
                label="Frecuencia"
                options={frequencyOptions}
                value={newSchedule?.frequency}
                onChange={(value) => setNewSchedule(prev => ({ ...prev, frequency: value }))}
              />

              {newSchedule?.frequency === 'weekly' && (
                <Select
                  label="Día de la Semana"
                  options={dayOfWeekOptions}
                  value={newSchedule?.dayOfWeek}
                  onChange={(value) => setNewSchedule(prev => ({ ...prev, dayOfWeek: value }))}
                />
              )}

              {(newSchedule?.frequency === 'monthly' || newSchedule?.frequency === 'quarterly') && (
                <Input
                  label="Día del Mes"
                  type="number"
                  min="1"
                  max="31"
                  value={newSchedule?.dayOfMonth}
                  onChange={(e) => setNewSchedule(prev => ({ ...prev, dayOfMonth: parseInt(e?.target?.value) }))}
                />
              )}

              <Input
                label="Hora de Envío"
                type="time"
                value={newSchedule?.time}
                onChange={(e) => setNewSchedule(prev => ({ ...prev, time: e?.target?.value }))}
              />

              <Select
                label="Formato"
                options={formatOptions}
                value={newSchedule?.format}
                onChange={(value) => setNewSchedule(prev => ({ ...prev, format: value }))}
              />
            </div>

            <div>
              <Input
                label="Destinatarios (separados por coma)"
                placeholder="email1@ejemplo.com, email2@ejemplo.com"
                value={newSchedule?.recipients?.join(', ')}
                onChange={(e) => setNewSchedule(prev => ({ 
                  ...prev, 
                  recipients: e?.target?.value?.split(',')?.map(email => email?.trim())?.filter(Boolean)
                }))}
                required
              />
            </div>

            <div className="flex justify-end space-x-3">
              <Button
                type="button"
                variant="outline"
                onClick={() => setShowNewScheduleForm(false)}
              >
                Cancelar
              </Button>
              <Button type="submit">
                Crear Programación
              </Button>
            </div>
          </form>
        </div>
      )}
      {/* Existing Schedules */}
      <div className="space-y-4">
        {schedules?.map((schedule) => (
          <div key={schedule?.id} className="bg-white border border-gray-200 rounded-lg p-5">
            <div className="flex items-start justify-between">
              <div className="flex-1">
                <div className="flex items-center space-x-3 mb-2">
                  <h4 className="text-lg font-semibold text-gray-900">{schedule?.name}</h4>
                  <span className={`
                    inline-flex items-center px-2 py-1 rounded-full text-xs font-medium
                    ${schedule?.isActive 
                      ? 'bg-success/10 text-success' :'bg-gray-100 text-gray-600'
                    }
                  `}>
                    {schedule?.isActive ? 'Activo' : 'Inactivo'}
                  </span>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 text-sm text-gray-600">
                  <div>
                    <span className="font-medium">Tipo:</span>
                    <div>{reportTypes?.find(t => t?.value === schedule?.type)?.label}</div>
                  </div>
                  <div>
                    <span className="font-medium">Frecuencia:</span>
                    <div>{getFrequencyText(schedule)}</div>
                  </div>
                  <div>
                    <span className="font-medium">Último Envío:</span>
                    <div>{formatDate(schedule?.lastSent)}</div>
                  </div>
                  <div>
                    <span className="font-medium">Próximo Envío:</span>
                    <div className="text-primary font-medium">{formatDate(schedule?.nextSend)}</div>
                  </div>
                </div>

                <div className="mt-3">
                  <span className="text-sm font-medium text-gray-700">Destinatarios:</span>
                  <div className="flex flex-wrap gap-2 mt-1">
                    {schedule?.recipients?.map((email, index) => (
                      <span key={index} className="inline-flex items-center px-2 py-1 bg-gray-100 text-gray-700 text-xs rounded-md">
                        {email}
                      </span>
                    ))}
                  </div>
                </div>
              </div>

              <div className="flex items-center space-x-2 ml-4">
                <button
                  onClick={() => toggleScheduleStatus(schedule?.id)}
                  className={`
                    p-2 rounded-md transition-colors duration-150
                    ${schedule?.isActive 
                      ? 'text-warning hover:bg-warning/10' :'text-success hover:bg-success/10'
                    }
                  `}
                  title={schedule?.isActive ? 'Pausar' : 'Activar'}
                >
                  <Icon name={schedule?.isActive ? 'Pause' : 'Play'} size={16} />
                </button>
                
                <button
                  onClick={() => deleteSchedule(schedule?.id)}
                  className="p-2 rounded-md text-error hover:bg-error/10 transition-colors duration-150"
                  title="Eliminar"
                >
                  <Icon name="Trash2" size={16} />
                </button>
              </div>
            </div>
          </div>
        ))}
      </div>
      {schedules?.length === 0 && (
        <div className="text-center py-12">
          <Icon name="Calendar" size={48} className="text-gray-300 mx-auto mb-4" />
          <h3 className="text-lg font-medium text-gray-900 mb-2">No hay reportes programados</h3>
          <p className="text-gray-600 mb-4">Crea tu primera programación para automatizar la entrega de reportes</p>
          <Button
            onClick={() => setShowNewScheduleForm(true)}
            iconName="Plus"
            iconPosition="left"
          >
            Crear Primera Programación
          </Button>
        </div>
      )}
    </div>
  );
};

export default AutomatedReportScheduling;