import React, { useState } from 'react';
import { LineChart, Line, BarChart, Bar, PieChart, Pie, Cell, XAxis, YAxis, CartesianGrid, Tooltip, Legend, ResponsiveContainer } from 'recharts';
import Icon from '../../../components/AppIcon';

const InteractiveChartsSection = ({ chartData }) => {
  const [activeChart, setActiveChart] = useState('income');

  const chartTypes = [
    { key: 'income', label: 'Ingresos Mensuales', icon: 'TrendingUp', type: 'line' },
    { key: 'occupancy', label: 'Tasa de Ocupación', icon: 'BarChart3', type: 'bar' },
    { key: 'payments', label: 'Estado de Pagos', icon: 'PieChart', type: 'pie' },
    { key: 'properties', label: 'Rendimiento por Propiedad', icon: 'Building2', type: 'bar' }
  ];

  const colors = ['#3a86ff', '#8338ec', '#38b000', '#ff9e00', '#ff006e'];

  const formatCurrency = (value) => {
    return new Intl.NumberFormat('es-CL', {
      style: 'currency',
      currency: 'CLP',
      minimumFractionDigits: 0
    })?.format(value);
  };

  const CustomTooltip = ({ active, payload, label }) => {
    if (active && payload && payload?.length) {
      return (
        <div className="bg-white p-3 border border-gray-200 rounded-lg shadow-lg">
          <p className="text-sm font-medium text-gray-900">{label}</p>
          {payload?.map((entry, index) => (
            <p key={index} className="text-sm" style={{ color: entry?.color }}>
              {entry?.name}: {entry?.name?.includes('Ingreso') ? formatCurrency(entry?.value) : `${entry?.value}%`}
            </p>
          ))}
        </div>
      );
    }
    return null;
  };

  const renderChart = () => {
    const data = chartData?.[activeChart];
    
    switch (chartTypes?.find(c => c?.key === activeChart)?.type) {
      case 'line':
        return (
          <ResponsiveContainer width="100%" height={400}>
            <LineChart data={data}>
              <CartesianGrid strokeDasharray="3 3" stroke="#f0f0f0" />
              <XAxis dataKey="month" stroke="#6b7280" />
              <YAxis stroke="#6b7280" tickFormatter={formatCurrency} />
              <Tooltip content={<CustomTooltip />} />
              <Legend />
              <Line 
                type="monotone" 
                dataKey="income" 
                stroke="#3a86ff" 
                strokeWidth={3}
                dot={{ fill: '#3a86ff', strokeWidth: 2, r: 6 }}
                activeDot={{ r: 8, stroke: '#3a86ff', strokeWidth: 2 }}
                name="Ingresos"
              />
            </LineChart>
          </ResponsiveContainer>
        );
      
      case 'bar':
        return (
          <ResponsiveContainer width="100%" height={400}>
            <BarChart data={data}>
              <CartesianGrid strokeDasharray="3 3" stroke="#f0f0f0" />
              <XAxis dataKey="name" stroke="#6b7280" />
              <YAxis stroke="#6b7280" />
              <Tooltip content={<CustomTooltip />} />
              <Legend />
              <Bar 
                dataKey="value" 
                fill="#3a86ff" 
                radius={[4, 4, 0, 0]}
                name={activeChart === 'occupancy' ? 'Ocupación (%)' : 'Rendimiento'}
              />
            </BarChart>
          </ResponsiveContainer>
        );
      
      case 'pie':
        return (
          <ResponsiveContainer width="100%" height={400}>
            <PieChart>
              <Pie
                data={data}
                cx="50%"
                cy="50%"
                labelLine={false}
                label={({ name, percent }) => `${name} ${(percent * 100)?.toFixed(0)}%`}
                outerRadius={120}
                fill="#8884d8"
                dataKey="value"
              >
                {data?.map((entry, index) => (
                  <Cell key={`cell-${index}`} fill={colors?.[index % colors?.length]} />
                ))}
              </Pie>
              <Tooltip />
            </PieChart>
          </ResponsiveContainer>
        );
      
      default:
        return null;
    }
  };

  return (
    <div className="glass-card p-6 rounded-lg mb-8">
      <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between mb-6">
        <h3 className="text-xl font-semibold text-gray-900 mb-4 lg:mb-0">
          Análisis Interactivo
        </h3>
        
        <div className="flex flex-wrap gap-2">
          {chartTypes?.map((chart) => (
            <button
              key={chart?.key}
              onClick={() => setActiveChart(chart?.key)}
              className={`
                flex items-center space-x-2 px-4 py-2 rounded-lg text-sm font-medium
                transition-all duration-150 hover-lift
                ${activeChart === chart?.key
                  ? 'bg-primary text-white shadow-sm'
                  : 'bg-gray-100 text-gray-700 hover:bg-gray-200'
                }
              `}
            >
              <Icon name={chart?.icon} size={16} />
              <span>{chart?.label}</span>
            </button>
          ))}
        </div>
      </div>
      <div className="bg-gray-50 rounded-lg p-4">
        {renderChart()}
      </div>
    </div>
  );
};

export default InteractiveChartsSection;