import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';

const PreConfiguredTemplates = ({ onGenerateTemplate }) => {
  const [generatingTemplate, setGeneratingTemplate] = useState(null);

  const templates = [
    {
      id: 'monthly_income',
      title: 'Estado de Ingresos Mensual',
      description: 'Reporte completo de ingresos del mes actual con comparativa del mes anterior',
      icon: 'DollarSign',
      bgColor: 'bg-success/10',
      iconColor: 'text-success',
      lastGenerated: '2025-08-25',
      frequency: 'Mensual',
      format: 'PDF + Excel'
    },
    {
      id: 'voucher_status',
      title: 'Resumen de Comprobantes',
      description: 'Estado actual de todos los comprobantes: generados, enviados, pagados y vencidos',
      icon: 'FileText',
      bgColor: 'bg-primary/10',
      iconColor: 'text-primary',
      lastGenerated: '2025-08-24',
      frequency: 'Semanal',
      format: 'PDF'
    },
    {
      id: 'tenant_payments',
      title: 'Historial de Pagos',
      description: 'Registro detallado de pagos de inquilinos con análisis de puntualidad',
      icon: 'CreditCard',
      bgColor: 'bg-secondary/10',
      iconColor: 'text-secondary',
      lastGenerated: '2025-08-23',
      frequency: 'Quincenal',
      format: 'Excel'
    },
    {
      id: 'owner_commissions',
      title: 'Comisiones de Propietarios',
      description: 'Cálculo automático de comisiones y distribución de ingresos por propietario',
      icon: 'UserCheck',
      bgColor: 'bg-warning/10',
      iconColor: 'text-warning',
      lastGenerated: '2025-08-22',
      frequency: 'Mensual',
      format: 'PDF + Excel'
    },
    {
      id: 'occupancy_analysis',
      title: 'Análisis de Ocupación',
      description: 'Métricas de ocupación por propiedad con tendencias y proyecciones',
      icon: 'BarChart3',
      bgColor: 'bg-accent/10',
      iconColor: 'text-gray-700',
      lastGenerated: '2025-08-21',
      frequency: 'Mensual',
      format: 'PDF'
    },
    {
      id: 'financial_summary',
      title: 'Resumen Financiero Completo',
      description: 'Dashboard ejecutivo con todos los KPIs y métricas de rendimiento',
      icon: 'TrendingUp',
      bgColor: 'bg-error/10',
      iconColor: 'text-error',
      lastGenerated: '2025-08-20',
      frequency: 'Mensual',
      format: 'PDF + Excel'
    }
  ];

  const handleGenerateTemplate = async (templateId) => {
    setGeneratingTemplate(templateId);
    
    try {
      // Simulate template generation
      await new Promise(resolve => setTimeout(resolve, 1500));
      
      if (onGenerateTemplate) {
        onGenerateTemplate(templateId);
      }
      
      const template = templates?.find(t => t?.id === templateId);
      alert(`Reporte "${template?.title}" generado exitosamente`);
    } catch (error) {
      alert('Error al generar el reporte. Inténtalo nuevamente.');
    } finally {
      setGeneratingTemplate(null);
    }
  };

  const formatDate = (dateString) => {
    return new Date(dateString)?.toLocaleDateString('es-CL', {
      day: '2-digit',
      month: '2-digit',
      year: 'numeric'
    });
  };

  return (
    <div className="glass-card p-6 rounded-lg mb-8">
      <div className="flex items-center justify-between mb-6">
        <div className="flex items-center space-x-3">
          <div className="w-10 h-10 bg-secondary/10 text-secondary rounded-lg flex items-center justify-center">
            <Icon name="Template" size={20} />
          </div>
          <div>
            <h3 className="text-xl font-semibold text-gray-900">Plantillas Preconfiguradas</h3>
            <p className="text-sm text-gray-600">Reportes listos para generar con un clic</p>
          </div>
        </div>
        
        <Button
          variant="outline"
          iconName="Settings"
          iconPosition="left"
          size="sm"
        >
          Configurar Plantillas
        </Button>
      </div>
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
        {templates?.map((template) => (
          <div key={template?.id} className="bg-white border border-gray-200 rounded-lg p-5 hover-lift transition-all duration-150">
            <div className="flex items-start justify-between mb-4">
              <div className={`w-12 h-12 rounded-lg flex items-center justify-center ${template?.bgColor}`}>
                <Icon name={template?.icon} size={24} className={template?.iconColor} />
              </div>
              <div className="text-right">
                <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-gray-100 text-gray-600">
                  {template?.frequency}
                </span>
              </div>
            </div>

            <div className="space-y-3 mb-4">
              <h4 className="text-lg font-semibold text-gray-900">{template?.title}</h4>
              <p className="text-sm text-gray-600 line-clamp-2">{template?.description}</p>
              
              <div className="flex items-center justify-between text-xs text-gray-500">
                <span>Último: {formatDate(template?.lastGenerated)}</span>
                <span>{template?.format}</span>
              </div>
            </div>

            <Button
              onClick={() => handleGenerateTemplate(template?.id)}
              loading={generatingTemplate === template?.id}
              iconName="Download"
              iconPosition="left"
              fullWidth
              size="sm"
            >
              {generatingTemplate === template?.id ? 'Generando...' : 'Generar Ahora'}
            </Button>
          </div>
        ))}
      </div>
    </div>
  );
};

export default PreConfiguredTemplates;