import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';

const ReportGenerationInterface = ({ onGenerateReport }) => {
  const [reportConfig, setReportConfig] = useState({
    type: '',
    dateRange: 'current_month',
    startDate: '',
    endDate: '',
    properties: [],
    format: 'pdf',
    includeCharts: true,
    includeDetails: true
  });

  const [isGenerating, setIsGenerating] = useState(false);

  const reportTypes = [
    { value: 'income_statement', label: 'Estado de Ingresos Mensual' },
    { value: 'voucher_summary', label: 'Resumen de Comprobantes' },
    { value: 'tenant_payments', label: 'Historial de Pagos de Inquilinos' },
    { value: 'owner_commissions', label: 'Comisiones de Propietarios' },
    { value: 'occupancy_report', label: 'Reporte de Ocupación' },
    { value: 'maintenance_costs', label: 'Costos de Mantenimiento' },
    { value: 'financial_summary', label: 'Resumen Financiero Completo' }
  ];

  const dateRangeOptions = [
    { value: 'current_month', label: 'Mes Actual' },
    { value: 'last_month', label: 'Mes Anterior' },
    { value: 'current_quarter', label: 'Trimestre Actual' },
    { value: 'last_quarter', label: 'Trimestre Anterior' },
    { value: 'current_year', label: 'Año Actual' },
    { value: 'last_year', label: 'Año Anterior' },
    { value: 'custom', label: 'Rango Personalizado' }
  ];

  const formatOptions = [
    { value: 'pdf', label: 'PDF' },
    { value: 'excel', label: 'Excel (XLSX)' },
    { value: 'csv', label: 'CSV' }
  ];

  const propertyOptions = [
    { value: 'all', label: 'Todas las Propiedades' },
    { value: 'prop_001', label: 'Las Condes 1234 - Depto 501' },
    { value: 'prop_002', label: 'Providencia 5678 - Casa' },
    { value: 'prop_003', label: 'Ñuñoa 9012 - Depto 203' },
    { value: 'prop_004', label: 'Santiago Centro 3456 - Oficina' }
  ];

  const handleInputChange = (field, value) => {
    setReportConfig(prev => ({
      ...prev,
      [field]: value
    }));
  };

  const handleGenerateReport = async () => {
    if (!reportConfig?.type) {
      alert('Por favor selecciona un tipo de reporte');
      return;
    }

    setIsGenerating(true);
    
    try {
      // Simulate report generation
      await new Promise(resolve => setTimeout(resolve, 2000));
      
      if (onGenerateReport) {
        onGenerateReport(reportConfig);
      }
      
      // Show success message
      alert(`Reporte ${reportTypes?.find(t => t?.value === reportConfig?.type)?.label} generado exitosamente`);
    } catch (error) {
      alert('Error al generar el reporte. Inténtalo nuevamente.');
    } finally {
      setIsGenerating(false);
    }
  };

  return (
    <div className="glass-card p-6 rounded-lg mb-8">
      <div className="flex items-center space-x-3 mb-6">
        <div className="w-10 h-10 bg-primary/10 text-primary rounded-lg flex items-center justify-center">
          <Icon name="FileText" size={20} />
        </div>
        <div>
          <h3 className="text-xl font-semibold text-gray-900">Generador de Reportes</h3>
          <p className="text-sm text-gray-600">Crea reportes personalizados con filtros avanzados</p>
        </div>
      </div>
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* Left Column */}
        <div className="space-y-4">
          <Select
            label="Tipo de Reporte"
            placeholder="Selecciona el tipo de reporte"
            options={reportTypes}
            value={reportConfig?.type}
            onChange={(value) => handleInputChange('type', value)}
            required
          />

          <Select
            label="Rango de Fechas"
            options={dateRangeOptions}
            value={reportConfig?.dateRange}
            onChange={(value) => handleInputChange('dateRange', value)}
          />

          {reportConfig?.dateRange === 'custom' && (
            <div className="grid grid-cols-2 gap-4">
              <Input
                label="Fecha Inicio"
                type="date"
                value={reportConfig?.startDate}
                onChange={(e) => handleInputChange('startDate', e?.target?.value)}
              />
              <Input
                label="Fecha Fin"
                type="date"
                value={reportConfig?.endDate}
                onChange={(e) => handleInputChange('endDate', e?.target?.value)}
              />
            </div>
          )}

          <Select
            label="Propiedades"
            placeholder="Selecciona las propiedades"
            options={propertyOptions}
            value={reportConfig?.properties}
            onChange={(value) => handleInputChange('properties', value)}
            multiple
            searchable
          />
        </div>

        {/* Right Column */}
        <div className="space-y-4">
          <Select
            label="Formato de Salida"
            options={formatOptions}
            value={reportConfig?.format}
            onChange={(value) => handleInputChange('format', value)}
          />

          <div className="space-y-3">
            <label className="text-sm font-medium text-gray-700">Opciones Adicionales</label>
            
            <div className="space-y-2">
              <label className="flex items-center space-x-3">
                <input
                  type="checkbox"
                  checked={reportConfig?.includeCharts}
                  onChange={(e) => handleInputChange('includeCharts', e?.target?.checked)}
                  className="w-4 h-4 text-primary border-gray-300 rounded focus:ring-primary"
                />
                <span className="text-sm text-gray-700">Incluir gráficos y visualizaciones</span>
              </label>
              
              <label className="flex items-center space-x-3">
                <input
                  type="checkbox"
                  checked={reportConfig?.includeDetails}
                  onChange={(e) => handleInputChange('includeDetails', e?.target?.checked)}
                  className="w-4 h-4 text-primary border-gray-300 rounded focus:ring-primary"
                />
                <span className="text-sm text-gray-700">Incluir detalles transaccionales</span>
              </label>
            </div>
          </div>

          <div className="pt-4">
            <Button
              onClick={handleGenerateReport}
              loading={isGenerating}
              iconName="Download"
              iconPosition="left"
              fullWidth
              disabled={!reportConfig?.type}
            >
              {isGenerating ? 'Generando Reporte...' : 'Generar Reporte'}
            </Button>
          </div>
        </div>
      </div>
    </div>
  );
};

export default ReportGenerationInterface;