import React, { useState, useEffect } from 'react';
import Header from '../../components/ui/Header';
import Sidebar from '../../components/ui/Sidebar';
import NavigationBreadcrumbs from '../../components/ui/NavigationBreadcrumbs';
import QuickActionFloatingButton from '../../components/ui/QuickActionFloatingButton';
import ReportCategoryFilter from './components/ReportCategoryFilter';
import KPIMetricsGrid from './components/KPIMetricsGrid';
import InteractiveChartsSection from './components/InteractiveChartsSection';
import ReportGenerationInterface from './components/ReportGenerationInterface';
import PreConfiguredTemplates from './components/PreConfiguredTemplates';
import AdvancedAnalyticsSection from './components/AdvancedAnalyticsSection';
import AutomatedReportScheduling from './components/AutomatedReportScheduling';

const ReportsAnalytics = () => {
  const [isSidebarOpen, setIsSidebarOpen] = useState(false);
  const [selectedCategory, setSelectedCategory] = useState('overview');
  const [isLoading, setIsLoading] = useState(true);

  useEffect(() => {
    // Simulate data loading
    const timer = setTimeout(() => {
      setIsLoading(false);
    }, 1000);

    return () => clearTimeout(timer);
  }, []);

  // Mock data for report categories
  const reportCategories = [
    {
      key: 'overview',
      label: 'Resumen General',
      icon: 'LayoutDashboard',
      count: 12,
      badge: 'Nuevo'
    },
    {
      key: 'financial',
      label: 'Reportes Financieros',
      icon: 'DollarSign',
      count: 8
    },
    {
      key: 'properties',
      label: 'Análisis de Propiedades',
      icon: 'Building2',
      count: 15
    },
    {
      key: 'tenants',
      label: 'Gestión de Inquilinos',
      icon: 'Users',
      count: 6
    },
    {
      key: 'owners',
      label: 'Comisiones y Propietarios',
      icon: 'UserCheck',
      count: 4
    },
    {
      key: 'operational',
      label: 'Reportes Operacionales',
      icon: 'Settings',
      count: 9
    }
  ];

  // Mock KPI metrics data
  const kpiMetrics = [
    {
      key: 'total_income',
      label: 'Ingresos Totales',
      value: 45750000,
      type: 'currency',
      change: 12.5,
      period: 'vs mes anterior',
      icon: 'DollarSign',
      bgColor: 'bg-success/10',
      iconColor: 'text-success'
    },
    {
      key: 'collection_rate',
      label: 'Tasa de Cobranza',
      value: '94.2%',
      type: 'percentage',
      change: 3.1,
      period: 'vs mes anterior',
      icon: 'TrendingUp',
      bgColor: 'bg-primary/10',
      iconColor: 'text-primary'
    },
    {
      key: 'occupancy_rate',
      label: 'Tasa de Ocupación',
      value: '87.5%',
      type: 'percentage',
      change: -2.3,
      period: 'vs mes anterior',
      icon: 'Home',
      bgColor: 'bg-warning/10',
      iconColor: 'text-warning'
    },
    {
      key: 'avg_rent',
      label: 'Renta Promedio',
      value: 850000,
      type: 'currency',
      change: 5.8,
      period: 'vs mes anterior',
      icon: 'BarChart3',
      bgColor: 'bg-secondary/10',
      iconColor: 'text-secondary'
    }
  ];

  // Mock chart data
  const chartData = {
    income: [
      { month: 'Ene', income: 38500000 },
      { month: 'Feb', income: 41200000 },
      { month: 'Mar', income: 39800000 },
      { month: 'Abr', income: 43100000 },
      { month: 'May', income: 44600000 },
      { month: 'Jun', income: 42300000 },
      { month: 'Jul', income: 45200000 },
      { month: 'Ago', income: 45750000 }
    ],
    occupancy: [
      { name: 'Las Condes', value: 92 },
      { name: 'Providencia', value: 88 },
      { name: 'Ñuñoa', value: 85 },
      { name: 'Santiago Centro', value: 79 },
      { name: 'Vitacura', value: 94 }
    ],
    payments: [
      { name: 'Pagados', value: 156 },
      { name: 'Pendientes', value: 23 },
      { name: 'Vencidos', value: 8 },
      { name: 'En Proceso', value: 12 }
    ],
    properties: [
      { name: 'Las Condes 1234', value: 2100000 },
      { name: 'Providencia 5678', value: 1850000 },
      { name: 'Ñuñoa 9012', value: 1200000 },
      { name: 'Santiago Centro 3456', value: 950000 },
      { name: 'Vitacura 7890', value: 2350000 }
    ]
  };

  // Mock advanced analytics data
  const analyticsData = {
    trends: {
      '12months': [
        { period: 'Sep 2024', income: 38200000, growth: 5.2 },
        { period: 'Oct 2024', income: 39100000, growth: 6.8 },
        { period: 'Nov 2024', income: 40500000, growth: 8.1 },
        { period: 'Dic 2024', income: 42800000, growth: 9.5 },
        { period: 'Ene 2025', income: 38500000, growth: 7.2 },
        { period: 'Feb 2025', income: 41200000, growth: 8.9 },
        { period: 'Mar 2025', income: 39800000, growth: 6.5 },
        { period: 'Abr 2025', income: 43100000, growth: 10.2 },
        { period: 'May 2025', income: 44600000, growth: 11.8 },
        { period: 'Jun 2025', income: 42300000, growth: 9.1 },
        { period: 'Jul 2025', income: 45200000, growth: 12.3 },
        { period: 'Ago 2025', income: 45750000, growth: 12.5 }
      ]
    },
    comparative: {
      '12months': [
        { month: 'Ene', currentYear: 38500000, previousYear: 34200000 },
        { month: 'Feb', currentYear: 41200000, previousYear: 36800000 },
        { month: 'Mar', currentYear: 39800000, previousYear: 35500000 },
        { month: 'Abr', currentYear: 43100000, previousYear: 38200000 },
        { month: 'May', currentYear: 44600000, previousYear: 39800000 },
        { month: 'Jun', currentYear: 42300000, previousYear: 37900000 },
        { month: 'Jul', currentYear: 45200000, previousYear: 40100000 },
        { month: 'Ago', currentYear: 45750000, previousYear: 40600000 }
      ]
    },
    forecasting: {
      '12months': [
        { period: 'Jun 2025', actual: 42300000, forecast: null, confidence: null },
        { period: 'Jul 2025', actual: 45200000, forecast: null, confidence: null },
        { period: 'Ago 2025', actual: 45750000, forecast: null, confidence: null },
        { period: 'Sep 2025', actual: null, forecast: 46800000, confidence: 44200000 },
        { period: 'Oct 2025', actual: null, forecast: 48200000, confidence: 45600000 },
        { period: 'Nov 2025', actual: null, forecast: 49100000, confidence: 46400000 },
        { period: 'Dic 2025', actual: null, forecast: 51500000, confidence: 48700000 }
      ]
    },
    performance: {
      '12months': [
        { property: 'Las Condes 1234', revenue: 2100000, costs: 320000 },
        { property: 'Vitacura 7890', revenue: 2350000, costs: 380000 },
        { property: 'Providencia 5678', revenue: 1850000, costs: 290000 },
        { property: 'Ñuñoa 9012', revenue: 1200000, costs: 180000 },
        { property: 'Santiago Centro 3456', revenue: 950000, costs: 160000 }
      ]
    }
  };

  const handleMenuToggle = () => {
    setIsSidebarOpen(!isSidebarOpen);
  };

  const handleSidebarClose = () => {
    setIsSidebarOpen(false);
  };

  const handleGenerateReport = (reportConfig) => {
    console.log('Generating report with config:', reportConfig);
    // Here you would typically make an API call to generate the report
  };

  const handleGenerateTemplate = (templateId) => {
    console.log('Generating template:', templateId);
    // Here you would typically make an API call to generate the template
  };

  const handleSaveSchedule = (schedule) => {
    console.log('Saving schedule:', schedule);
    // Here you would typically make an API call to save the schedule
  };

  if (isLoading) {
    return (
      <div className="min-h-screen bg-background">
        <Header onMenuToggle={handleMenuToggle} isMenuOpen={isSidebarOpen} />
        <Sidebar isOpen={isSidebarOpen} onClose={handleSidebarClose} />
        
        <main className="lg:ml-60 pt-16">
          <div className="p-6">
            <div className="flex items-center justify-center h-96">
              <div className="text-center">
                <div className="w-12 h-12 border-4 border-primary border-t-transparent rounded-full animate-spin mx-auto mb-4"></div>
                <p className="text-gray-600">Cargando datos de reportes...</p>
              </div>
            </div>
          </div>
        </main>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-background">
      <Header onMenuToggle={handleMenuToggle} isMenuOpen={isSidebarOpen} />
      <Sidebar isOpen={isSidebarOpen} onClose={handleSidebarClose} />
      
      <main className="lg:ml-60 pt-16">
        <div className="flex h-[calc(100vh-4rem)]">
          {/* Report Category Sidebar */}
          <ReportCategoryFilter
            selectedCategory={selectedCategory}
            onCategoryChange={setSelectedCategory}
            categories={reportCategories}
          />
          
          {/* Main Content Area */}
          <div className="flex-1 overflow-y-auto">
            <div className="p-6">
              <NavigationBreadcrumbs />
              
              <div className="mb-8">
                <h1 className="text-3xl font-bold text-gray-900 mb-2">
                  Reportes y Análisis
                </h1>
                <p className="text-gray-600">
                  Inteligencia de negocio y exportación de datos para toma de decisiones informadas
                </p>
              </div>

              {/* KPI Metrics */}
              <KPIMetricsGrid metrics={kpiMetrics} />

              {/* Interactive Charts */}
              <InteractiveChartsSection chartData={chartData} />

              {/* Report Generation Interface */}
              <ReportGenerationInterface onGenerateReport={handleGenerateReport} />

              {/* Pre-configured Templates */}
              <PreConfiguredTemplates onGenerateTemplate={handleGenerateTemplate} />

              {/* Advanced Analytics */}
              <AdvancedAnalyticsSection analyticsData={analyticsData} />

              {/* Automated Report Scheduling */}
              <AutomatedReportScheduling onSaveSchedule={handleSaveSchedule} />
            </div>
          </div>
        </div>
      </main>

      <QuickActionFloatingButton />
    </div>
  );
};

export default ReportsAnalytics;