import React, { useState } from 'react';

import Button from '../../../components/ui/Button';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';

const AddTenantModal = ({ isOpen, onClose, onSave, properties = [] }) => {
  const [formData, setFormData] = useState({
    name: '',
    rut: '',
    email: '',
    phone: '',
    emergencyContact: '',
    emergencyPhone: '',
    propertyId: '',
    leaseStart: '',
    leaseEnd: '',
    monthlyRent: '',
    deposit: '',
    notes: ''
  });

  const [errors, setErrors] = useState({});
  const [isSubmitting, setIsSubmitting] = useState(false);

  const propertyOptions = properties?.map(property => ({
    value: property?.id,
    label: `${property?.name} - ${property?.address}`
  }));

  const handleInputChange = (field, value) => {
    setFormData(prev => ({ ...prev, [field]: value }));
    if (errors?.[field]) {
      setErrors(prev => ({ ...prev, [field]: '' }));
    }
  };

  const validateForm = () => {
    const newErrors = {};

    if (!formData?.name?.trim()) newErrors.name = 'El nombre es obligatorio';
    if (!formData?.rut?.trim()) newErrors.rut = 'El RUT es obligatorio';
    if (!formData?.email?.trim()) newErrors.email = 'El email es obligatorio';
    if (!formData?.phone?.trim()) newErrors.phone = 'El teléfono es obligatorio';
    if (!formData?.propertyId) newErrors.propertyId = 'Debe seleccionar una propiedad';
    if (!formData?.leaseStart) newErrors.leaseStart = 'La fecha de inicio es obligatoria';
    if (!formData?.leaseEnd) newErrors.leaseEnd = 'La fecha de término es obligatoria';
    if (!formData?.monthlyRent) newErrors.monthlyRent = 'El arriendo mensual es obligatorio';

    // Email validation
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (formData?.email && !emailRegex?.test(formData?.email)) {
      newErrors.email = 'Formato de email inválido';
    }

    // Phone validation (Chilean format)
    const phoneRegex = /^(\+56)?[0-9]{8,9}$/;
    if (formData?.phone && !phoneRegex?.test(formData?.phone?.replace(/\s/g, ''))) {
      newErrors.phone = 'Formato de teléfono inválido';
    }

    // Date validation
    if (formData?.leaseStart && formData?.leaseEnd) {
      const startDate = new Date(formData.leaseStart);
      const endDate = new Date(formData.leaseEnd);
      if (endDate <= startDate) {
        newErrors.leaseEnd = 'La fecha de término debe ser posterior al inicio';
      }
    }

    setErrors(newErrors);
    return Object.keys(newErrors)?.length === 0;
  };

  const handleSubmit = async (e) => {
    e?.preventDefault();
    
    if (!validateForm()) return;

    setIsSubmitting(true);
    
    try {
      const tenantData = {
        ...formData,
        id: Date.now()?.toString(),
        createdAt: new Date()?.toISOString(),
        paymentStatus: 'pending',
        leaseStatus: 'active'
      };
      
      await onSave(tenantData);
      handleClose();
    } catch (error) {
      console.error('Error saving tenant:', error);
    } finally {
      setIsSubmitting(false);
    }
  };

  const handleClose = () => {
    setFormData({
      name: '',
      rut: '',
      email: '',
      phone: '',
      emergencyContact: '',
      emergencyPhone: '',
      propertyId: '',
      leaseStart: '',
      leaseEnd: '',
      monthlyRent: '',
      deposit: '',
      notes: ''
    });
    setErrors({});
    onClose();
  };

  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 z-1000 flex items-center justify-center p-4">
      <div className="bg-white rounded-lg shadow-xl w-full max-w-2xl max-h-[90vh] overflow-hidden">
        {/* Header */}
        <div className="flex items-center justify-between p-6 border-b border-gray-200">
          <h2 className="text-xl font-semibold text-gray-900">Agregar Nuevo Inquilino</h2>
          <Button
            variant="ghost"
            size="sm"
            iconName="X"
            onClick={handleClose}
            className="text-gray-500 hover:text-gray-700"
          />
        </div>

        {/* Form */}
        <form onSubmit={handleSubmit} className="overflow-y-auto max-h-[calc(90vh-140px)]">
          <div className="p-6 space-y-6">
            {/* Personal Information */}
            <div>
              <h3 className="text-lg font-medium text-gray-900 mb-4">Información Personal</h3>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <Input
                  label="Nombre Completo"
                  type="text"
                  placeholder="Ej: Juan Pérez González"
                  value={formData?.name}
                  onChange={(e) => handleInputChange('name', e?.target?.value)}
                  error={errors?.name}
                  required
                />
                
                <Input
                  label="RUT"
                  type="text"
                  placeholder="Ej: 12.345.678-9"
                  value={formData?.rut}
                  onChange={(e) => handleInputChange('rut', e?.target?.value)}
                  error={errors?.rut}
                  required
                />
                
                <Input
                  label="Email"
                  type="email"
                  placeholder="Ej: juan.perez@email.com"
                  value={formData?.email}
                  onChange={(e) => handleInputChange('email', e?.target?.value)}
                  error={errors?.email}
                  required
                />
                
                <Input
                  label="Teléfono"
                  type="tel"
                  placeholder="Ej: +56 9 1234 5678"
                  value={formData?.phone}
                  onChange={(e) => handleInputChange('phone', e?.target?.value)}
                  error={errors?.phone}
                  required
                />
              </div>
            </div>

            {/* Emergency Contact */}
            <div>
              <h3 className="text-lg font-medium text-gray-900 mb-4">Contacto de Emergencia</h3>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <Input
                  label="Nombre Contacto"
                  type="text"
                  placeholder="Ej: María Pérez"
                  value={formData?.emergencyContact}
                  onChange={(e) => handleInputChange('emergencyContact', e?.target?.value)}
                  error={errors?.emergencyContact}
                />
                
                <Input
                  label="Teléfono Contacto"
                  type="tel"
                  placeholder="Ej: +56 9 8765 4321"
                  value={formData?.emergencyPhone}
                  onChange={(e) => handleInputChange('emergencyPhone', e?.target?.value)}
                  error={errors?.emergencyPhone}
                />
              </div>
            </div>

            {/* Property and Lease Information */}
            <div>
              <h3 className="text-lg font-medium text-gray-900 mb-4">Información de Arriendo</h3>
              <div className="space-y-4">
                <Select
                  label="Propiedad"
                  options={propertyOptions}
                  value={formData?.propertyId}
                  onChange={(value) => handleInputChange('propertyId', value)}
                  placeholder="Seleccionar propiedad"
                  error={errors?.propertyId}
                  required
                  searchable
                />
                
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <Input
                    label="Fecha Inicio Contrato"
                    type="date"
                    value={formData?.leaseStart}
                    onChange={(e) => handleInputChange('leaseStart', e?.target?.value)}
                    error={errors?.leaseStart}
                    required
                  />
                  
                  <Input
                    label="Fecha Término Contrato"
                    type="date"
                    value={formData?.leaseEnd}
                    onChange={(e) => handleInputChange('leaseEnd', e?.target?.value)}
                    error={errors?.leaseEnd}
                    required
                  />
                </div>
                
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <Input
                    label="Arriendo Mensual (CLP)"
                    type="number"
                    placeholder="Ej: 450000"
                    value={formData?.monthlyRent}
                    onChange={(e) => handleInputChange('monthlyRent', e?.target?.value)}
                    error={errors?.monthlyRent}
                    required
                  />
                  
                  <Input
                    label="Garantía (CLP)"
                    type="number"
                    placeholder="Ej: 450000"
                    value={formData?.deposit}
                    onChange={(e) => handleInputChange('deposit', e?.target?.value)}
                    error={errors?.deposit}
                  />
                </div>
              </div>
            </div>

            {/* Additional Notes */}
            <div>
              <h3 className="text-lg font-medium text-gray-900 mb-4">Notas Adicionales</h3>
              <div className="space-y-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Observaciones
                  </label>
                  <textarea
                    rows={3}
                    placeholder="Información adicional sobre el inquilino..."
                    value={formData?.notes}
                    onChange={(e) => handleInputChange('notes', e?.target?.value)}
                    className="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-primary focus:border-primary resize-none"
                  />
                </div>
              </div>
            </div>
          </div>

          {/* Footer */}
          <div className="flex items-center justify-end space-x-3 p-6 border-t border-gray-200 bg-gray-50">
            <Button
              type="button"
              variant="outline"
              onClick={handleClose}
              disabled={isSubmitting}
            >
              Cancelar
            </Button>
            <Button
              type="submit"
              variant="default"
              loading={isSubmitting}
              iconName="Save"
              iconPosition="left"
            >
              Guardar Inquilino
            </Button>
          </div>
        </form>
      </div>
    </div>
  );
};

export default AddTenantModal;