import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Image from '../../../components/AppImage';
import Button from '../../../components/ui/Button';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';
import StatusIndicatorBadges from '../../../components/ui/StatusIndicatorBadges';

const PaymentHistoryModal = ({ isOpen, onClose, tenant, onRecordPayment }) => {
  const [activeTab, setActiveTab] = useState('history');
  const [showAddPayment, setShowAddPayment] = useState(false);
  const [paymentData, setPaymentData] = useState({
    amount: '',
    date: new Date()?.toISOString()?.split('T')?.[0],
    method: 'transfer',
    reference: '',
    notes: ''
  });

  // Mock payment history data
  const paymentHistory = [
    {
      id: 1,
      date: '2024-08-01',
      amount: 450000,
      method: 'Transferencia',
      reference: 'TRF-240801-001',
      status: 'paid',
      notes: 'Pago mensual agosto 2024'
    },
    {
      id: 2,
      date: '2024-07-01',
      amount: 450000,
      method: 'Transferencia',
      reference: 'TRF-240701-001',
      status: 'paid',
      notes: 'Pago mensual julio 2024'
    },
    {
      id: 3,
      date: '2024-06-01',
      amount: 450000,
      method: 'Efectivo',
      reference: 'EFE-240601-001',
      status: 'paid',
      notes: 'Pago mensual junio 2024'
    },
    {
      id: 4,
      date: '2024-09-01',
      amount: 450000,
      method: 'Pendiente',
      reference: 'PEN-240901-001',
      status: 'pending',
      notes: 'Pago mensual septiembre 2024'
    }
  ];

  const paymentMethods = [
    { value: 'transfer', label: 'Transferencia Bancaria' },
    { value: 'cash', label: 'Efectivo' },
    { value: 'check', label: 'Cheque' },
    { value: 'deposit', label: 'Depósito Bancario' }
  ];

  const handleAddPayment = (e) => {
    e?.preventDefault();
    
    const newPayment = {
      id: Date.now(),
      date: paymentData?.date,
      amount: parseInt(paymentData?.amount),
      method: paymentMethods?.find(m => m?.value === paymentData?.method)?.label,
      reference: paymentData?.reference,
      status: 'paid',
      notes: paymentData?.notes
    };

    onRecordPayment(tenant?.id, newPayment);
    
    // Reset form
    setPaymentData({
      amount: '',
      date: new Date()?.toISOString()?.split('T')?.[0],
      method: 'transfer',
      reference: '',
      notes: ''
    });
    
    setShowAddPayment(false);
  };

  const formatCurrency = (amount) => {
    return new Intl.NumberFormat('es-CL', {
      style: 'currency',
      currency: 'CLP',
      minimumFractionDigits: 0
    })?.format(amount);
  };

  const formatDate = (dateString) => {
    return new Date(dateString)?.toLocaleDateString('es-CL', {
      day: '2-digit',
      month: '2-digit',
      year: 'numeric'
    });
  };

  const calculateTotalPaid = () => {
    return paymentHistory?.filter(payment => payment?.status === 'paid')?.reduce((total, payment) => total + payment?.amount, 0);
  };

  const calculatePendingAmount = () => {
    return paymentHistory?.filter(payment => payment?.status === 'pending')?.reduce((total, payment) => total + payment?.amount, 0);
  };

  if (!isOpen || !tenant) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 z-1000 flex items-center justify-center p-4">
      <div className="bg-white rounded-lg shadow-xl w-full max-w-4xl max-h-[90vh] overflow-hidden">
        {/* Header */}
        <div className="flex items-center justify-between p-6 border-b border-gray-200">
          <div className="flex items-center space-x-3">
            <div className="w-10 h-10 bg-gray-200 rounded-full overflow-hidden">
              {tenant?.avatar ? (
                <Image
                  src={tenant?.avatar}
                  alt={tenant?.name}
                  className="w-full h-full object-cover"
                />
              ) : (
                <div className="w-full h-full bg-primary/10 flex items-center justify-center">
                  <Icon name="User" size={20} className="text-primary" />
                </div>
              )}
            </div>
            <div>
              <h2 className="text-xl font-semibold text-gray-900">Historial de Pagos</h2>
              <p className="text-sm text-gray-500">{tenant?.name} - {tenant?.property}</p>
            </div>
          </div>
          <Button
            variant="ghost"
            size="sm"
            iconName="X"
            onClick={onClose}
            className="text-gray-500 hover:text-gray-700"
          />
        </div>

        {/* Summary Cards */}
        <div className="p-6 bg-gray-50 border-b border-gray-200">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div className="bg-white p-4 rounded-lg border border-gray-200">
              <div className="flex items-center space-x-3">
                <div className="w-10 h-10 bg-success/10 rounded-lg flex items-center justify-center">
                  <Icon name="CheckCircle" size={20} className="text-success" />
                </div>
                <div>
                  <p className="text-sm text-gray-600">Total Pagado</p>
                  <p className="text-lg font-semibold text-gray-900">{formatCurrency(calculateTotalPaid())}</p>
                </div>
              </div>
            </div>
            
            <div className="bg-white p-4 rounded-lg border border-gray-200">
              <div className="flex items-center space-x-3">
                <div className="w-10 h-10 bg-warning/10 rounded-lg flex items-center justify-center">
                  <Icon name="Clock" size={20} className="text-warning" />
                </div>
                <div>
                  <p className="text-sm text-gray-600">Monto Pendiente</p>
                  <p className="text-lg font-semibold text-gray-900">{formatCurrency(calculatePendingAmount())}</p>
                </div>
              </div>
            </div>
            
            <div className="bg-white p-4 rounded-lg border border-gray-200">
              <div className="flex items-center space-x-3">
                <div className="w-10 h-10 bg-primary/10 rounded-lg flex items-center justify-center">
                  <Icon name="Calendar" size={20} className="text-primary" />
                </div>
                <div>
                  <p className="text-sm text-gray-600">Último Pago</p>
                  <p className="text-lg font-semibold text-gray-900">
                    {formatDate(paymentHistory?.find(p => p?.status === 'paid')?.date || new Date())}
                  </p>
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* Tabs */}
        <div className="border-b border-gray-200">
          <nav className="flex space-x-8 px-6">
            <button
              onClick={() => setActiveTab('history')}
              className={`py-4 px-1 border-b-2 font-medium text-sm transition-colors duration-150 ${
                activeTab === 'history' ?'border-primary text-primary' :'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
              }`}
            >
              Historial de Pagos
            </button>
            <button
              onClick={() => setActiveTab('summary')}
              className={`py-4 px-1 border-b-2 font-medium text-sm transition-colors duration-150 ${
                activeTab === 'summary' ?'border-primary text-primary' :'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
              }`}
            >
              Resumen Anual
            </button>
          </nav>
        </div>

        {/* Content */}
        <div className="overflow-y-auto max-h-[calc(90vh-300px)]">
          {activeTab === 'history' && (
            <div className="p-6">
              {/* Add Payment Button */}
              <div className="flex justify-between items-center mb-6">
                <h3 className="text-lg font-medium text-gray-900">Registro de Pagos</h3>
                <Button
                  variant="default"
                  iconName="Plus"
                  iconPosition="left"
                  onClick={() => setShowAddPayment(true)}
                >
                  Registrar Pago
                </Button>
              </div>

              {/* Add Payment Form */}
              {showAddPayment && (
                <div className="mb-6 p-4 bg-gray-50 rounded-lg border border-gray-200">
                  <h4 className="text-md font-medium text-gray-900 mb-4">Registrar Nuevo Pago</h4>
                  <form onSubmit={handleAddPayment} className="space-y-4">
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                      <Input
                        label="Monto (CLP)"
                        type="number"
                        placeholder="450000"
                        value={paymentData?.amount}
                        onChange={(e) => setPaymentData(prev => ({ ...prev, amount: e?.target?.value }))}
                        required
                      />
                      
                      <Input
                        label="Fecha de Pago"
                        type="date"
                        value={paymentData?.date}
                        onChange={(e) => setPaymentData(prev => ({ ...prev, date: e?.target?.value }))}
                        required
                      />
                      
                      <Select
                        label="Método de Pago"
                        options={paymentMethods}
                        value={paymentData?.method}
                        onChange={(value) => setPaymentData(prev => ({ ...prev, method: value }))}
                        required
                      />
                      
                      <Input
                        label="Referencia"
                        type="text"
                        placeholder="Número de transferencia, cheque, etc."
                        value={paymentData?.reference}
                        onChange={(e) => setPaymentData(prev => ({ ...prev, reference: e?.target?.value }))}
                      />
                    </div>
                    
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Notas
                      </label>
                      <textarea
                        rows={2}
                        placeholder="Observaciones adicionales..."
                        value={paymentData?.notes}
                        onChange={(e) => setPaymentData(prev => ({ ...prev, notes: e?.target?.value }))}
                        className="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-primary focus:border-primary resize-none"
                      />
                    </div>
                    
                    <div className="flex items-center space-x-3">
                      <Button type="submit" variant="default" iconName="Save">
                        Guardar Pago
                      </Button>
                      <Button
                        type="button"
                        variant="outline"
                        onClick={() => setShowAddPayment(false)}
                      >
                        Cancelar
                      </Button>
                    </div>
                  </form>
                </div>
              )}

              {/* Payment History Table */}
              <div className="bg-white border border-gray-200 rounded-lg overflow-hidden">
                <table className="w-full">
                  <thead className="bg-gray-50">
                    <tr>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Fecha
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Monto
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Método
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Referencia
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Estado
                      </th>
                      <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                        Notas
                      </th>
                    </tr>
                  </thead>
                  <tbody className="bg-white divide-y divide-gray-200">
                    {paymentHistory?.map((payment) => (
                      <tr key={payment?.id} className="hover:bg-gray-50">
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                          {formatDate(payment?.date)}
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                          {formatCurrency(payment?.amount)}
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                          {payment?.method}
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                          {payment?.reference}
                        </td>
                        <td className="px-6 py-4 whitespace-nowrap">
                          <StatusIndicatorBadges status={payment?.status} size="sm" />
                        </td>
                        <td className="px-6 py-4 text-sm text-gray-500 max-w-xs truncate">
                          {payment?.notes}
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </div>
            </div>
          )}

          {activeTab === 'summary' && (
            <div className="p-6">
              <h3 className="text-lg font-medium text-gray-900 mb-6">Resumen Anual 2024</h3>
              
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div className="bg-white border border-gray-200 rounded-lg p-6">
                  <h4 className="text-md font-medium text-gray-900 mb-4">Pagos por Mes</h4>
                  <div className="space-y-3">
                    {['Enero', 'Febrero', 'Marzo', 'Abril', 'Mayo', 'Junio', 'Julio', 'Agosto']?.map((month, index) => (
                      <div key={month} className="flex items-center justify-between">
                        <span className="text-sm text-gray-600">{month}</span>
                        <div className="flex items-center space-x-2">
                          <span className="text-sm font-medium text-gray-900">
                            {formatCurrency(450000)}
                          </span>
                          <StatusIndicatorBadges status="paid" size="sm" />
                        </div>
                      </div>
                    ))}
                    <div className="flex items-center justify-between border-t pt-3">
                      <span className="text-sm text-gray-600">Septiembre</span>
                      <div className="flex items-center space-x-2">
                        <span className="text-sm font-medium text-gray-900">
                          {formatCurrency(450000)}
                        </span>
                        <StatusIndicatorBadges status="pending" size="sm" />
                      </div>
                    </div>
                  </div>
                </div>
                
                <div className="bg-white border border-gray-200 rounded-lg p-6">
                  <h4 className="text-md font-medium text-gray-900 mb-4">Estadísticas</h4>
                  <div className="space-y-4">
                    <div>
                      <p className="text-sm text-gray-600">Pagos a Tiempo</p>
                      <p className="text-2xl font-bold text-success">8/9</p>
                      <p className="text-xs text-gray-500">88.9% de cumplimiento</p>
                    </div>
                    
                    <div>
                      <p className="text-sm text-gray-600">Promedio Mensual</p>
                      <p className="text-2xl font-bold text-gray-900">{formatCurrency(450000)}</p>
                    </div>
                    
                    <div>
                      <p className="text-sm text-gray-600">Total Año</p>
                      <p className="text-2xl font-bold text-primary">{formatCurrency(3600000)}</p>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          )}
        </div>

        {/* Footer */}
        <div className="flex items-center justify-end p-6 border-t border-gray-200 bg-gray-50">
          <Button variant="outline" onClick={onClose}>
            Cerrar
          </Button>
        </div>
      </div>
    </div>
  );
};

export default PaymentHistoryModal;