import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Image from '../../../components/AppImage';
import Button from '../../../components/ui/Button';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';

const TenantCommunicationModal = ({ isOpen, onClose, tenant, onSendMessage }) => {
  const [messageData, setMessageData] = useState({
    type: 'email',
    template: '',
    subject: '',
    message: '',
    scheduledDate: '',
    scheduledTime: ''
  });

  const [isSubmitting, setIsSubmitting] = useState(false);
  const [previewMode, setPreviewMode] = useState(false);

  const messageTypes = [
    { value: 'email', label: 'Correo Electrónico' },
    { value: 'sms', label: 'SMS' },
    { value: 'whatsapp', label: 'WhatsApp' }
  ];

  const messageTemplates = [
    { value: '', label: 'Mensaje Personalizado' },
    { value: 'payment_reminder', label: 'Recordatorio de Pago' },
    { value: 'lease_expiry', label: 'Vencimiento de Contrato' },
    { value: 'maintenance_notice', label: 'Aviso de Mantenimiento' },
    { value: 'general_notice', label: 'Aviso General' },
    { value: 'welcome', label: 'Mensaje de Bienvenida' }
  ];

  const templateMessages = {
    payment_reminder: {
      subject: 'Recordatorio de Pago - {property}',
      message: `Estimado/a {tenant_name},\n\nEsperamos que se encuentre bien. Le recordamos que el pago del arriendo de {property} correspondiente al mes actual tiene vencimiento próximo.\n\nMonto: ${tenant?.monthlyRent ? new Intl.NumberFormat('es-CL')?.format(tenant?.monthlyRent) : '0'} CLP\nFecha de vencimiento: {due_date}\n\nPuede realizar el pago a través de los medios habituales. Si ya realizó el pago, por favor ignore este mensaje.\n\nSaludos cordiales,\nEquipo PropManager`
    },
    lease_expiry: {
      subject: 'Próximo Vencimiento de Contrato - {property}',
      message: `Estimado/a {tenant_name},\n\nLe informamos que su contrato de arriendo para {property} vencerá próximamente.\n\nFecha de vencimiento: {lease_end_date}\n\nLe solicitamos ponerse en contacto con nosotros para coordinar la renovación o finalización del contrato.\n\nSaludos cordiales,\nEquipo PropManager`
    },
    maintenance_notice: {
      subject: 'Aviso de Mantenimiento - {property}',
      message: `Estimado/a {tenant_name},\n\nLe informamos que se realizarán trabajos de mantenimiento en {property}.\n\nFecha programada: {maintenance_date}\nHorario estimado: {maintenance_time}\n\nAgradecemos su comprensión y colaboración.\n\nSaludos cordiales,\nEquipo PropManager`
    },
    general_notice: {
      subject: 'Aviso Importante - {property}',
      message: `Estimado/a {tenant_name},\n\nEsperamos que se encuentre bien. Le escribimos para informarle sobre:\n\n[Escriba aquí el contenido del aviso]\n\nSi tiene alguna consulta, no dude en contactarnos.\n\nSaludos cordiales,\nEquipo PropManager`
    },
    welcome: {
      subject: 'Bienvenido/a a {property}',
      message: `Estimado/a {tenant_name},\n\n¡Bienvenido/a a su nuevo hogar en {property}!\n\nNos complace tenerle como inquilino/a. Si tiene alguna consulta o necesita asistencia, no dude en contactarnos.\n\nDatos de contacto:\nTeléfono: +56 2 1234 5678\nEmail: contacto@propmanager.cl\n\n¡Esperamos que disfrute su estadía!\n\nSaludos cordiales,\nEquipo PropManager`
    }
  };

  const handleTemplateChange = (templateValue) => {
    setMessageData(prev => ({
      ...prev,
      template: templateValue,
      subject: templateMessages?.[templateValue]?.subject || '',
      message: templateMessages?.[templateValue]?.message || ''
    }));
  };

  const handleInputChange = (field, value) => {
    setMessageData(prev => ({ ...prev, [field]: value }));
  };

  const replaceVariables = (text) => {
    if (!tenant) return text;
    
    return text?.replace(/{tenant_name}/g, tenant?.name)?.replace(/{property}/g, tenant?.property)?.replace(/{due_date}/g, new Date()?.toLocaleDateString('es-CL'))?.replace(/{lease_end_date}/g, new Date(tenant.leaseExpiry)?.toLocaleDateString('es-CL'))?.replace(/{maintenance_date}/g, new Date()?.toLocaleDateString('es-CL'))?.replace(/{maintenance_time}/g, '09:00 - 17:00');
  };

  const handleSubmit = async (e) => {
    e?.preventDefault();
    setIsSubmitting(true);

    try {
      const messagePayload = {
        ...messageData,
        tenantId: tenant?.id,
        subject: replaceVariables(messageData?.subject),
        message: replaceVariables(messageData?.message),
        sentAt: new Date()?.toISOString(),
        status: messageData?.scheduledDate ? 'scheduled' : 'sent'
      };

      await onSendMessage(messagePayload);
      handleClose();
    } catch (error) {
      console.error('Error sending message:', error);
    } finally {
      setIsSubmitting(false);
    }
  };

  const handleClose = () => {
    setMessageData({
      type: 'email',
      template: '',
      subject: '',
      message: '',
      scheduledDate: '',
      scheduledTime: ''
    });
    setPreviewMode(false);
    onClose();
  };

  if (!isOpen || !tenant) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 z-1000 flex items-center justify-center p-4">
      <div className="bg-white rounded-lg shadow-xl w-full max-w-3xl max-h-[90vh] overflow-hidden">
        {/* Header */}
        <div className="flex items-center justify-between p-6 border-b border-gray-200">
          <div className="flex items-center space-x-3">
            <div className="w-10 h-10 bg-gray-200 rounded-full overflow-hidden">
              {tenant?.avatar ? (
                <Image
                  src={tenant?.avatar}
                  alt={tenant?.name}
                  className="w-full h-full object-cover"
                />
              ) : (
                <div className="w-full h-full bg-primary/10 flex items-center justify-center">
                  <Icon name="User" size={20} className="text-primary" />
                </div>
              )}
            </div>
            <div>
              <h2 className="text-xl font-semibold text-gray-900">Enviar Mensaje</h2>
              <p className="text-sm text-gray-500">{tenant?.name} - {tenant?.property}</p>
            </div>
          </div>
          <Button
            variant="ghost"
            size="sm"
            iconName="X"
            onClick={handleClose}
            className="text-gray-500 hover:text-gray-700"
          />
        </div>

        {/* Content */}
        <div className="overflow-y-auto max-h-[calc(90vh-140px)]">
          {!previewMode ? (
            <form onSubmit={handleSubmit} className="p-6 space-y-6">
              {/* Message Configuration */}
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <Select
                  label="Tipo de Mensaje"
                  options={messageTypes}
                  value={messageData?.type}
                  onChange={(value) => handleInputChange('type', value)}
                  required
                />
                
                <Select
                  label="Plantilla"
                  options={messageTemplates}
                  value={messageData?.template}
                  onChange={handleTemplateChange}
                  placeholder="Seleccionar plantilla"
                />
              </div>

              {/* Subject (for email) */}
              {messageData?.type === 'email' && (
                <Input
                  label="Asunto"
                  type="text"
                  placeholder="Asunto del correo"
                  value={messageData?.subject}
                  onChange={(e) => handleInputChange('subject', e?.target?.value)}
                  required
                />
              )}

              {/* Message Content */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Mensaje *
                </label>
                <textarea
                  rows={8}
                  placeholder="Escriba su mensaje aquí..."
                  value={messageData?.message}
                  onChange={(e) => handleInputChange('message', e?.target?.value)}
                  className="w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:ring-primary focus:border-primary resize-none"
                  required
                />
                <p className="mt-1 text-xs text-gray-500">
                  Variables disponibles: {'{tenant_name}'}, {'{property}'}, {'{due_date}'}, {'{lease_end_date}'}
                </p>
              </div>

              {/* Scheduling Options */}
              <div>
                <h3 className="text-lg font-medium text-gray-900 mb-4">Programar Envío (Opcional)</h3>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <Input
                    label="Fecha de Envío"
                    type="date"
                    value={messageData?.scheduledDate}
                    onChange={(e) => handleInputChange('scheduledDate', e?.target?.value)}
                    min={new Date()?.toISOString()?.split('T')?.[0]}
                  />
                  
                  <Input
                    label="Hora de Envío"
                    type="time"
                    value={messageData?.scheduledTime}
                    onChange={(e) => handleInputChange('scheduledTime', e?.target?.value)}
                  />
                </div>
              </div>

              {/* Contact Information Display */}
              <div className="bg-gray-50 p-4 rounded-lg">
                <h4 className="text-sm font-medium text-gray-900 mb-2">Información de Contacto</h4>
                <div className="text-sm text-gray-600 space-y-1">
                  <p><strong>Email:</strong> {tenant?.email}</p>
                  <p><strong>Teléfono:</strong> {tenant?.phone}</p>
                  <p><strong>Propiedad:</strong> {tenant?.property}</p>
                </div>
              </div>
            </form>
          ) : (
            /* Preview Mode */
            (<div className="p-6">
              <div className="bg-gray-50 p-6 rounded-lg">
                <h3 className="text-lg font-medium text-gray-900 mb-4">Vista Previa del Mensaje</h3>
                
                {messageData?.type === 'email' && (
                  <div className="mb-4">
                    <p className="text-sm text-gray-600">Asunto:</p>
                    <p className="font-medium">{replaceVariables(messageData?.subject)}</p>
                  </div>
                )}
                
                <div className="mb-4">
                  <p className="text-sm text-gray-600">Mensaje:</p>
                  <div className="mt-2 p-4 bg-white border rounded-md">
                    <pre className="whitespace-pre-wrap text-sm text-gray-900 font-sans">
                      {replaceVariables(messageData?.message)}
                    </pre>
                  </div>
                </div>
                
                <div className="text-sm text-gray-500">
                  <p><strong>Destinatario:</strong> {tenant?.name} ({tenant?.email})</p>
                  <p><strong>Tipo:</strong> {messageTypes?.find(t => t?.value === messageData?.type)?.label}</p>
                  {messageData?.scheduledDate && (
                    <p><strong>Programado para:</strong> {messageData?.scheduledDate} {messageData?.scheduledTime}</p>
                  )}
                </div>
              </div>
            </div>)
          )}
        </div>

        {/* Footer */}
        <div className="flex items-center justify-between p-6 border-t border-gray-200 bg-gray-50">
          <div className="flex items-center space-x-3">
            <Button
              type="button"
              variant="outline"
              onClick={handleClose}
              disabled={isSubmitting}
            >
              Cancelar
            </Button>
            
            {!previewMode && (
              <Button
                type="button"
                variant="ghost"
                iconName="Eye"
                onClick={() => setPreviewMode(true)}
                disabled={!messageData?.message?.trim()}
              >
                Vista Previa
              </Button>
            )}
          </div>
          
          <div className="flex items-center space-x-3">
            {previewMode && (
              <Button
                type="button"
                variant="outline"
                iconName="Edit"
                onClick={() => setPreviewMode(false)}
              >
                Editar
              </Button>
            )}
            
            <Button
              type="submit"
              variant="default"
              loading={isSubmitting}
              iconName="Send"
              iconPosition="left"
              onClick={previewMode ? handleSubmit : undefined}
              form={!previewMode ? undefined : 'message-form'}
              disabled={!messageData?.message?.trim()}
            >
              {messageData?.scheduledDate ? 'Programar Envío' : 'Enviar Ahora'}
            </Button>
          </div>
        </div>
      </div>
    </div>
  );
};

export default TenantCommunicationModal;