import React, { useState, useMemo } from 'react';
import { useNavigate } from 'react-router-dom';
import Icon from '../../../components/AppIcon';
import Image from '../../../components/AppImage';
import Button from '../../../components/ui/Button';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';
import { Checkbox } from '../../../components/ui/Checkbox';
import StatusIndicatorBadges from '../../../components/ui/StatusIndicatorBadges';

const TenantsTable = ({ 
  tenants, 
  onAddTenant, 
  onEditTenant, 
  onDeleteTenant, 
  onBulkAction,
  onSendMessage,
  onViewPaymentHistory 
}) => {
  const navigate = useNavigate();
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState('all');
  const [leaseFilter, setLeaseFilter] = useState('all');
  const [sortConfig, setSortConfig] = useState({ key: 'name', direction: 'asc' });
  const [selectedTenants, setSelectedTenants] = useState([]);
  const [currentPage, setCurrentPage] = useState(1);
  const itemsPerPage = 10;

  const statusOptions = [
    { value: 'all', label: 'Todos los Estados' },
    { value: 'paid', label: 'Al Día' },
    { value: 'pending', label: 'Pendiente' },
    { value: 'overdue', label: 'Atrasado' }
  ];

  const leaseOptions = [
    { value: 'all', label: 'Todos los Contratos' },
    { value: 'active', label: 'Contratos Activos' },
    { value: 'expiring', label: 'Por Vencer (30 días)' },
    { value: 'expired', label: 'Vencidos' }
  ];

  const filteredAndSortedTenants = useMemo(() => {
    let filtered = tenants?.filter(tenant => {
      const matchesSearch = tenant?.name?.toLowerCase()?.includes(searchTerm?.toLowerCase()) ||
                           tenant?.email?.toLowerCase()?.includes(searchTerm?.toLowerCase()) ||
                           tenant?.property?.toLowerCase()?.includes(searchTerm?.toLowerCase());
      
      const matchesStatus = statusFilter === 'all' || tenant?.paymentStatus === statusFilter;
      
      const matchesLease = leaseFilter === 'all' || tenant?.leaseStatus === leaseFilter;
      
      return matchesSearch && matchesStatus && matchesLease;
    });

    if (sortConfig?.key) {
      filtered?.sort((a, b) => {
        let aValue = a?.[sortConfig?.key];
        let bValue = b?.[sortConfig?.key];
        
        if (sortConfig?.key === 'leaseExpiry') {
          aValue = new Date(aValue);
          bValue = new Date(bValue);
        }
        
        if (aValue < bValue) return sortConfig?.direction === 'asc' ? -1 : 1;
        if (aValue > bValue) return sortConfig?.direction === 'asc' ? 1 : -1;
        return 0;
      });
    }

    return filtered;
  }, [tenants, searchTerm, statusFilter, leaseFilter, sortConfig]);

  const paginatedTenants = useMemo(() => {
    const startIndex = (currentPage - 1) * itemsPerPage;
    return filteredAndSortedTenants?.slice(startIndex, startIndex + itemsPerPage);
  }, [filteredAndSortedTenants, currentPage]);

  const totalPages = Math.ceil(filteredAndSortedTenants?.length / itemsPerPage);

  const handleSort = (key) => {
    setSortConfig(prev => ({
      key,
      direction: prev?.key === key && prev?.direction === 'asc' ? 'desc' : 'asc'
    }));
  };

  const handleSelectAll = (checked) => {
    if (checked) {
      setSelectedTenants(paginatedTenants?.map(tenant => tenant?.id));
    } else {
      setSelectedTenants([]);
    }
  };

  const handleSelectTenant = (tenantId, checked) => {
    if (checked) {
      setSelectedTenants(prev => [...prev, tenantId]);
    } else {
      setSelectedTenants(prev => prev?.filter(id => id !== tenantId));
    }
  };

  const handleBulkAction = (action) => {
    if (selectedTenants?.length === 0) return;
    onBulkAction(action, selectedTenants);
    setSelectedTenants([]);
  };

  const getSortIcon = (key) => {
    if (sortConfig?.key !== key) return 'ArrowUpDown';
    return sortConfig?.direction === 'asc' ? 'ArrowUp' : 'ArrowDown';
  };

  const formatDate = (dateString) => {
    return new Date(dateString)?.toLocaleDateString('es-CL', {
      day: '2-digit',
      month: '2-digit',
      year: 'numeric'
    });
  };

  const formatPhone = (phone) => {
    return phone?.replace(/(\+56)(\d{1})(\d{4})(\d{4})/, '$1 $2 $3 $4');
  };

  return (
    <div className="bg-white rounded-lg shadow-sm border border-gray-200">
      {/* Header with Search and Filters */}
      <div className="p-6 border-b border-gray-200">
        <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between space-y-4 lg:space-y-0">
          <div className="flex-1 max-w-md">
            <Input
              type="search"
              placeholder="Buscar por nombre, email o propiedad..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e?.target?.value)}
              className="w-full"
            />
          </div>
          
          <div className="flex flex-col sm:flex-row space-y-2 sm:space-y-0 sm:space-x-3">
            <Select
              options={statusOptions}
              value={statusFilter}
              onChange={setStatusFilter}
              placeholder="Estado de Pago"
              className="w-full sm:w-48"
            />
            
            <Select
              options={leaseOptions}
              value={leaseFilter}
              onChange={setLeaseFilter}
              placeholder="Estado de Contrato"
              className="w-full sm:w-48"
            />
            
            <Button
              variant="default"
              iconName="Plus"
              iconPosition="left"
              onClick={onAddTenant}
              className="w-full sm:w-auto"
            >
              Nuevo Inquilino
            </Button>
          </div>
        </div>

        {/* Bulk Actions */}
        {selectedTenants?.length > 0 && (
          <div className="mt-4 p-3 bg-blue-50 rounded-lg border border-blue-200">
            <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between space-y-2 sm:space-y-0">
              <span className="text-sm text-blue-700 font-medium">
                {selectedTenants?.length} inquilino{selectedTenants?.length > 1 ? 's' : ''} seleccionado{selectedTenants?.length > 1 ? 's' : ''}
              </span>
              <div className="flex space-x-2">
                <Button
                  variant="outline"
                  size="sm"
                  iconName="Mail"
                  onClick={() => handleBulkAction('send-reminder')}
                >
                  Enviar Recordatorio
                </Button>
                <Button
                  variant="outline"
                  size="sm"
                  iconName="FileText"
                  onClick={() => handleBulkAction('generate-report')}
                >
                  Generar Reporte
                </Button>
                <Button
                  variant="ghost"
                  size="sm"
                  iconName="X"
                  onClick={() => setSelectedTenants([])}
                >
                  Cancelar
                </Button>
              </div>
            </div>
          </div>
        )}
      </div>
      {/* Desktop Table */}
      <div className="hidden lg:block overflow-x-auto">
        <table className="w-full">
          <thead className="bg-gray-50 border-b border-gray-200">
            <tr>
              <th className="w-12 px-6 py-3 text-left">
                <Checkbox
                  checked={selectedTenants?.length === paginatedTenants?.length && paginatedTenants?.length > 0}
                  onChange={(e) => handleSelectAll(e?.target?.checked)}
                />
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                <button
                  onClick={() => handleSort('name')}
                  className="flex items-center space-x-1 hover:text-gray-700"
                >
                  <span>Inquilino</span>
                  <Icon name={getSortIcon('name')} size={14} />
                </button>
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                <button
                  onClick={() => handleSort('property')}
                  className="flex items-center space-x-1 hover:text-gray-700"
                >
                  <span>Propiedad</span>
                  <Icon name={getSortIcon('property')} size={14} />
                </button>
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Contacto
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                <button
                  onClick={() => handleSort('paymentStatus')}
                  className="flex items-center space-x-1 hover:text-gray-700"
                >
                  <span>Estado Pago</span>
                  <Icon name={getSortIcon('paymentStatus')} size={14} />
                </button>
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                <button
                  onClick={() => handleSort('leaseExpiry')}
                  className="flex items-center space-x-1 hover:text-gray-700"
                >
                  <span>Vence Contrato</span>
                  <Icon name={getSortIcon('leaseExpiry')} size={14} />
                </button>
              </th>
              <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                Acciones
              </th>
            </tr>
          </thead>
          <tbody className="bg-white divide-y divide-gray-200">
            {paginatedTenants?.map((tenant) => (
              <tr key={tenant?.id} className="hover:bg-gray-50 transition-colors duration-150">
                <td className="px-6 py-4">
                  <Checkbox
                    checked={selectedTenants?.includes(tenant?.id)}
                    onChange={(e) => handleSelectTenant(tenant?.id, e?.target?.checked)}
                  />
                </td>
                <td className="px-6 py-4">
                  <div className="flex items-center space-x-3">
                    <div className="w-10 h-10 bg-gray-200 rounded-full overflow-hidden flex-shrink-0">
                      {tenant?.avatar ? (
                        <Image
                          src={tenant?.avatar}
                          alt={tenant?.name}
                          className="w-full h-full object-cover"
                        />
                      ) : (
                        <div className="w-full h-full bg-primary/10 flex items-center justify-center">
                          <Icon name="User" size={20} className="text-primary" />
                        </div>
                      )}
                    </div>
                    <div>
                      <p className="text-sm font-medium text-gray-900">{tenant?.name}</p>
                      <p className="text-sm text-gray-500">{tenant?.rut}</p>
                    </div>
                  </div>
                </td>
                <td className="px-6 py-4">
                  <div>
                    <p className="text-sm font-medium text-gray-900">{tenant?.property}</p>
                    <p className="text-sm text-gray-500">{tenant?.propertyAddress}</p>
                  </div>
                </td>
                <td className="px-6 py-4">
                  <div>
                    <p className="text-sm text-gray-900">{tenant?.email}</p>
                    <p className="text-sm text-gray-500">{formatPhone(tenant?.phone)}</p>
                  </div>
                </td>
                <td className="px-6 py-4">
                  <StatusIndicatorBadges status={tenant?.paymentStatus} size="sm" />
                </td>
                <td className="px-6 py-4">
                  <div>
                    <p className="text-sm text-gray-900">{formatDate(tenant?.leaseExpiry)}</p>
                    <StatusIndicatorBadges status={tenant?.leaseStatus} size="sm" />
                  </div>
                </td>
                <td className="px-6 py-4 text-right">
                  <div className="flex items-center justify-end space-x-2">
                    <Button
                      variant="ghost"
                      size="sm"
                      iconName="MessageCircle"
                      onClick={() => onSendMessage(tenant)}
                      className="text-gray-500 hover:text-gray-700"
                    />
                    <Button
                      variant="ghost"
                      size="sm"
                      iconName="CreditCard"
                      onClick={() => onViewPaymentHistory(tenant)}
                      className="text-gray-500 hover:text-gray-700"
                    />
                    <Button
                      variant="ghost"
                      size="sm"
                      iconName="Edit"
                      onClick={() => onEditTenant(tenant)}
                      className="text-gray-500 hover:text-gray-700"
                    />
                  </div>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
      {/* Mobile Card Layout */}
      <div className="lg:hidden divide-y divide-gray-200">
        {paginatedTenants?.map((tenant) => (
          <div key={tenant?.id} className="p-4">
            <div className="flex items-start space-x-3">
              <Checkbox
                checked={selectedTenants?.includes(tenant?.id)}
                onChange={(e) => handleSelectTenant(tenant?.id, e?.target?.checked)}
                className="mt-1"
              />
              
              <div className="w-12 h-12 bg-gray-200 rounded-full overflow-hidden flex-shrink-0">
                {tenant?.avatar ? (
                  <Image
                    src={tenant?.avatar}
                    alt={tenant?.name}
                    className="w-full h-full object-cover"
                  />
                ) : (
                  <div className="w-full h-full bg-primary/10 flex items-center justify-center">
                    <Icon name="User" size={24} className="text-primary" />
                  </div>
                )}
              </div>
              
              <div className="flex-1 min-w-0">
                <div className="flex items-start justify-between">
                  <div>
                    <h3 className="text-sm font-medium text-gray-900 truncate">{tenant?.name}</h3>
                    <p className="text-sm text-gray-500">{tenant?.rut}</p>
                  </div>
                  <div className="flex space-x-1 ml-2">
                    <Button
                      variant="ghost"
                      size="sm"
                      iconName="MessageCircle"
                      onClick={() => onSendMessage(tenant)}
                    />
                    <Button
                      variant="ghost"
                      size="sm"
                      iconName="Edit"
                      onClick={() => onEditTenant(tenant)}
                    />
                  </div>
                </div>
                
                <div className="mt-2 space-y-2">
                  <div>
                    <p className="text-sm font-medium text-gray-900">{tenant?.property}</p>
                    <p className="text-xs text-gray-500">{tenant?.propertyAddress}</p>
                  </div>
                  
                  <div className="flex items-center space-x-4">
                    <StatusIndicatorBadges status={tenant?.paymentStatus} size="sm" />
                    <StatusIndicatorBadges status={tenant?.leaseStatus} size="sm" />
                  </div>
                  
                  <div className="text-xs text-gray-500">
                    <p>{tenant?.email}</p>
                    <p>{formatPhone(tenant?.phone)}</p>
                    <p>Contrato vence: {formatDate(tenant?.leaseExpiry)}</p>
                  </div>
                </div>
              </div>
            </div>
          </div>
        ))}
      </div>
      {/* Pagination */}
      {totalPages > 1 && (
        <div className="px-6 py-4 border-t border-gray-200">
          <div className="flex items-center justify-between">
            <div className="text-sm text-gray-500">
              Mostrando {((currentPage - 1) * itemsPerPage) + 1} a {Math.min(currentPage * itemsPerPage, filteredAndSortedTenants?.length)} de {filteredAndSortedTenants?.length} inquilinos
            </div>
            
            <div className="flex items-center space-x-2">
              <Button
                variant="outline"
                size="sm"
                iconName="ChevronLeft"
                disabled={currentPage === 1}
                onClick={() => setCurrentPage(prev => prev - 1)}
              >
                Anterior
              </Button>
              
              <div className="flex items-center space-x-1">
                {Array.from({ length: Math.min(5, totalPages) }, (_, i) => {
                  const page = i + 1;
                  return (
                    <Button
                      key={page}
                      variant={currentPage === page ? "default" : "ghost"}
                      size="sm"
                      onClick={() => setCurrentPage(page)}
                      className="w-8 h-8"
                    >
                      {page}
                    </Button>
                  );
                })}
              </div>
              
              <Button
                variant="outline"
                size="sm"
                iconName="ChevronRight"
                disabled={currentPage === totalPages}
                onClick={() => setCurrentPage(prev => prev + 1)}
              >
                Siguiente
              </Button>
            </div>
          </div>
        </div>
      )}
      {/* Empty State */}
      {filteredAndSortedTenants?.length === 0 && (
        <div className="text-center py-12">
          <Icon name="Users" size={48} className="text-gray-400 mx-auto mb-4" />
          <h3 className="text-lg font-medium text-gray-900 mb-2">No se encontraron inquilinos</h3>
          <p className="text-gray-500 mb-6">
            {searchTerm || statusFilter !== 'all' || leaseFilter !== 'all' ?'Intenta ajustar los filtros de búsqueda' :'Comienza agregando tu primer inquilino al sistema'
            }
          </p>
          {(!searchTerm && statusFilter === 'all' && leaseFilter === 'all') && (
            <Button
              variant="default"
              iconName="Plus"
              iconPosition="left"
              onClick={onAddTenant}
            >
              Agregar Primer Inquilino
            </Button>
          )}
        </div>
      )}
    </div>
  );
};

export default TenantsTable;