import React, { useState, useEffect } from 'react';
import { Helmet } from 'react-helmet';
import Header from '../../components/ui/Header';
import Sidebar from '../../components/ui/Sidebar';
import NavigationBreadcrumbs from '../../components/ui/NavigationBreadcrumbs';
import QuickActionFloatingButton from '../../components/ui/QuickActionFloatingButton';
import TenantsTable from './components/TenantsTable';
import AddTenantModal from './components/AddTenantModal';
import TenantCommunicationModal from './components/TenantCommunicationModal';
import PaymentHistoryModal from './components/PaymentHistoryModal';

const TenantsManagement = () => {
  const [isSidebarOpen, setIsSidebarOpen] = useState(false);
  const [showAddModal, setShowAddModal] = useState(false);
  const [showCommunicationModal, setShowCommunicationModal] = useState(false);
  const [showPaymentModal, setShowPaymentModal] = useState(false);
  const [selectedTenant, setSelectedTenant] = useState(null);
  const [tenants, setTenants] = useState([]);
  const [properties, setProperties] = useState([]);

  // Mock data for tenants
  const mockTenants = [
    {
      id: 1,
      name: 'Carlos Mendoza Ruiz',
      rut: '12.345.678-9',
      email: 'carlos.mendoza@email.com',
      phone: '+56 9 1234 5678',
      property: 'Departamento Las Condes 101',
      propertyAddress: 'Av. Apoquindo 1234, Las Condes',
      paymentStatus: 'paid',
      leaseStatus: 'active',
      leaseExpiry: '2025-03-15',
      monthlyRent: 450000,
      avatar: 'https://randomuser.me/api/portraits/men/32.jpg',
      emergencyContact: 'María Mendoza',
      emergencyPhone: '+56 9 8765 4321',
      deposit: 450000,
      notes: 'Inquilino puntual, excelente historial de pagos'
    },
    {
      id: 2,
      name: 'Ana Patricia González',
      rut: '23.456.789-0',
      email: 'ana.gonzalez@email.com',
      phone: '+56 9 2345 6789',
      property: 'Casa Providencia 205',
      propertyAddress: 'Calle Los Leones 567, Providencia',
      paymentStatus: 'pending',
      leaseStatus: 'active',
      leaseExpiry: '2024-12-31',
      monthlyRent: 650000,
      avatar: 'https://randomuser.me/api/portraits/women/44.jpg',
      emergencyContact: 'Pedro González',
      emergencyPhone: '+56 9 7654 3210',
      deposit: 650000,
      notes: 'Contrato próximo a vencer, evaluar renovación'
    },
    {
      id: 3,
      name: 'Roberto Silva Morales',
      rut: '34.567.890-1',
      email: 'roberto.silva@email.com',
      phone: '+56 9 3456 7890',
      property: 'Oficina Centro 302',
      propertyAddress: 'Av. Libertador 890, Santiago Centro',
      paymentStatus: 'overdue',
      leaseStatus: 'active',
      leaseExpiry: '2025-06-30',
      monthlyRent: 380000,
      avatar: 'https://randomuser.me/api/portraits/men/56.jpg',
      emergencyContact: 'Carmen Silva',
      emergencyPhone: '+56 9 6543 2109',
      deposit: 380000,
      notes: 'Pago atrasado, enviar recordatorio urgente'
    },
    {
      id: 4,
      name: 'Francisca Torres López',
      rut: '45.678.901-2',
      email: 'francisca.torres@email.com',
      phone: '+56 9 4567 8901',
      property: 'Departamento Ñuñoa 150',
      propertyAddress: 'Av. Irarrázaval 1500, Ñuñoa',
      paymentStatus: 'paid',
      leaseStatus: 'expiring',
      leaseExpiry: '2024-10-15',
      monthlyRent: 420000,
      avatar: 'https://randomuser.me/api/portraits/women/68.jpg',
      emergencyContact: 'Luis Torres',
      emergencyPhone: '+56 9 5432 1098',
      deposit: 420000,
      notes: 'Contrato vence en 30 días, coordinar renovación'
    },
    {
      id: 5,
      name: 'Miguel Ángel Herrera',
      rut: '56.789.012-3',
      email: 'miguel.herrera@email.com',
      phone: '+56 9 5678 9012',
      property: 'Local Comercial Maipú',
      propertyAddress: 'Av. Pajaritos 2100, Maipú',
      paymentStatus: 'paid',
      leaseStatus: 'active',
      leaseExpiry: '2025-08-20',
      monthlyRent: 750000,
      avatar: 'https://randomuser.me/api/portraits/men/72.jpg',
      emergencyContact: 'Rosa Herrera',
      emergencyPhone: '+56 9 4321 0987',
      deposit: 750000,
      notes: 'Inquilino comercial, excelente relación'
    },
    {
      id: 6,
      name: 'Valentina Rojas Castro',
      rut: '67.890.123-4',
      email: 'valentina.rojas@email.com',
      phone: '+56 9 6789 0123',
      property: 'Departamento Vitacura 401',
      propertyAddress: 'Av. Vitacura 3456, Vitacura',
      paymentStatus: 'pending',
      leaseStatus: 'active',
      leaseExpiry: '2025-01-10',
      monthlyRent: 580000,
      avatar: 'https://randomuser.me/api/portraits/women/25.jpg',
      emergencyContact: 'Carlos Rojas',
      emergencyPhone: '+56 9 3210 9876',
      deposit: 580000,
      notes: 'Pago pendiente del mes actual'
    }
  ];

  // Mock data for properties
  const mockProperties = [
    { id: 1, name: 'Departamento Las Condes 101', address: 'Av. Apoquindo 1234, Las Condes' },
    { id: 2, name: 'Casa Providencia 205', address: 'Calle Los Leones 567, Providencia' },
    { id: 3, name: 'Oficina Centro 302', address: 'Av. Libertador 890, Santiago Centro' },
    { id: 4, name: 'Departamento Ñuñoa 150', address: 'Av. Irarrázaval 1500, Ñuñoa' },
    { id: 5, name: 'Local Comercial Maipú', address: 'Av. Pajaritos 2100, Maipú' },
    { id: 6, name: 'Departamento Vitacura 401', address: 'Av. Vitacura 3456, Vitacura' }
  ];

  useEffect(() => {
    // Simulate loading data
    setTenants(mockTenants);
    setProperties(mockProperties);
  }, []);

  const handleMenuToggle = () => {
    setIsSidebarOpen(!isSidebarOpen);
  };

  const handleCloseSidebar = () => {
    setIsSidebarOpen(false);
  };

  const handleAddTenant = () => {
    setShowAddModal(true);
  };

  const handleSaveTenant = async (tenantData) => {
    // Simulate API call
    await new Promise(resolve => setTimeout(resolve, 1000));
    
    setTenants(prev => [...prev, tenantData]);
    setShowAddModal(false);
  };

  const handleEditTenant = (tenant) => {
    setSelectedTenant(tenant);
    setShowAddModal(true);
  };

  const handleDeleteTenant = (tenantId) => {
    setTenants(prev => prev?.filter(tenant => tenant?.id !== tenantId));
  };

  const handleBulkAction = (action, selectedIds) => {
    console.log(`Bulk action: ${action} for tenants:`, selectedIds);
    
    switch (action) {
      case 'send-reminder':
        // Handle bulk reminder sending
        break;
      case 'generate-report':
        // Handle bulk report generation
        break;
      default:
        break;
    }
  };

  const handleSendMessage = (tenant) => {
    setSelectedTenant(tenant);
    setShowCommunicationModal(true);
  };

  const handleSendMessageAction = async (messageData) => {
    // Simulate API call
    await new Promise(resolve => setTimeout(resolve, 1000));
    
    console.log('Message sent:', messageData);
    setShowCommunicationModal(false);
    setSelectedTenant(null);
  };

  const handleViewPaymentHistory = (tenant) => {
    setSelectedTenant(tenant);
    setShowPaymentModal(true);
  };

  const handleRecordPayment = (tenantId, paymentData) => {
    console.log('Recording payment for tenant:', tenantId, paymentData);
    
    // Update tenant payment status
    setTenants(prev => prev?.map(tenant => 
      tenant?.id === tenantId 
        ? { ...tenant, paymentStatus: 'paid' }
        : tenant
    ));
  };

  return (
    <div className="min-h-screen bg-background">
      <Helmet>
        <title>Gestión de Inquilinos - PropManager</title>
        <meta name="description" content="Administra inquilinos, contratos y comunicaciones de manera eficiente con PropManager" />
      </Helmet>
      <Header onMenuToggle={handleMenuToggle} isMenuOpen={isSidebarOpen} />
      <Sidebar isOpen={isSidebarOpen} onClose={handleCloseSidebar} />
      <main className="lg:ml-60 pt-16">
        <div className="p-6">
          <NavigationBreadcrumbs />
          
          {/* Page Header */}
          <div className="mb-8">
            <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between">
              <div>
                <h1 className="text-2xl font-bold text-gray-900">Gestión de Inquilinos</h1>
                <p className="mt-1 text-sm text-gray-500">
                  Administra la información de inquilinos, contratos y comunicaciones
                </p>
              </div>
              
              <div className="mt-4 sm:mt-0 flex items-center space-x-3">
                <div className="text-sm text-gray-500">
                  Total: {tenants?.length} inquilinos
                </div>
              </div>
            </div>
          </div>

          {/* Main Content */}
          <TenantsTable
            tenants={tenants}
            onAddTenant={handleAddTenant}
            onEditTenant={handleEditTenant}
            onDeleteTenant={handleDeleteTenant}
            onBulkAction={handleBulkAction}
            onSendMessage={handleSendMessage}
            onViewPaymentHistory={handleViewPaymentHistory}
          />
        </div>
      </main>
      {/* Modals */}
      <AddTenantModal
        isOpen={showAddModal}
        onClose={() => {
          setShowAddModal(false);
          setSelectedTenant(null);
        }}
        onSave={handleSaveTenant}
        properties={properties}
        tenant={selectedTenant}
      />
      <TenantCommunicationModal
        isOpen={showCommunicationModal}
        onClose={() => {
          setShowCommunicationModal(false);
          setSelectedTenant(null);
        }}
        tenant={selectedTenant}
        onSendMessage={handleSendMessageAction}
      />
      <PaymentHistoryModal
        isOpen={showPaymentModal}
        onClose={() => {
          setShowPaymentModal(false);
          setSelectedTenant(null);
        }}
        tenant={selectedTenant}
        onRecordPayment={handleRecordPayment}
      />
      <QuickActionFloatingButton />
    </div>
  );
};

export default TenantsManagement;