import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Input from '../../../components/ui/Input';
import Button from '../../../components/ui/Button';
import StatusIndicatorBadges from '../../../components/ui/StatusIndicatorBadges';

const AccountSecurityTab = () => {
  const [passwordForm, setPasswordForm] = useState({
    currentPassword: '',
    newPassword: '',
    confirmPassword: ''
  });
  const [isChangingPassword, setIsChangingPassword] = useState(false);
  const [twoFactorEnabled, setTwoFactorEnabled] = useState(false);
  const [showQRCode, setShowQRCode] = useState(false);

  const activeSessions = [
    {
      id: 1,
      device: 'Chrome en Windows',
      location: 'Santiago, Chile',
      ipAddress: '192.168.1.100',
      lastActive: '2025-08-26 21:20:00',
      isCurrent: true
    },
    {
      id: 2,
      device: 'Safari en iPhone',
      location: 'Santiago, Chile',
      ipAddress: '192.168.1.101',
      lastActive: '2025-08-26 18:45:00',
      isCurrent: false
    },
    {
      id: 3,
      device: 'Chrome en Android',
      location: 'Valparaíso, Chile',
      ipAddress: '10.0.0.50',
      lastActive: '2025-08-25 14:30:00',
      isCurrent: false
    }
  ];

  const securityEvents = [
    {
      id: 1,
      type: 'login',
      description: 'Inicio de sesión exitoso',
      timestamp: '2025-08-26 21:20:00',
      ipAddress: '192.168.1.100',
      status: 'success'
    },
    {
      id: 2,
      type: 'password_change',
      description: 'Contraseña actualizada',
      timestamp: '2025-08-24 16:15:00',
      ipAddress: '192.168.1.100',
      status: 'success'
    },
    {
      id: 3,
      type: 'failed_login',
      description: 'Intento de inicio de sesión fallido',
      timestamp: '2025-08-23 09:30:00',
      ipAddress: '203.0.113.1',
      status: 'warning'
    }
  ];

  const handlePasswordChange = async (e) => {
    e?.preventDefault();
    setIsChangingPassword(true);

    // Simulate API call
    await new Promise(resolve => setTimeout(resolve, 2000));

    setPasswordForm({
      currentPassword: '',
      newPassword: '',
      confirmPassword: ''
    });
    setIsChangingPassword(false);
    console.log('Contraseña actualizada exitosamente');
  };

  const handleTerminateSession = (sessionId) => {
    console.log('Terminando sesión:', sessionId);
  };

  const handleEnable2FA = () => {
    if (!twoFactorEnabled) {
      setShowQRCode(true);
    } else {
      setTwoFactorEnabled(false);
      setShowQRCode(false);
    }
  };

  const handleConfirm2FA = () => {
    setTwoFactorEnabled(true);
    setShowQRCode(false);
    console.log('Autenticación de dos factores habilitada');
  };

  const getEventIcon = (type) => {
    switch (type) {
      case 'login': return 'LogIn';
      case 'password_change': return 'Key';
      case 'failed_login': return 'AlertTriangle';
      default: return 'Activity';
    }
  };

  const formatDateTime = (dateString) => {
    const date = new Date(dateString);
    return date?.toLocaleString('es-CL', {
      day: '2-digit',
      month: '2-digit',
      year: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };

  return (
    <div className="space-y-8">
      {/* Password Change */}
      <div className="glass-card p-6 rounded-lg">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Cambiar Contraseña</h3>
        
        <form onSubmit={handlePasswordChange} className="space-y-6">
          <Input
            label="Contraseña Actual"
            type="password"
            value={passwordForm?.currentPassword}
            onChange={(e) => setPasswordForm(prev => ({ ...prev, currentPassword: e?.target?.value }))}
            required
          />
          
          <Input
            label="Nueva Contraseña"
            type="password"
            value={passwordForm?.newPassword}
            onChange={(e) => setPasswordForm(prev => ({ ...prev, newPassword: e?.target?.value }))}
            description="Mínimo 8 caracteres, incluye mayúsculas, minúsculas y números"
            required
          />
          
          <Input
            label="Confirmar Nueva Contraseña"
            type="password"
            value={passwordForm?.confirmPassword}
            onChange={(e) => setPasswordForm(prev => ({ ...prev, confirmPassword: e?.target?.value }))}
            error={passwordForm?.confirmPassword && passwordForm?.newPassword !== passwordForm?.confirmPassword ? 'Las contraseñas no coinciden' : ''}
            required
          />
          
          <div className="flex justify-end">
            <Button
              type="submit"
              variant="default"
              loading={isChangingPassword}
              disabled={!passwordForm?.currentPassword || !passwordForm?.newPassword || passwordForm?.newPassword !== passwordForm?.confirmPassword}
              iconName="Key"
              iconPosition="left"
            >
              {isChangingPassword ? 'Actualizando...' : 'Cambiar Contraseña'}
            </Button>
          </div>
        </form>
      </div>
      {/* Two-Factor Authentication */}
      <div className="glass-card p-6 rounded-lg">
        <div className="flex items-center justify-between mb-6">
          <div>
            <h3 className="text-lg font-semibold text-gray-900">Autenticación de Dos Factores</h3>
            <p className="text-sm text-gray-500 mt-1">
              Agrega una capa adicional de seguridad a tu cuenta
            </p>
          </div>
          <StatusIndicatorBadges 
            status={twoFactorEnabled ? 'active' : 'inactive'}
            size="sm"
          />
        </div>

        {!showQRCode ? (
          <div className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
            <div className="flex items-center space-x-3">
              <div className="w-10 h-10 bg-primary/10 text-primary rounded-full flex items-center justify-center">
                <Icon name="Shield" size={20} />
              </div>
              <div>
                <p className="font-medium text-gray-900">
                  {twoFactorEnabled ? 'Autenticación habilitada' : 'Autenticación deshabilitada'}
                </p>
                <p className="text-sm text-gray-500">
                  {twoFactorEnabled 
                    ? 'Tu cuenta está protegida con 2FA' :'Habilita 2FA para mayor seguridad'
                  }
                </p>
              </div>
            </div>
            <Button
              variant={twoFactorEnabled ? 'destructive' : 'default'}
              size="sm"
              onClick={handleEnable2FA}
              iconName={twoFactorEnabled ? 'ShieldOff' : 'Shield'}
              iconPosition="left"
            >
              {twoFactorEnabled ? 'Deshabilitar' : 'Habilitar'}
            </Button>
          </div>
        ) : (
          <div className="space-y-6">
            <div className="text-center">
              <div className="w-48 h-48 bg-gray-100 rounded-lg mx-auto mb-4 flex items-center justify-center">
                <div className="text-center">
                  <Icon name="QrCode" size={64} className="text-gray-400 mx-auto mb-2" />
                  <p className="text-sm text-gray-500">Código QR para 2FA</p>
                </div>
              </div>
              <p className="text-sm text-gray-600 mb-4">
                Escanea este código con tu aplicación de autenticación
              </p>
              <div className="bg-gray-50 p-3 rounded-lg">
                <p className="text-xs font-mono text-gray-700">
                  Código manual: ABCD-EFGH-IJKL-MNOP
                </p>
              </div>
            </div>
            
            <div className="flex space-x-3">
              <Button
                variant="outline"
                fullWidth
                onClick={() => setShowQRCode(false)}
              >
                Cancelar
              </Button>
              <Button
                variant="default"
                fullWidth
                onClick={handleConfirm2FA}
                iconName="Check"
                iconPosition="left"
              >
                Confirmar
              </Button>
            </div>
          </div>
        )}
      </div>
      {/* Active Sessions */}
      <div className="glass-card p-6 rounded-lg">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Sesiones Activas</h3>
        
        <div className="space-y-4">
          {activeSessions?.map((session) => (
            <div key={session?.id} className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
              <div className="flex items-center space-x-4">
                <div className="w-10 h-10 bg-primary/10 text-primary rounded-full flex items-center justify-center">
                  <Icon name={session?.device?.includes('iPhone') ? 'Smartphone' : 'Monitor'} size={20} />
                </div>
                <div>
                  <div className="flex items-center space-x-2">
                    <p className="font-medium text-gray-900">{session?.device}</p>
                    {session?.isCurrent && (
                      <StatusIndicatorBadges status="active" size="sm">
                        Actual
                      </StatusIndicatorBadges>
                    )}
                  </div>
                  <p className="text-sm text-gray-500">{session?.location}</p>
                  <p className="text-xs text-gray-400">
                    IP: {session?.ipAddress} • Última actividad: {formatDateTime(session?.lastActive)}
                  </p>
                </div>
              </div>
              
              {!session?.isCurrent && (
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => handleTerminateSession(session?.id)}
                  iconName="LogOut"
                  iconPosition="left"
                >
                  Terminar
                </Button>
              )}
            </div>
          ))}
        </div>
      </div>
      {/* Security Activity Log */}
      <div className="glass-card p-6 rounded-lg">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Registro de Actividad</h3>
        
        <div className="space-y-4">
          {securityEvents?.map((event) => (
            <div key={event?.id} className="flex items-center space-x-4 p-4 bg-gray-50 rounded-lg">
              <div className={`w-10 h-10 rounded-full flex items-center justify-center ${
                event?.status === 'success' ? 'bg-success/10 text-success' :
                event?.status === 'warning'? 'bg-warning/10 text-warning' : 'bg-gray-100 text-gray-600'
              }`}>
                <Icon name={getEventIcon(event?.type)} size={20} />
              </div>
              <div className="flex-1">
                <p className="font-medium text-gray-900">{event?.description}</p>
                <p className="text-sm text-gray-500">
                  {formatDateTime(event?.timestamp)} • IP: {event?.ipAddress}
                </p>
              </div>
              <StatusIndicatorBadges 
                status={event?.status === 'success' ? 'active' : event?.status === 'warning' ? 'pending' : 'inactive'}
                size="sm"
              />
            </div>
          ))}
        </div>
        
        <div className="mt-6 text-center">
          <Button variant="outline" size="sm">
            Ver Historial Completo
          </Button>
        </div>
      </div>
    </div>
  );
};

export default AccountSecurityTab;