import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Select from '../../../components/ui/Select';
import { Checkbox } from '../../../components/ui/Checkbox';
import Button from '../../../components/ui/Button';

const ApplicationSettingsTab = () => {
  const [settings, setSettings] = useState({
    language: 'es-CL',
    timezone: 'America/Santiago',
    currency: 'CLP',
    dateFormat: 'DD/MM/YYYY',
    theme: 'light',
    autoSave: true,
    compactView: false,
    showTutorials: true,
    enableAnimations: true,
    defaultView: 'grid',
    itemsPerPage: 25,
    enableNotificationSounds: true,
    autoLogout: 60
  });

  const [isSaving, setIsSaving] = useState(false);

  const languageOptions = [
    { value: 'es-CL', label: 'Español (Chile)' },
    { value: 'es-ES', label: 'Español (España)' },
    { value: 'en-US', label: 'English (US)' },
    { value: 'pt-BR', label: 'Português (Brasil)' }
  ];

  const timezoneOptions = [
    { value: 'America/Santiago', label: 'Santiago (GMT-3)' },
    { value: 'America/Argentina/Buenos_Aires', label: 'Buenos Aires (GMT-3)' },
    { value: 'America/Sao_Paulo', label: 'São Paulo (GMT-3)' },
    { value: 'America/Lima', label: 'Lima (GMT-5)' }
  ];

  const currencyOptions = [
    { value: 'CLP', label: 'Peso Chileno (CLP)' },
    { value: 'UF', label: 'Unidad de Fomento (UF)' },
    { value: 'USD', label: 'Dólar Americano (USD)' },
    { value: 'EUR', label: 'Euro (EUR)' }
  ];

  const dateFormatOptions = [
    { value: 'DD/MM/YYYY', label: '26/08/2025' },
    { value: 'MM/DD/YYYY', label: '08/26/2025' },
    { value: 'YYYY-MM-DD', label: '2025-08-26' },
    { value: 'DD-MM-YYYY', label: '26-08-2025' }
  ];

  const themeOptions = [
    { value: 'light', label: 'Claro' },
    { value: 'dark', label: 'Oscuro' },
    { value: 'auto', label: 'Automático' }
  ];

  const defaultViewOptions = [
    { value: 'grid', label: 'Vista de Cuadrícula' },
    { value: 'list', label: 'Vista de Lista' },
    { value: 'table', label: 'Vista de Tabla' }
  ];

  const itemsPerPageOptions = [
    { value: 10, label: '10 elementos' },
    { value: 25, label: '25 elementos' },
    { value: 50, label: '50 elementos' },
    { value: 100, label: '100 elementos' }
  ];

  const autoLogoutOptions = [
    { value: 15, label: '15 minutos' },
    { value: 30, label: '30 minutos' },
    { value: 60, label: '1 hora' },
    { value: 120, label: '2 horas' },
    { value: 0, label: 'Nunca' }
  ];

  const handleSettingChange = (key, value) => {
    setSettings(prev => ({
      ...prev,
      [key]: value
    }));
  };

  const handleSave = async () => {
    setIsSaving(true);
    
    // Simulate API call
    await new Promise(resolve => setTimeout(resolve, 1500));
    
    setIsSaving(false);
    console.log('Configuración de aplicación guardada');
  };

  const handleResetToDefaults = () => {
    setSettings({
      language: 'es-CL',
      timezone: 'America/Santiago',
      currency: 'CLP',
      dateFormat: 'DD/MM/YYYY',
      theme: 'light',
      autoSave: true,
      compactView: false,
      showTutorials: true,
      enableAnimations: true,
      defaultView: 'grid',
      itemsPerPage: 25,
      enableNotificationSounds: true,
      autoLogout: 60
    });
  };

  return (
    <div className="space-y-8">
      {/* Regional Settings */}
      <div className="glass-card p-6 rounded-lg">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Configuración Regional</h3>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <Select
            label="Idioma"
            description="Idioma de la interfaz de usuario"
            options={languageOptions}
            value={settings?.language}
            onChange={(value) => handleSettingChange('language', value)}
          />
          
          <Select
            label="Zona Horaria"
            description="Zona horaria para fechas y horarios"
            options={timezoneOptions}
            value={settings?.timezone}
            onChange={(value) => handleSettingChange('timezone', value)}
          />
          
          <Select
            label="Moneda Principal"
            description="Moneda por defecto para mostrar valores"
            options={currencyOptions}
            value={settings?.currency}
            onChange={(value) => handleSettingChange('currency', value)}
          />
          
          <Select
            label="Formato de Fecha"
            description="Formato para mostrar fechas"
            options={dateFormatOptions}
            value={settings?.dateFormat}
            onChange={(value) => handleSettingChange('dateFormat', value)}
          />
        </div>
      </div>
      {/* Appearance Settings */}
      <div className="glass-card p-6 rounded-lg">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Apariencia</h3>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <Select
            label="Tema"
            description="Esquema de colores de la aplicación"
            options={themeOptions}
            value={settings?.theme}
            onChange={(value) => handleSettingChange('theme', value)}
          />
          
          <Select
            label="Vista por Defecto"
            description="Vista predeterminada para listas"
            options={defaultViewOptions}
            value={settings?.defaultView}
            onChange={(value) => handleSettingChange('defaultView', value)}
          />
        </div>
        
        <div className="mt-6 space-y-4">
          <Checkbox
            label="Vista Compacta"
            description="Reduce el espaciado para mostrar más información"
            checked={settings?.compactView}
            onChange={(e) => handleSettingChange('compactView', e?.target?.checked)}
          />
          
          <Checkbox
            label="Habilitar Animaciones"
            description="Muestra transiciones y efectos visuales"
            checked={settings?.enableAnimations}
            onChange={(e) => handleSettingChange('enableAnimations', e?.target?.checked)}
          />
        </div>
      </div>
      {/* Behavior Settings */}
      <div className="glass-card p-6 rounded-lg">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Comportamiento</h3>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <Select
            label="Elementos por Página"
            description="Número de elementos a mostrar en listas"
            options={itemsPerPageOptions}
            value={settings?.itemsPerPage}
            onChange={(value) => handleSettingChange('itemsPerPage', value)}
          />
          
          <Select
            label="Cierre de Sesión Automático"
            description="Tiempo de inactividad antes del cierre automático"
            options={autoLogoutOptions}
            value={settings?.autoLogout}
            onChange={(value) => handleSettingChange('autoLogout', value)}
          />
        </div>
        
        <div className="mt-6 space-y-4">
          <Checkbox
            label="Guardado Automático"
            description="Guarda cambios automáticamente sin confirmación"
            checked={settings?.autoSave}
            onChange={(e) => handleSettingChange('autoSave', e?.target?.checked)}
          />
          
          <Checkbox
            label="Mostrar Tutoriales"
            description="Muestra ayudas y consejos para nuevas funcionalidades"
            checked={settings?.showTutorials}
            onChange={(e) => handleSettingChange('showTutorials', e?.target?.checked)}
          />
          
          <Checkbox
            label="Sonidos de Notificación"
            description="Reproduce sonidos para notificaciones importantes"
            checked={settings?.enableNotificationSounds}
            onChange={(e) => handleSettingChange('enableNotificationSounds', e?.target?.checked)}
          />
        </div>
      </div>
      {/* Data & Privacy */}
      <div className="glass-card p-6 rounded-lg">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Datos y Privacidad</h3>
        
        <div className="space-y-6">
          <div className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
            <div className="flex items-center space-x-3">
              <Icon name="Download" size={20} className="text-primary" />
              <div>
                <h4 className="font-medium text-gray-900">Exportar Datos</h4>
                <p className="text-sm text-gray-500">Descarga una copia de todos tus datos</p>
              </div>
            </div>
            <Button variant="outline" size="sm">
              Exportar
            </Button>
          </div>
          
          <div className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
            <div className="flex items-center space-x-3">
              <Icon name="Trash2" size={20} className="text-error" />
              <div>
                <h4 className="font-medium text-gray-900">Eliminar Cuenta</h4>
                <p className="text-sm text-gray-500">Elimina permanentemente tu cuenta y datos</p>
              </div>
            </div>
            <Button variant="destructive" size="sm">
              Eliminar
            </Button>
          </div>
        </div>
      </div>
      {/* Performance Settings */}
      <div className="glass-card p-6 rounded-lg">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Rendimiento</h3>
        
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
          <div className="text-center p-4 bg-gray-50 rounded-lg">
            <Icon name="Zap" size={24} className="text-success mx-auto mb-2" />
            <h4 className="font-medium text-gray-900">Velocidad</h4>
            <p className="text-2xl font-bold text-success">Rápida</p>
            <p className="text-xs text-gray-500">Tiempo de carga optimizado</p>
          </div>
          
          <div className="text-center p-4 bg-gray-50 rounded-lg">
            <Icon name="Database" size={24} className="text-primary mx-auto mb-2" />
            <h4 className="font-medium text-gray-900">Almacenamiento</h4>
            <p className="text-2xl font-bold text-primary">2.4 MB</p>
            <p className="text-xs text-gray-500">Datos locales utilizados</p>
          </div>
          
          <div className="text-center p-4 bg-gray-50 rounded-lg">
            <Icon name="Activity" size={24} className="text-warning mx-auto mb-2" />
            <h4 className="font-medium text-gray-900">Memoria</h4>
            <p className="text-2xl font-bold text-warning">45 MB</p>
            <p className="text-xs text-gray-500">Uso actual de memoria</p>
          </div>
        </div>
        
        <div className="mt-6 text-center">
          <Button variant="outline" size="sm" iconName="RotateCcw" iconPosition="left">
            Limpiar Caché
          </Button>
        </div>
      </div>
      {/* Action Buttons */}
      <div className="flex items-center justify-between pt-6 border-t border-gray-200">
        <Button
          variant="outline"
          onClick={handleResetToDefaults}
          iconName="RotateCcw"
          iconPosition="left"
        >
          Restaurar Valores por Defecto
        </Button>
        
        <Button
          variant="default"
          onClick={handleSave}
          loading={isSaving}
          iconName="Save"
          iconPosition="left"
        >
          {isSaving ? 'Guardando...' : 'Guardar Configuración'}
        </Button>
      </div>
    </div>
  );
};

export default ApplicationSettingsTab;