import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import { Checkbox } from '../../../components/ui/Checkbox';
import Button from '../../../components/ui/Button';
import Select from '../../../components/ui/Select';

const NotificationPreferencesTab = () => {
  const [preferences, setPreferences] = useState({
    email: {
      enabled: true,
      frequency: 'immediate',
      types: {
        payments: true,
        reminders: true,
        maintenance: true,
        reports: false,
        marketing: false
      }
    },
    sms: {
      enabled: true,
      frequency: 'urgent',
      types: {
        payments: true,
        reminders: false,
        maintenance: true,
        reports: false,
        marketing: false
      }
    },
    push: {
      enabled: true,
      frequency: 'immediate',
      types: {
        payments: true,
        reminders: true,
        maintenance: true,
        reports: true,
        marketing: false
      }
    },
    whatsapp: {
      enabled: false,
      frequency: 'daily',
      types: {
        payments: false,
        reminders: false,
        maintenance: false,
        reports: false,
        marketing: false
      }
    }
  });

  const [isSaving, setIsSaving] = useState(false);

  const frequencyOptions = [
    { value: 'immediate', label: 'Inmediato' },
    { value: 'hourly', label: 'Cada hora' },
    { value: 'daily', label: 'Diario' },
    { value: 'weekly', label: 'Semanal' },
    { value: 'urgent', label: 'Solo urgentes' }
  ];

  const notificationTypes = [
    {
      key: 'payments',
      label: 'Pagos y Facturación',
      description: 'Notificaciones sobre pagos recibidos, vencimientos y estados de facturación'
    },
    {
      key: 'reminders',
      label: 'Recordatorios',
      description: 'Recordatorios automáticos de vencimientos y tareas pendientes'
    },
    {
      key: 'maintenance',
      label: 'Mantenimiento',
      description: 'Solicitudes de mantenimiento y actualizaciones de estado'
    },
    {
      key: 'reports',
      label: 'Reportes',
      description: 'Reportes mensuales y análisis de rendimiento'
    },
    {
      key: 'marketing',
      label: 'Marketing y Promociones',
      description: 'Ofertas especiales, nuevas funcionalidades y contenido promocional'
    }
  ];

  const channels = [
    {
      key: 'email',
      label: 'Correo Electrónico',
      icon: 'Mail',
      description: 'Notificaciones enviadas a tu correo electrónico'
    },
    {
      key: 'sms',
      label: 'SMS',
      icon: 'MessageSquare',
      description: 'Mensajes de texto a tu teléfono móvil'
    },
    {
      key: 'push',
      label: 'Notificaciones Push',
      icon: 'Bell',
      description: 'Notificaciones en tiempo real en la aplicación'
    },
    {
      key: 'whatsapp',
      label: 'WhatsApp',
      icon: 'MessageCircle',
      description: 'Mensajes a través de WhatsApp Business'
    }
  ];

  const handleChannelToggle = (channel) => {
    setPreferences(prev => ({
      ...prev,
      [channel]: {
        ...prev?.[channel],
        enabled: !prev?.[channel]?.enabled
      }
    }));
  };

  const handleFrequencyChange = (channel, frequency) => {
    setPreferences(prev => ({
      ...prev,
      [channel]: {
        ...prev?.[channel],
        frequency
      }
    }));
  };

  const handleTypeToggle = (channel, type) => {
    setPreferences(prev => ({
      ...prev,
      [channel]: {
        ...prev?.[channel],
        types: {
          ...prev?.[channel]?.types,
          [type]: !prev?.[channel]?.types?.[type]
        }
      }
    }));
  };

  const handleSave = async () => {
    setIsSaving(true);
    
    // Simulate API call
    await new Promise(resolve => setTimeout(resolve, 1500));
    
    setIsSaving(false);
    console.log('Preferencias de notificación guardadas');
  };

  const handleTestNotification = (channel) => {
    console.log(`Enviando notificación de prueba por ${channel}`);
  };

  return (
    <div className="space-y-8">
      {/* Quick Settings */}
      <div className="glass-card p-6 rounded-lg">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Configuración Rápida</h3>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div className="p-4 bg-gray-50 rounded-lg">
            <div className="flex items-center space-x-3 mb-3">
              <Icon name="Zap" size={20} className="text-primary" />
              <h4 className="font-medium text-gray-900">Notificaciones Críticas</h4>
            </div>
            <p className="text-sm text-gray-600 mb-4">
              Solo pagos vencidos y emergencias de mantenimiento
            </p>
            <Button variant="outline" size="sm" fullWidth>
              Activar Modo Crítico
            </Button>
          </div>
          
          <div className="p-4 bg-gray-50 rounded-lg">
            <div className="flex items-center space-x-3 mb-3">
              <Icon name="Volume2" size={20} className="text-success" />
              <h4 className="font-medium text-gray-900">Todas las Notificaciones</h4>
            </div>
            <p className="text-sm text-gray-600 mb-4">
              Recibe todas las actualizaciones y recordatorios
            </p>
            <Button variant="outline" size="sm" fullWidth>
              Activar Todo
            </Button>
          </div>
        </div>
      </div>
      {/* Channel Configuration */}
      {channels?.map((channel) => (
        <div key={channel?.key} className="glass-card p-6 rounded-lg">
          <div className="flex items-center justify-between mb-6">
            <div className="flex items-center space-x-3">
              <div className="w-10 h-10 bg-primary/10 text-primary rounded-full flex items-center justify-center">
                <Icon name={channel?.icon} size={20} />
              </div>
              <div>
                <h3 className="text-lg font-semibold text-gray-900">{channel?.label}</h3>
                <p className="text-sm text-gray-500">{channel?.description}</p>
              </div>
            </div>
            
            <div className="flex items-center space-x-3">
              <Button
                variant="ghost"
                size="sm"
                onClick={() => handleTestNotification(channel?.key)}
                iconName="Send"
                iconPosition="left"
                disabled={!preferences?.[channel?.key]?.enabled}
              >
                Probar
              </Button>
              <Checkbox
                checked={preferences?.[channel?.key]?.enabled}
                onChange={() => handleChannelToggle(channel?.key)}
                label=""
              />
            </div>
          </div>

          {preferences?.[channel?.key]?.enabled && (
            <div className="space-y-6 pl-13">
              {/* Frequency Setting */}
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <Select
                  label="Frecuencia"
                  options={frequencyOptions}
                  value={preferences?.[channel?.key]?.frequency}
                  onChange={(value) => handleFrequencyChange(channel?.key, value)}
                />
              </div>

              {/* Notification Types */}
              <div>
                <h4 className="font-medium text-gray-900 mb-4">Tipos de Notificación</h4>
                <div className="space-y-4">
                  {notificationTypes?.map((type) => (
                    <div key={type?.key} className="flex items-start space-x-3 p-3 bg-gray-50 rounded-lg">
                      <Checkbox
                        checked={preferences?.[channel?.key]?.types?.[type?.key]}
                        onChange={() => handleTypeToggle(channel?.key, type?.key)}
                        className="mt-1"
                      />
                      <div className="flex-1">
                        <h5 className="font-medium text-gray-900">{type?.label}</h5>
                        <p className="text-sm text-gray-600">{type?.description}</p>
                      </div>
                    </div>
                  ))}
                </div>
              </div>
            </div>
          )}
        </div>
      ))}
      {/* Quiet Hours */}
      <div className="glass-card p-6 rounded-lg">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Horarios de Silencio</h3>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Desde
            </label>
            <input
              type="time"
              defaultValue="22:00"
              className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent"
            />
          </div>
          
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Hasta
            </label>
            <input
              type="time"
              defaultValue="08:00"
              className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-primary focus:border-transparent"
            />
          </div>
        </div>
        
        <div className="mt-4">
          <Checkbox
            label="Aplicar horarios de silencio"
            description="Las notificaciones no críticas se pausarán durante estos horarios"
            checked
          />
        </div>
      </div>
      {/* Save Button */}
      <div className="flex justify-end pt-6 border-t border-gray-200">
        <Button
          variant="default"
          onClick={handleSave}
          loading={isSaving}
          iconName="Save"
          iconPosition="left"
        >
          {isSaving ? 'Guardando...' : 'Guardar Preferencias'}
        </Button>
      </div>
    </div>
  );
};

export default NotificationPreferencesTab;