import React, { useState, useRef } from 'react';
import Icon from '../../../components/AppIcon';
import Image from '../../../components/AppImage';
import Input from '../../../components/ui/Input';
import Button from '../../../components/ui/Button';

const ProfileInformationTab = () => {
  const fileInputRef = useRef(null);
  const [isEditing, setIsEditing] = useState(false);
  const [isSaving, setIsSaving] = useState(false);
  const [profileData, setProfileData] = useState({
    firstName: 'María',
    lastName: 'González',
    email: 'maria.gonzalez@propmanager.cl',
    phone: '+56 9 8765 4321',
    position: 'Administradora de Propiedades',
    company: 'PropManager Chile',
    address: 'Av. Providencia 1234, Providencia',
    city: 'Santiago',
    region: 'Región Metropolitana',
    postalCode: '7500000',
    avatar: null
  });

  const [originalData, setOriginalData] = useState({ ...profileData });

  const handleInputChange = (field, value) => {
    setProfileData(prev => ({
      ...prev,
      [field]: value
    }));
  };

  const handleSave = async () => {
    setIsSaving(true);
    
    // Simulate API call
    await new Promise(resolve => setTimeout(resolve, 1500));
    
    setOriginalData({ ...profileData });
    setIsEditing(false);
    setIsSaving(false);
    
    // Show success notification
    console.log('Perfil actualizado exitosamente');
  };

  const handleCancel = () => {
    setProfileData({ ...originalData });
    setIsEditing(false);
  };

  const handleAvatarChange = (event) => {
    const file = event?.target?.files?.[0];
    if (file) {
      const reader = new FileReader();
      reader.onload = (e) => {
        setProfileData(prev => ({
          ...prev,
          avatar: e?.target?.result
        }));
        setIsEditing(true);
      };
      reader?.readAsDataURL(file);
    }
  };

  const hasChanges = JSON.stringify(profileData) !== JSON.stringify(originalData);

  return (
    <div className="space-y-8">
      {/* Profile Photo Section */}
      <div className="glass-card p-6 rounded-lg">
        <h3 className="text-lg font-semibold text-gray-900 mb-4">Foto de Perfil</h3>
        <div className="flex items-center space-x-6">
          <div className="relative">
            <div className="w-24 h-24 rounded-full overflow-hidden bg-gray-100 border-4 border-white shadow-lg">
              {profileData?.avatar ? (
                <Image 
                  src={profileData?.avatar} 
                  alt="Foto de perfil" 
                  className="w-full h-full object-cover"
                />
              ) : (
                <div className="w-full h-full bg-primary text-white flex items-center justify-center text-2xl font-semibold">
                  {profileData?.firstName?.charAt(0)}{profileData?.lastName?.charAt(0)}
                </div>
              )}
            </div>
            <button
              onClick={() => fileInputRef?.current?.click()}
              className="absolute -bottom-2 -right-2 w-8 h-8 bg-primary text-white rounded-full flex items-center justify-center hover:bg-primary/90 transition-colors duration-150 shadow-lg"
            >
              <Icon name="Camera" size={16} />
            </button>
          </div>
          
          <div className="flex-1">
            <h4 className="font-medium text-gray-900">{profileData?.firstName} {profileData?.lastName}</h4>
            <p className="text-sm text-gray-500 mb-3">{profileData?.position}</p>
            <div className="flex space-x-3">
              <Button
                variant="outline"
                size="sm"
                iconName="Upload"
                iconPosition="left"
                onClick={() => fileInputRef?.current?.click()}
              >
                Cambiar Foto
              </Button>
              {profileData?.avatar && (
                <Button
                  variant="ghost"
                  size="sm"
                  iconName="Trash2"
                  iconPosition="left"
                  onClick={() => handleInputChange('avatar', null)}
                >
                  Eliminar
                </Button>
              )}
            </div>
          </div>
        </div>
        
        <input
          ref={fileInputRef}
          type="file"
          accept="image/*"
          onChange={handleAvatarChange}
          className="hidden"
        />
      </div>
      {/* Personal Information */}
      <div className="glass-card p-6 rounded-lg">
        <div className="flex items-center justify-between mb-6">
          <h3 className="text-lg font-semibold text-gray-900">Información Personal</h3>
          {!isEditing && (
            <Button
              variant="outline"
              size="sm"
              iconName="Edit"
              iconPosition="left"
              onClick={() => setIsEditing(true)}
            >
              Editar
            </Button>
          )}
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <Input
            label="Nombre"
            type="text"
            value={profileData?.firstName}
            onChange={(e) => handleInputChange('firstName', e?.target?.value)}
            disabled={!isEditing}
            required
          />
          
          <Input
            label="Apellido"
            type="text"
            value={profileData?.lastName}
            onChange={(e) => handleInputChange('lastName', e?.target?.value)}
            disabled={!isEditing}
            required
          />
          
          <Input
            label="Correo Electrónico"
            type="email"
            value={profileData?.email}
            onChange={(e) => handleInputChange('email', e?.target?.value)}
            disabled={!isEditing}
            description="Este será tu nombre de usuario para iniciar sesión"
            required
          />
          
          <Input
            label="Teléfono"
            type="tel"
            value={profileData?.phone}
            onChange={(e) => handleInputChange('phone', e?.target?.value)}
            disabled={!isEditing}
            placeholder="+56 9 XXXX XXXX"
          />
          
          <Input
            label="Cargo"
            type="text"
            value={profileData?.position}
            onChange={(e) => handleInputChange('position', e?.target?.value)}
            disabled={!isEditing}
          />
          
          <Input
            label="Empresa"
            type="text"
            value={profileData?.company}
            onChange={(e) => handleInputChange('company', e?.target?.value)}
            disabled={!isEditing}
          />
        </div>
      </div>
      {/* Address Information */}
      <div className="glass-card p-6 rounded-lg">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Dirección</h3>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div className="md:col-span-2">
            <Input
              label="Dirección"
              type="text"
              value={profileData?.address}
              onChange={(e) => handleInputChange('address', e?.target?.value)}
              disabled={!isEditing}
              placeholder="Calle, número, departamento"
            />
          </div>
          
          <Input
            label="Ciudad"
            type="text"
            value={profileData?.city}
            onChange={(e) => handleInputChange('city', e?.target?.value)}
            disabled={!isEditing}
          />
          
          <Input
            label="Región"
            type="text"
            value={profileData?.region}
            onChange={(e) => handleInputChange('region', e?.target?.value)}
            disabled={!isEditing}
          />
          
          <Input
            label="Código Postal"
            type="text"
            value={profileData?.postalCode}
            onChange={(e) => handleInputChange('postalCode', e?.target?.value)}
            disabled={!isEditing}
            placeholder="1234567"
          />
        </div>
      </div>
      {/* Action Buttons */}
      {isEditing && (
        <div className="flex items-center justify-end space-x-4 pt-6 border-t border-gray-200">
          <Button
            variant="outline"
            onClick={handleCancel}
            disabled={isSaving}
          >
            Cancelar
          </Button>
          <Button
            variant="default"
            onClick={handleSave}
            loading={isSaving}
            disabled={!hasChanges}
            iconName="Save"
            iconPosition="left"
          >
            {isSaving ? 'Guardando...' : 'Guardar Cambios'}
          </Button>
        </div>
      )}
    </div>
  );
};

export default ProfileInformationTab;