import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';
import { Checkbox } from '../../../components/ui/Checkbox';
import Select from '../../../components/ui/Select';
import StatusIndicatorBadges from '../../../components/ui/StatusIndicatorBadges';

const SystemConfigurationTab = () => {
  const [systemSettings, setSystemSettings] = useState({
    autoBackup: true,
    backupFrequency: 'daily',
    dataRetention: 365,
    apiAccess: false,
    debugMode: false,
    maintenanceMode: false,
    systemNotifications: true,
    performanceMonitoring: true
  });

  const [backupStatus, setBackupStatus] = useState({
    lastBackup: '2025-08-26 03:00:00',
    nextBackup: '2025-08-27 03:00:00',
    backupSize: '2.4 GB',
    status: 'success'
  });

  const [systemHealth, setSystemHealth] = useState({
    database: 'healthy',
    storage: 'healthy',
    memory: 'warning',
    network: 'healthy'
  });

  const [isExporting, setIsExporting] = useState(false);
  const [isSaving, setIsSaving] = useState(false);

  const backupFrequencyOptions = [
    { value: 'hourly', label: 'Cada hora' },
    { value: 'daily', label: 'Diario' },
    { value: 'weekly', label: 'Semanal' },
    { value: 'monthly', label: 'Mensual' }
  ];

  const dataRetentionOptions = [
    { value: 30, label: '30 días' },
    { value: 90, label: '90 días' },
    { value: 180, label: '6 meses' },
    { value: 365, label: '1 año' },
    { value: 730, label: '2 años' },
    { value: -1, label: 'Permanente' }
  ];

  const userPermissions = [
    {
      id: 1,
      name: 'María González',
      email: 'maria.gonzalez@propmanager.cl',
      role: 'Administrador',
      permissions: ['read', 'write', 'delete', 'admin'],
      lastAccess: '2025-08-26 21:20:00',
      status: 'active'
    },
    {
      id: 2,
      name: 'Carlos Rodríguez',
      email: 'carlos.rodriguez@propmanager.cl',
      role: 'Editor',
      permissions: ['read', 'write'],
      lastAccess: '2025-08-26 18:45:00',
      status: 'active'
    },
    {
      id: 3,
      name: 'Ana Martínez',
      email: 'ana.martinez@propmanager.cl',
      role: 'Visualizador',
      permissions: ['read'],
      lastAccess: '2025-08-25 14:30:00',
      status: 'inactive'
    }
  ];

  const integrations = [
    {
      name: 'API de Servicios Básicos',
      description: 'Integración con proveedores de electricidad, agua y gas',
      status: 'connected',
      lastSync: '2025-08-26 20:15:00'
    },
    {
      name: 'Sistema de Pagos',
      description: 'Procesamiento de pagos y transacciones',
      status: 'connected',
      lastSync: '2025-08-26 21:00:00'
    },
    {
      name: 'Servicio de SMS',
      description: 'Envío de notificaciones por mensaje de texto',
      status: 'warning',
      lastSync: '2025-08-26 12:30:00'
    },
    {
      name: 'WhatsApp Business',
      description: 'Comunicación a través de WhatsApp',
      status: 'inactive',
      lastSync: null
    }
  ];

  const handleSettingChange = (key, value) => {
    setSystemSettings(prev => ({
      ...prev,
      [key]: value
    }));
  };

  const handleExportData = async (format) => {
    setIsExporting(true);
    
    // Simulate export process
    await new Promise(resolve => setTimeout(resolve, 3000));
    
    setIsExporting(false);
    console.log(`Exportando datos en formato ${format}`);
  };

  const handleBackupNow = async () => {
    console.log('Iniciando respaldo manual');
    setBackupStatus(prev => ({
      ...prev,
      status: 'pending'
    }));
    
    // Simulate backup process
    await new Promise(resolve => setTimeout(resolve, 2000));
    
    setBackupStatus(prev => ({
      ...prev,
      lastBackup: new Date()?.toISOString()?.slice(0, 19)?.replace('T', ' '),
      status: 'success'
    }));
  };

  const handleSave = async () => {
    setIsSaving(true);
    
    // Simulate API call
    await new Promise(resolve => setTimeout(resolve, 1500));
    
    setIsSaving(false);
    console.log('Configuración del sistema guardada');
  };

  const formatDateTime = (dateString) => {
    if (!dateString) return 'Nunca';
    const date = new Date(dateString);
    return date?.toLocaleString('es-CL', {
      day: '2-digit',
      month: '2-digit',
      year: 'numeric',
      hour: '2-digit',
      minute: '2-digit'
    });
  };

  const getHealthIcon = (status) => {
    switch (status) {
      case 'healthy': return 'CheckCircle';
      case 'warning': return 'AlertTriangle';
      case 'error': return 'XCircle';
      default: return 'Circle';
    }
  };

  const getHealthColor = (status) => {
    switch (status) {
      case 'healthy': return 'text-success';
      case 'warning': return 'text-warning';
      case 'error': return 'text-error';
      default: return 'text-gray-500';
    }
  };

  return (
    <div className="space-y-8">
      {/* System Health */}
      <div className="glass-card p-6 rounded-lg">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Estado del Sistema</h3>
        
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
          {Object.entries(systemHealth)?.map(([component, status]) => (
            <div key={component} className="p-4 bg-gray-50 rounded-lg text-center">
              <Icon 
                name={getHealthIcon(status)} 
                size={24} 
                className={`mx-auto mb-2 ${getHealthColor(status)}`}
              />
              <h4 className="font-medium text-gray-900 capitalize">{component}</h4>
              <StatusIndicatorBadges status={status} size="sm" className="mt-2" />
            </div>
          ))}
        </div>
      </div>
      {/* Backup Configuration */}
      <div className="glass-card p-6 rounded-lg">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Configuración de Respaldos</h3>
        
        <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
          <div className="space-y-6">
            <Checkbox
              label="Respaldo Automático"
              description="Crear respaldos automáticos del sistema"
              checked={systemSettings?.autoBackup}
              onChange={(e) => handleSettingChange('autoBackup', e?.target?.checked)}
            />
            
            {systemSettings?.autoBackup && (
              <>
                <Select
                  label="Frecuencia de Respaldo"
                  options={backupFrequencyOptions}
                  value={systemSettings?.backupFrequency}
                  onChange={(value) => handleSettingChange('backupFrequency', value)}
                />
                
                <Select
                  label="Retención de Datos"
                  description="Tiempo que se conservan los respaldos"
                  options={dataRetentionOptions}
                  value={systemSettings?.dataRetention}
                  onChange={(value) => handleSettingChange('dataRetention', value)}
                />
              </>
            )}
          </div>
          
          <div className="p-4 bg-gray-50 rounded-lg">
            <h4 className="font-medium text-gray-900 mb-4">Estado del Último Respaldo</h4>
            <div className="space-y-3">
              <div className="flex justify-between">
                <span className="text-sm text-gray-600">Último respaldo:</span>
                <span className="text-sm font-medium">{formatDateTime(backupStatus?.lastBackup)}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-sm text-gray-600">Próximo respaldo:</span>
                <span className="text-sm font-medium">{formatDateTime(backupStatus?.nextBackup)}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-sm text-gray-600">Tamaño:</span>
                <span className="text-sm font-medium">{backupStatus?.backupSize}</span>
              </div>
              <div className="flex justify-between items-center">
                <span className="text-sm text-gray-600">Estado:</span>
                <StatusIndicatorBadges status={backupStatus?.status} size="sm" />
              </div>
            </div>
            
            <Button
              variant="outline"
              size="sm"
              fullWidth
              className="mt-4"
              onClick={handleBackupNow}
              iconName="Download"
              iconPosition="left"
            >
              Respaldar Ahora
            </Button>
          </div>
        </div>
      </div>
      {/* Data Export */}
      <div className="glass-card p-6 rounded-lg">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Exportación de Datos</h3>
        
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <div className="p-4 bg-gray-50 rounded-lg text-center">
            <Icon name="FileText" size={24} className="text-primary mx-auto mb-2" />
            <h4 className="font-medium text-gray-900">Excel/CSV</h4>
            <p className="text-sm text-gray-500 mb-3">Datos tabulares para análisis</p>
            <Button
              variant="outline"
              size="sm"
              fullWidth
              loading={isExporting}
              onClick={() => handleExportData('excel')}
            >
              Exportar
            </Button>
          </div>
          
          <div className="p-4 bg-gray-50 rounded-lg text-center">
            <Icon name="FileJson" size={24} className="text-secondary mx-auto mb-2" />
            <h4 className="font-medium text-gray-900">JSON</h4>
            <p className="text-sm text-gray-500 mb-3">Formato estructurado para APIs</p>
            <Button
              variant="outline"
              size="sm"
              fullWidth
              loading={isExporting}
              onClick={() => handleExportData('json')}
            >
              Exportar
            </Button>
          </div>
          
          <div className="p-4 bg-gray-50 rounded-lg text-center">
            <Icon name="FileImage" size={24} className="text-success mx-auto mb-2" />
            <h4 className="font-medium text-gray-900">PDF</h4>
            <p className="text-sm text-gray-500 mb-3">Reportes formateados</p>
            <Button
              variant="outline"
              size="sm"
              fullWidth
              loading={isExporting}
              onClick={() => handleExportData('pdf')}
            >
              Exportar
            </Button>
          </div>
        </div>
      </div>
      {/* User Permissions */}
      <div className="glass-card p-6 rounded-lg">
        <div className="flex items-center justify-between mb-6">
          <h3 className="text-lg font-semibold text-gray-900">Gestión de Usuarios</h3>
          <Button variant="outline" size="sm" iconName="UserPlus" iconPosition="left">
            Invitar Usuario
          </Button>
        </div>
        
        <div className="space-y-4">
          {userPermissions?.map((user) => (
            <div key={user?.id} className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
              <div className="flex items-center space-x-4">
                <div className="w-10 h-10 bg-primary text-white rounded-full flex items-center justify-center text-sm font-medium">
                  {user?.name?.split(' ')?.map(n => n?.[0])?.join('')}
                </div>
                <div>
                  <h4 className="font-medium text-gray-900">{user?.name}</h4>
                  <p className="text-sm text-gray-500">{user?.email}</p>
                  <p className="text-xs text-gray-400">
                    Último acceso: {formatDateTime(user?.lastAccess)}
                  </p>
                </div>
              </div>
              
              <div className="flex items-center space-x-3">
                <StatusIndicatorBadges status={user?.status} size="sm" />
                <span className="text-sm font-medium text-gray-700">{user?.role}</span>
                <Button variant="ghost" size="sm" iconName="Settings">
                  Editar
                </Button>
              </div>
            </div>
          ))}
        </div>
      </div>
      {/* System Integrations */}
      <div className="glass-card p-6 rounded-lg">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Integraciones del Sistema</h3>
        
        <div className="space-y-4">
          {integrations?.map((integration, index) => (
            <div key={index} className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
              <div className="flex items-center space-x-4">
                <div className="w-10 h-10 bg-primary/10 text-primary rounded-full flex items-center justify-center">
                  <Icon name="Plug" size={20} />
                </div>
                <div>
                  <h4 className="font-medium text-gray-900">{integration?.name}</h4>
                  <p className="text-sm text-gray-500">{integration?.description}</p>
                  {integration?.lastSync && (
                    <p className="text-xs text-gray-400">
                      Última sincronización: {formatDateTime(integration?.lastSync)}
                    </p>
                  )}
                </div>
              </div>
              
              <div className="flex items-center space-x-3">
                <StatusIndicatorBadges 
                  status={integration?.status === 'connected' ? 'active' : integration?.status} 
                  size="sm" 
                />
                <Button variant="outline" size="sm">
                  {integration?.status === 'connected' ? 'Configurar' : 'Conectar'}
                </Button>
              </div>
            </div>
          ))}
        </div>
      </div>
      {/* Advanced Settings */}
      <div className="glass-card p-6 rounded-lg">
        <h3 className="text-lg font-semibold text-gray-900 mb-6">Configuración Avanzada</h3>
        
        <div className="space-y-4">
          <Checkbox
            label="Acceso a API"
            description="Permitir acceso programático a través de API REST"
            checked={systemSettings?.apiAccess}
            onChange={(e) => handleSettingChange('apiAccess', e?.target?.checked)}
          />
          
          <Checkbox
            label="Modo de Depuración"
            description="Habilitar logs detallados para diagnóstico"
            checked={systemSettings?.debugMode}
            onChange={(e) => handleSettingChange('debugMode', e?.target?.checked)}
          />
          
          <Checkbox
            label="Notificaciones del Sistema"
            description="Recibir alertas sobre el estado del sistema"
            checked={systemSettings?.systemNotifications}
            onChange={(e) => handleSettingChange('systemNotifications', e?.target?.checked)}
          />
          
          <Checkbox
            label="Monitoreo de Rendimiento"
            description="Recopilar métricas de rendimiento del sistema"
            checked={systemSettings?.performanceMonitoring}
            onChange={(e) => handleSettingChange('performanceMonitoring', e?.target?.checked)}
          />
        </div>
      </div>
      {/* Save Button */}
      <div className="flex justify-end pt-6 border-t border-gray-200">
        <Button
          variant="default"
          onClick={handleSave}
          loading={isSaving}
          iconName="Save"
          iconPosition="left"
        >
          {isSaving ? 'Guardando...' : 'Guardar Configuración'}
        </Button>
      </div>
    </div>
  );
};

export default SystemConfigurationTab;