import React, { useState, useEffect } from 'react';
import Header from '../../components/ui/Header';
import Sidebar from '../../components/ui/Sidebar';
import NavigationBreadcrumbs from '../../components/ui/NavigationBreadcrumbs';
import QuickActionFloatingButton from '../../components/ui/QuickActionFloatingButton';
import Icon from '../../components/AppIcon';
import ProfileInformationTab from './components/ProfileInformationTab';
import AccountSecurityTab from './components/AccountSecurityTab';
import NotificationPreferencesTab from './components/NotificationPreferencesTab';
import ApplicationSettingsTab from './components/ApplicationSettingsTab';
import SystemConfigurationTab from './components/SystemConfigurationTab';

const UserProfileSettings = () => {
  const [isSidebarOpen, setIsSidebarOpen] = useState(false);
  const [activeTab, setActiveTab] = useState('profile');
  const [isMobile, setIsMobile] = useState(false);

  useEffect(() => {
    const checkMobile = () => {
      const mobile = window.innerWidth < 1024;
      setIsMobile(mobile);
      if (!mobile) {
        setIsSidebarOpen(false);
      }
    };
    
    checkMobile();
    window.addEventListener('resize', checkMobile);
    return () => window.removeEventListener('resize', checkMobile);
  }, []);

  const tabs = [
    {
      id: 'profile',
      label: 'Perfil',
      icon: 'User',
      description: 'Información personal y foto de perfil'
    },
    {
      id: 'security',
      label: 'Seguridad',
      icon: 'Shield',
      description: 'Contraseña y autenticación'
    },
    {
      id: 'notifications',
      label: 'Notificaciones',
      icon: 'Bell',
      description: 'Preferencias de comunicación'
    },
    {
      id: 'application',
      label: 'Aplicación',
      icon: 'Settings',
      description: 'Configuración de la interfaz'
    },
    {
      id: 'system',
      label: 'Sistema',
      icon: 'Server',
      description: 'Configuración avanzada'
    }
  ];

  const handleTabChange = (tabId) => {
    setActiveTab(tabId);
    if (isMobile) {
      // Scroll to top on mobile when changing tabs
      window.scrollTo({ top: 0, behavior: 'smooth' });
    }
  };

  const renderTabContent = () => {
    switch (activeTab) {
      case 'profile':
        return <ProfileInformationTab />;
      case 'security':
        return <AccountSecurityTab />;
      case 'notifications':
        return <NotificationPreferencesTab />;
      case 'application':
        return <ApplicationSettingsTab />;
      case 'system':
        return <SystemConfigurationTab />;
      default:
        return <ProfileInformationTab />;
    }
  };

  const customBreadcrumbs = [
    { label: 'Inicio', path: '/dashboard', icon: 'Home' },
    { label: 'Perfil y Configuración', path: '/user-profile-settings', icon: 'Settings', isLast: true }
  ];

  return (
    <div className="min-h-screen bg-background">
      <Header 
        onMenuToggle={() => setIsSidebarOpen(!isSidebarOpen)}
        isMenuOpen={isSidebarOpen}
      />
      <Sidebar 
        isOpen={isSidebarOpen} 
        onClose={() => setIsSidebarOpen(false)} 
      />
      <main className="lg:ml-60 pt-16">
        <div className="p-6">
          <NavigationBreadcrumbs customBreadcrumbs={customBreadcrumbs} />
          
          {/* Page Header */}
          <div className="mb-8">
            <div className="flex items-center space-x-3 mb-2">
              <div className="w-10 h-10 bg-primary/10 text-primary rounded-full flex items-center justify-center">
                <Icon name="Settings" size={24} />
              </div>
              <div>
                <h1 className="text-2xl font-bold text-gray-900">Perfil y Configuración</h1>
                <p className="text-gray-600">Gestiona tu cuenta y personaliza la aplicación</p>
              </div>
            </div>
          </div>

          <div className="grid grid-cols-1 lg:grid-cols-4 gap-8">
            {/* Settings Navigation - Desktop Sidebar */}
            <div className="hidden lg:block">
              <div className="glass-card p-4 rounded-lg sticky top-24">
                <nav className="space-y-2">
                  {tabs?.map((tab) => (
                    <button
                      key={tab?.id}
                      onClick={() => handleTabChange(tab?.id)}
                      className={`
                        w-full flex items-center space-x-3 px-3 py-2.5 rounded-lg text-left
                        transition-all duration-150 hover-lift
                        ${activeTab === tab?.id
                          ? 'bg-primary text-white shadow-sm'
                          : 'text-gray-700 hover:bg-gray-100 hover:text-gray-900'
                        }
                      `}
                    >
                      <Icon 
                        name={tab?.icon} 
                        size={18} 
                        className={activeTab === tab?.id ? 'text-white' : 'text-gray-500'}
                      />
                      <div className="flex-1 min-w-0">
                        <p className="font-medium truncate">{tab?.label}</p>
                        <p className={`text-xs truncate ${
                          activeTab === tab?.id ? 'text-white/80' : 'text-gray-500'
                        }`}>
                          {tab?.description}
                        </p>
                      </div>
                    </button>
                  ))}
                </nav>
              </div>
            </div>

            {/* Settings Navigation - Mobile Tabs */}
            <div className="lg:hidden col-span-1">
              <div className="glass-card p-2 rounded-lg mb-6">
                <div className="flex overflow-x-auto space-x-1 pb-2">
                  {tabs?.map((tab) => (
                    <button
                      key={tab?.id}
                      onClick={() => handleTabChange(tab?.id)}
                      className={`
                        flex-shrink-0 flex flex-col items-center space-y-1 px-3 py-2 rounded-lg
                        transition-all duration-150 min-w-20
                        ${activeTab === tab?.id
                          ? 'bg-primary text-white' :'text-gray-600 hover:bg-gray-100'
                        }
                      `}
                    >
                      <Icon 
                        name={tab?.icon} 
                        size={16} 
                        className={activeTab === tab?.id ? 'text-white' : 'text-gray-500'}
                      />
                      <span className="text-xs font-medium">{tab?.label}</span>
                    </button>
                  ))}
                </div>
              </div>
            </div>

            {/* Main Content Area */}
            <div className="lg:col-span-3">
              <div className="space-y-6">
                {/* Active Tab Header - Mobile Only */}
                <div className="lg:hidden glass-card p-4 rounded-lg">
                  <div className="flex items-center space-x-3">
                    <div className="w-8 h-8 bg-primary/10 text-primary rounded-full flex items-center justify-center">
                      <Icon name={tabs?.find(tab => tab?.id === activeTab)?.icon} size={16} />
                    </div>
                    <div>
                      <h2 className="text-lg font-semibold text-gray-900">
                        {tabs?.find(tab => tab?.id === activeTab)?.label}
                      </h2>
                      <p className="text-sm text-gray-600">
                        {tabs?.find(tab => tab?.id === activeTab)?.description}
                      </p>
                    </div>
                  </div>
                </div>

                {/* Tab Content */}
                <div className="min-h-96">
                  {renderTabContent()}
                </div>
              </div>
            </div>
          </div>
        </div>
      </main>
      <QuickActionFloatingButton />
    </div>
  );
};

export default UserProfileSettings;