import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';

const BulkActionsBar = ({ selectedCount, onBulkAction, onClearSelection }) => {
  const [isExpanded, setIsExpanded] = useState(false);

  if (selectedCount === 0) return null;

  const bulkActions = [
    { 
      key: 'send-all', 
      label: 'Enviar comprobantes', 
      icon: 'Send', 
      variant: 'default',
      description: 'Enviar por email a todos los inquilinos seleccionados'
    },
    { 
      key: 'mark-paid-all', 
      label: 'Marcar como pagados', 
      icon: 'CheckCircle', 
      variant: 'success',
      description: 'Cambiar estado a pagado para todos los seleccionados'
    },
    { 
      key: 'apply-discount-all', 
      label: 'Aplicar descuento', 
      icon: 'Percent', 
      variant: 'secondary',
      description: 'Aplicar descuento porcentual o fijo'
    },
    { 
      key: 'generate-report', 
      label: 'Generar reporte', 
      icon: 'BarChart3', 
      variant: 'outline',
      description: 'Crear reporte de los comprobantes seleccionados'
    },
    { 
      key: 'export-pdf', 
      label: 'Exportar PDF', 
      icon: 'Download', 
      variant: 'outline',
      description: 'Descargar PDFs de todos los comprobantes'
    },
    { 
      key: 'delete-all', 
      label: 'Eliminar', 
      icon: 'Trash2', 
      variant: 'destructive',
      description: 'Eliminar permanentemente los comprobantes seleccionados'
    }
  ];

  const primaryActions = bulkActions?.slice(0, 3);
  const secondaryActions = bulkActions?.slice(3);

  return (
    <div className="fixed bottom-6 left-1/2 transform -translate-x-1/2 z-50">
      <div className="bg-white rounded-lg shadow-lg border border-gray-200 p-4 min-w-96">
        {/* Header */}
        <div className="flex items-center justify-between mb-4">
          <div className="flex items-center space-x-3">
            <div className="w-8 h-8 bg-primary text-white rounded-full flex items-center justify-center">
              <Icon name="CheckSquare" size={16} />
            </div>
            <div>
              <p className="text-sm font-medium text-gray-900">
                {selectedCount} comprobante{selectedCount !== 1 ? 's' : ''} seleccionado{selectedCount !== 1 ? 's' : ''}
              </p>
              <p className="text-xs text-gray-500">Selecciona una acción para continuar</p>
            </div>
          </div>
          <Button
            variant="ghost"
            size="icon"
            onClick={onClearSelection}
          >
            <Icon name="X" size={16} />
          </Button>
        </div>

        {/* Primary Actions */}
        <div className="flex flex-wrap gap-2 mb-3">
          {primaryActions?.map((action) => (
            <Button
              key={action?.key}
              variant={action?.variant}
              size="sm"
              onClick={() => onBulkAction(action?.key)}
              iconName={action?.icon}
              iconPosition="left"
            >
              {action?.label}
            </Button>
          ))}
        </div>

        {/* Secondary Actions Toggle */}
        <div className="border-t border-gray-200 pt-3">
          <Button
            variant="ghost"
            size="sm"
            fullWidth
            onClick={() => setIsExpanded(!isExpanded)}
            iconName={isExpanded ? "ChevronUp" : "ChevronDown"}
            iconPosition="right"
            className="justify-between"
          >
            Más acciones
          </Button>

          {isExpanded && (
            <div className="mt-3 space-y-2">
              {secondaryActions?.map((action) => (
                <button
                  key={action?.key}
                  onClick={() => onBulkAction(action?.key)}
                  className={`
                    w-full flex items-center justify-between p-3 rounded-lg border
                    transition-all duration-150 hover:bg-gray-50
                    ${action?.variant === 'destructive' ?'border-red-200 hover:border-red-300 text-error' :'border-gray-200 hover:border-gray-300 text-gray-700'
                    }
                  `}
                >
                  <div className="flex items-center space-x-3">
                    <Icon 
                      name={action?.icon} 
                      size={16} 
                      className={action?.variant === 'destructive' ? 'text-error' : 'text-gray-500'}
                    />
                    <div className="text-left">
                      <p className="text-sm font-medium">{action?.label}</p>
                      <p className="text-xs text-gray-500">{action?.description}</p>
                    </div>
                  </div>
                  <Icon name="ChevronRight" size={14} className="text-gray-400" />
                </button>
              ))}
            </div>
          )}
        </div>

        {/* Quick Stats */}
        <div className="mt-4 pt-3 border-t border-gray-200">
          <div className="grid grid-cols-3 gap-4 text-center">
            <div>
              <p className="text-xs text-gray-500">Total</p>
              <p className="text-sm font-medium text-gray-900">{selectedCount}</p>
            </div>
            <div>
              <p className="text-xs text-gray-500">Pendientes</p>
              <p className="text-sm font-medium text-warning">
                {Math.floor(selectedCount * 0.6)}
              </p>
            </div>
            <div>
              <p className="text-xs text-gray-500">Vencidos</p>
              <p className="text-sm font-medium text-error">
                {Math.floor(selectedCount * 0.2)}
              </p>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default BulkActionsBar;