import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';
import Button from '../../../components/ui/Button';

const VoucherFiltersPanel = ({ 
  filters, 
  onFiltersChange, 
  onClearFilters, 
  isCollapsed, 
  onToggleCollapse 
}) => {
  const statusOptions = [
    { value: '', label: 'Todos los estados' },
    { value: 'generated', label: 'Generado' },
    { value: 'sent', label: 'Enviado' },
    { value: 'paid', label: 'Pagado' },
    { value: 'expired', label: 'Vencido' }
  ];

  const propertyOptions = [
    { value: '', label: 'Todas las propiedades' },
    { value: 'prop-001', label: 'Departamento Las Condes #301' },
    { value: 'prop-002', label: 'Casa Providencia #45' },
    { value: 'prop-003', label: 'Oficina Ñuñoa #12' },
    { value: 'prop-004', label: 'Local Comercial Vitacura #8' },
    { value: 'prop-005', label: 'Departamento Santiago Centro #205' }
  ];

  const monthOptions = [
    { value: '', label: 'Todos los meses' },
    { value: '2024-01', label: 'Enero 2024' },
    { value: '2024-02', label: 'Febrero 2024' },
    { value: '2024-03', label: 'Marzo 2024' },
    { value: '2024-04', label: 'Abril 2024' },
    { value: '2024-05', label: 'Mayo 2024' },
    { value: '2024-06', label: 'Junio 2024' },
    { value: '2024-07', label: 'Julio 2024' },
    { value: '2024-08', label: 'Agosto 2024' }
  ];

  const handleFilterChange = (key, value) => {
    onFiltersChange({
      ...filters,
      [key]: value
    });
  };

  const hasActiveFilters = Object.values(filters)?.some(value => value && value !== '');

  return (
    <div className={`
      bg-white border-r border-gray-200 transition-all duration-300
      ${isCollapsed ? 'w-0 overflow-hidden' : 'w-80'}
    `}>
      <div className="p-6 space-y-6">
        {/* Header */}
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-2">
            <Icon name="Filter" size={20} className="text-primary" />
            <h3 className="text-lg font-semibold text-gray-900">Filtros</h3>
          </div>
          <Button
            variant="ghost"
            size="icon"
            onClick={onToggleCollapse}
            className="lg:hidden"
          >
            <Icon name="X" size={16} />
          </Button>
        </div>

        {/* Search */}
        <div>
          <Input
            label="Buscar comprobante"
            type="search"
            placeholder="Número, inquilino, propiedad..."
            value={filters?.search || ''}
            onChange={(e) => handleFilterChange('search', e?.target?.value)}
          />
        </div>

        {/* Status Filter */}
        <div>
          <Select
            label="Estado"
            options={statusOptions}
            value={filters?.status || ''}
            onChange={(value) => handleFilterChange('status', value)}
          />
        </div>

        {/* Property Filter */}
        <div>
          <Select
            label="Propiedad"
            options={propertyOptions}
            value={filters?.property || ''}
            onChange={(value) => handleFilterChange('property', value)}
            searchable
          />
        </div>

        {/* Month Filter */}
        <div>
          <Select
            label="Mes de generación"
            options={monthOptions}
            value={filters?.month || ''}
            onChange={(value) => handleFilterChange('month', value)}
          />
        </div>

        {/* Amount Range */}
        <div className="space-y-4">
          <label className="text-sm font-medium text-gray-700">Rango de monto</label>
          <div className="grid grid-cols-2 gap-3">
            <Input
              type="number"
              placeholder="Mínimo"
              value={filters?.minAmount || ''}
              onChange={(e) => handleFilterChange('minAmount', e?.target?.value)}
            />
            <Input
              type="number"
              placeholder="Máximo"
              value={filters?.maxAmount || ''}
              onChange={(e) => handleFilterChange('maxAmount', e?.target?.value)}
            />
          </div>
        </div>

        {/* Date Range */}
        <div className="space-y-4">
          <label className="text-sm font-medium text-gray-700">Rango de fechas</label>
          <div className="space-y-3">
            <Input
              type="date"
              label="Desde"
              value={filters?.startDate || ''}
              onChange={(e) => handleFilterChange('startDate', e?.target?.value)}
            />
            <Input
              type="date"
              label="Hasta"
              value={filters?.endDate || ''}
              onChange={(e) => handleFilterChange('endDate', e?.target?.value)}
            />
          </div>
        </div>

        {/* Clear Filters */}
        {hasActiveFilters && (
          <Button
            variant="outline"
            fullWidth
            onClick={onClearFilters}
            iconName="X"
            iconPosition="left"
          >
            Limpiar filtros
          </Button>
        )}

        {/* Quick Filters */}
        <div className="pt-4 border-t border-gray-200">
          <label className="text-sm font-medium text-gray-700 mb-3 block">Filtros rápidos</label>
          <div className="space-y-2">
            <Button
              variant="ghost"
              size="sm"
              fullWidth
              onClick={() => handleFilterChange('status', 'expired')}
              className="justify-start"
            >
              <Icon name="AlertCircle" size={14} className="text-error mr-2" />
              Comprobantes vencidos
            </Button>
            <Button
              variant="ghost"
              size="sm"
              fullWidth
              onClick={() => handleFilterChange('status', 'pending')}
              className="justify-start"
            >
              <Icon name="Clock" size={14} className="text-warning mr-2" />
              Pendientes de pago
            </Button>
            <Button
              variant="ghost"
              size="sm"
              fullWidth
              onClick={() => {
                const today = new Date();
                const thisMonth = today?.toISOString()?.slice(0, 7);
                handleFilterChange('month', thisMonth);
              }}
              className="justify-start"
            >
              <Icon name="Calendar" size={14} className="text-primary mr-2" />
              Este mes
            </Button>
          </div>
        </div>
      </div>
    </div>
  );
};

export default VoucherFiltersPanel;