import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';
import { Checkbox } from '../../../components/ui/Checkbox';

const VoucherGenerationModal = ({ isOpen, onClose, onGenerate }) => {
  const [generationConfig, setGenerationConfig] = useState({
    month: new Date()?.toISOString()?.slice(0, 7),
    properties: [],
    includeAllProperties: true,
    dueDate: '',
    autoSend: false,
    applyIncrease: false,
    increasePercentage: 0,
    generateReminders: true
  });

  const [isGenerating, setIsGenerating] = useState(false);
  const [generationProgress, setGenerationProgress] = useState(0);

  const propertyOptions = [
    { value: 'prop-001', label: 'Departamento Las Condes #301' },
    { value: 'prop-002', label: 'Casa Providencia #45' },
    { value: 'prop-003', label: 'Oficina Ñuñoa #12' },
    { value: 'prop-004', label: 'Local Comercial Vitacura #8' },
    { value: 'prop-005', label: 'Departamento Santiago Centro #205' }
  ];

  const monthOptions = [
    { value: '2024-08', label: 'Agosto 2024' },
    { value: '2024-09', label: 'Septiembre 2024' },
    { value: '2024-10', label: 'Octubre 2024' },
    { value: '2024-11', label: 'Noviembre 2024' },
    { value: '2024-12', label: 'Diciembre 2024' }
  ];

  if (!isOpen) return null;

  const handleConfigChange = (key, value) => {
    setGenerationConfig(prev => ({
      ...prev,
      [key]: value
    }));
  };

  const handleGenerate = async () => {
    setIsGenerating(true);
    setGenerationProgress(0);

    // Simulate generation progress
    const interval = setInterval(() => {
      setGenerationProgress(prev => {
        if (prev >= 100) {
          clearInterval(interval);
          setTimeout(() => {
            setIsGenerating(false);
            onGenerate(generationConfig);
            onClose();
          }, 500);
          return 100;
        }
        return prev + 10;
      });
    }, 200);
  };

  const getEstimatedCount = () => {
    if (generationConfig?.includeAllProperties) {
      return propertyOptions?.length;
    }
    return generationConfig?.properties?.length;
  };

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-1000 p-4">
      <div className="bg-white rounded-lg shadow-xl max-w-2xl w-full max-h-[90vh] overflow-y-auto">
        {/* Header */}
        <div className="flex items-center justify-between p-6 border-b border-gray-200">
          <div className="flex items-center space-x-3">
            <div className="w-10 h-10 bg-primary text-white rounded-lg flex items-center justify-center">
              <Icon name="Zap" size={20} />
            </div>
            <div>
              <h2 className="text-xl font-semibold text-gray-900">Generar Comprobantes</h2>
              <p className="text-sm text-gray-500">Configuración de generación automática</p>
            </div>
          </div>
          <Button variant="ghost" size="icon" onClick={onClose}>
            <Icon name="X" size={20} />
          </Button>
        </div>

        {/* Content */}
        <div className="p-6 space-y-6">
          {!isGenerating ? (
            <>
              {/* Month Selection */}
              <div>
                <Select
                  label="Mes de generación"
                  description="Selecciona el mes para el cual generar los comprobantes"
                  options={monthOptions}
                  value={generationConfig?.month}
                  onChange={(value) => handleConfigChange('month', value)}
                  required
                />
              </div>

              {/* Property Selection */}
              <div className="space-y-4">
                <Checkbox
                  label="Incluir todas las propiedades activas"
                  description="Generar comprobantes para todas las propiedades con contratos vigentes"
                  checked={generationConfig?.includeAllProperties}
                  onChange={(e) => handleConfigChange('includeAllProperties', e?.target?.checked)}
                />

                {!generationConfig?.includeAllProperties && (
                  <Select
                    label="Seleccionar propiedades"
                    description="Elige las propiedades específicas para generar comprobantes"
                    options={propertyOptions}
                    value={generationConfig?.properties}
                    onChange={(value) => handleConfigChange('properties', value)}
                    multiple
                    searchable
                    required
                  />
                )}
              </div>

              {/* Due Date */}
              <div>
                <Input
                  label="Fecha de vencimiento"
                  description="Fecha límite para el pago de los comprobantes"
                  type="date"
                  value={generationConfig?.dueDate}
                  onChange={(e) => handleConfigChange('dueDate', e?.target?.value)}
                  required
                />
              </div>

              {/* Rent Increase */}
              <div className="space-y-4">
                <Checkbox
                  label="Aplicar reajuste de arriendo"
                  description="Aplicar aumento porcentual según reajuste anual"
                  checked={generationConfig?.applyIncrease}
                  onChange={(e) => handleConfigChange('applyIncrease', e?.target?.checked)}
                />

                {generationConfig?.applyIncrease && (
                  <Input
                    label="Porcentaje de reajuste (%)"
                    type="number"
                    placeholder="Ej: 3.5"
                    value={generationConfig?.increasePercentage}
                    onChange={(e) => handleConfigChange('increasePercentage', parseFloat(e?.target?.value) || 0)}
                    min="0"
                    max="100"
                    step="0.1"
                  />
                )}
              </div>

              {/* Additional Options */}
              <div className="space-y-4 pt-4 border-t border-gray-200">
                <h3 className="text-sm font-medium text-gray-900">Opciones adicionales</h3>
                
                <Checkbox
                  label="Enviar automáticamente por email"
                  description="Los comprobantes se enviarán inmediatamente a los inquilinos"
                  checked={generationConfig?.autoSend}
                  onChange={(e) => handleConfigChange('autoSend', e?.target?.checked)}
                />

                <Checkbox
                  label="Programar recordatorios automáticos"
                  description="Crear recordatorios de pago según configuración del sistema"
                  checked={generationConfig?.generateReminders}
                  onChange={(e) => handleConfigChange('generateReminders', e?.target?.checked)}
                />
              </div>

              {/* Summary */}
              <div className="bg-gray-50 rounded-lg p-4">
                <h3 className="text-sm font-medium text-gray-900 mb-3">Resumen de generación</h3>
                <div className="grid grid-cols-2 gap-4 text-sm">
                  <div>
                    <span className="text-gray-500">Comprobantes a generar:</span>
                    <span className="ml-2 font-medium text-gray-900">{getEstimatedCount()}</span>
                  </div>
                  <div>
                    <span className="text-gray-500">Mes:</span>
                    <span className="ml-2 font-medium text-gray-900">
                      {monthOptions?.find(m => m?.value === generationConfig?.month)?.label}
                    </span>
                  </div>
                  <div>
                    <span className="text-gray-500">Envío automático:</span>
                    <span className="ml-2 font-medium text-gray-900">
                      {generationConfig?.autoSend ? 'Sí' : 'No'}
                    </span>
                  </div>
                  <div>
                    <span className="text-gray-500">Reajuste:</span>
                    <span className="ml-2 font-medium text-gray-900">
                      {generationConfig?.applyIncrease ? `${generationConfig?.increasePercentage}%` : 'No'}
                    </span>
                  </div>
                </div>
              </div>
            </>
          ) : (
            /* Generation Progress */
            (<div className="text-center py-8">
              <div className="w-16 h-16 bg-primary/10 text-primary rounded-full flex items-center justify-center mx-auto mb-4">
                <Icon name="Zap" size={24} />
              </div>
              <h3 className="text-lg font-medium text-gray-900 mb-2">Generando comprobantes...</h3>
              <p className="text-gray-500 mb-6">Por favor espera mientras procesamos la información</p>
              <div className="w-full bg-gray-200 rounded-full h-2 mb-4">
                <div 
                  className="bg-primary h-2 rounded-full transition-all duration-300"
                  style={{ width: `${generationProgress}%` }}
                />
              </div>
              <p className="text-sm text-gray-600">{generationProgress}% completado</p>
            </div>)
          )}
        </div>

        {/* Footer */}
        {!isGenerating && (
          <div className="flex items-center justify-end space-x-3 p-6 border-t border-gray-200">
            <Button variant="outline" onClick={onClose}>
              Cancelar
            </Button>
            <Button 
              onClick={handleGenerate}
              disabled={!generationConfig?.month || !generationConfig?.dueDate}
              iconName="Zap"
              iconPosition="left"
            >
              Generar {getEstimatedCount()} comprobantes
            </Button>
          </div>
        )}
      </div>
    </div>
  );
};

export default VoucherGenerationModal;