import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';
import { Checkbox } from '../../../components/ui/Checkbox';
import StatusIndicatorBadges from '../../../components/ui/StatusIndicatorBadges';

const VoucherMobileCard = ({ 
  voucher, 
  isSelected, 
  onSelect, 
  onAction 
}) => {
  const [isExpanded, setIsExpanded] = useState(false);
  const [showActions, setShowActions] = useState(false);

  const formatCurrency = (amount) => {
    return new Intl.NumberFormat('es-CL', {
      style: 'currency',
      currency: 'CLP',
      minimumFractionDigits: 0
    })?.format(amount);
  };

  const formatDate = (dateString) => {
    return new Date(dateString)?.toLocaleDateString('es-CL', {
      day: '2-digit',
      month: '2-digit',
      year: 'numeric'
    });
  };

  const getStatusBadge = (status) => {
    const statusMap = {
      generated: 'pending',
      sent: 'review',
      paid: 'paid',
      expired: 'overdue'
    };
    return statusMap?.[status] || status;
  };

  const quickActions = [
    { key: 'send', label: 'Enviar', icon: 'Send', variant: 'default' },
    { key: 'mark-paid', label: 'Marcar pagado', icon: 'CheckCircle', variant: 'success' },
    { key: 'download', label: 'Descargar', icon: 'Download', variant: 'outline' }
  ];

  const isOverdue = new Date(voucher.dueDate) < new Date() && voucher?.status !== 'paid';

  return (
    <div className={`
      bg-white rounded-lg border border-gray-200 p-4 space-y-4
      ${isSelected ? 'ring-2 ring-primary border-primary' : ''}
      ${isOverdue ? 'border-l-4 border-l-error' : ''}
    `}>
      {/* Header */}
      <div className="flex items-start justify-between">
        <div className="flex items-start space-x-3">
          <Checkbox
            checked={isSelected}
            onChange={(e) => onSelect(voucher?.id, e?.target?.checked)}
          />
          <div className="flex-1 min-w-0">
            <div className="flex items-center space-x-2 mb-1">
              <span className="font-mono text-sm font-medium text-gray-900">
                {voucher?.number}
              </span>
              {voucher?.isAutoGenerated && (
                <Icon name="Zap" size={12} className="text-accent" />
              )}
              {isOverdue && (
                <Icon name="AlertTriangle" size={12} className="text-error" />
              )}
            </div>
            <p className="text-xs text-gray-500 truncate">{voucher?.property?.name}</p>
          </div>
        </div>
        
        <div className="flex items-center space-x-2">
          <StatusIndicatorBadges status={getStatusBadge(voucher?.status)} size="sm" />
          <Button
            variant="ghost"
            size="icon"
            onClick={() => setShowActions(!showActions)}
          >
            <Icon name="MoreVertical" size={16} />
          </Button>
        </div>
      </div>
      {/* Main Info */}
      <div className="grid grid-cols-2 gap-4">
        <div>
          <p className="text-xs text-gray-500 mb-1">Inquilino</p>
          <p className="text-sm font-medium text-gray-900 truncate">{voucher?.tenant?.name}</p>
        </div>
        <div>
          <p className="text-xs text-gray-500 mb-1">Monto</p>
          <p className="text-sm font-medium text-gray-900">{formatCurrency(voucher?.amount)}</p>
        </div>
      </div>
      {/* Dates */}
      <div className="grid grid-cols-2 gap-4">
        <div>
          <p className="text-xs text-gray-500 mb-1">Generado</p>
          <p className="text-sm text-gray-900">{formatDate(voucher?.generatedDate)}</p>
        </div>
        <div>
          <p className="text-xs text-gray-500 mb-1">Vencimiento</p>
          <p className={`text-sm ${isOverdue ? 'text-error font-medium' : 'text-gray-900'}`}>
            {formatDate(voucher?.dueDate)}
          </p>
        </div>
      </div>
      {/* Expandable Details */}
      {isExpanded && (
        <div className="pt-4 border-t border-gray-200 space-y-3">
          <div>
            <p className="text-xs text-gray-500 mb-1">Dirección de la propiedad</p>
            <p className="text-sm text-gray-900">{voucher?.property?.address}</p>
          </div>
          <div>
            <p className="text-xs text-gray-500 mb-1">Email del inquilino</p>
            <p className="text-sm text-gray-900">{voucher?.tenant?.email}</p>
          </div>
          {voucher?.notes && (
            <div>
              <p className="text-xs text-gray-500 mb-1">Notas</p>
              <p className="text-sm text-gray-900">{voucher?.notes}</p>
            </div>
          )}
        </div>
      )}
      {/* Quick Actions */}
      {showActions && (
        <div className="pt-4 border-t border-gray-200">
          <div className="grid grid-cols-3 gap-2">
            {quickActions?.map((action) => (
              <Button
                key={action?.key}
                variant={action?.variant}
                size="sm"
                onClick={() => {
                  onAction(action?.key, voucher);
                  setShowActions(false);
                }}
                iconName={action?.icon}
                iconPosition="left"
                className="text-xs"
              >
                {action?.label}
              </Button>
            ))}
          </div>
        </div>
      )}
      {/* Footer */}
      <div className="flex items-center justify-between pt-2">
        <Button
          variant="ghost"
          size="sm"
          onClick={() => setIsExpanded(!isExpanded)}
          iconName={isExpanded ? "ChevronUp" : "ChevronDown"}
          iconPosition="right"
          className="text-xs"
        >
          {isExpanded ? 'Menos detalles' : 'Más detalles'}
        </Button>
        
        {voucher?.lastPayment && (
          <div className="text-right">
            <p className="text-xs text-gray-500">Último pago</p>
            <p className="text-xs text-gray-900">{formatDate(voucher?.lastPayment)}</p>
          </div>
        )}
      </div>
    </div>
  );
};

export default VoucherMobileCard;