import React from 'react';
import Icon from '../../../components/AppIcon';

const VoucherStatsCards = ({ stats }) => {
  const formatCurrency = (amount) => {
    return new Intl.NumberFormat('es-CL', {
      style: 'currency',
      currency: 'CLP',
      minimumFractionDigits: 0
    })?.format(amount);
  };

  const statsCards = [
    {
      key: 'total',
      title: 'Total Comprobantes',
      value: stats?.total?.toString(),
      icon: 'FileText',
      color: 'primary',
      bgColor: 'bg-primary/10',
      textColor: 'text-primary',
      change: stats?.totalChange,
      changeType: stats?.totalChange >= 0 ? 'positive' : 'negative'
    },
    {
      key: 'pending',
      title: 'Pendientes de Pago',
      value: stats?.pending?.toString(),
      icon: 'Clock',
      color: 'warning',
      bgColor: 'bg-warning/10',
      textColor: 'text-warning',
      change: stats?.pendingChange,
      changeType: stats?.pendingChange <= 0 ? 'positive' : 'negative'
    },
    {
      key: 'paid',
      title: 'Pagados',
      value: stats?.paid?.toString(),
      icon: 'CheckCircle',
      color: 'success',
      bgColor: 'bg-success/10',
      textColor: 'text-success',
      change: stats?.paidChange,
      changeType: stats?.paidChange >= 0 ? 'positive' : 'negative'
    },
    {
      key: 'overdue',
      title: 'Vencidos',
      value: stats?.overdue?.toString(),
      icon: 'AlertCircle',
      color: 'error',
      bgColor: 'bg-error/10',
      textColor: 'text-error',
      change: stats?.overdueChange,
      changeType: stats?.overdueChange <= 0 ? 'positive' : 'negative'
    },
    {
      key: 'revenue',
      title: 'Ingresos del Mes',
      value: formatCurrency(stats?.monthlyRevenue),
      icon: 'DollarSign',
      color: 'accent',
      bgColor: 'bg-accent/10',
      textColor: 'text-gray-900',
      change: stats?.revenueChange,
      changeType: stats?.revenueChange >= 0 ? 'positive' : 'negative'
    },
    {
      key: 'collection',
      title: 'Tasa de Cobranza',
      value: `${stats?.collectionRate}%`,
      icon: 'TrendingUp',
      color: 'secondary',
      bgColor: 'bg-secondary/10',
      textColor: 'text-secondary',
      change: stats?.collectionChange,
      changeType: stats?.collectionChange >= 0 ? 'positive' : 'negative'
    }
  ];

  return (
    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-6 gap-6 mb-8">
      {statsCards?.map((card) => (
        <div key={card?.key} className="bg-white rounded-lg border border-gray-200 p-6 hover-lift">
          {/* Header */}
          <div className="flex items-center justify-between mb-4">
            <div className={`w-12 h-12 ${card?.bgColor} rounded-lg flex items-center justify-center`}>
              <Icon name={card?.icon} size={20} className={card?.textColor} />
            </div>
            {card?.change !== undefined && (
              <div className={`flex items-center space-x-1 text-xs font-medium ${
                card?.changeType === 'positive' ? 'text-success' : 'text-error'
              }`}>
                <Icon 
                  name={card?.changeType === 'positive' ? 'TrendingUp' : 'TrendingDown'} 
                  size={12} 
                />
                <span>{Math.abs(card?.change)}%</span>
              </div>
            )}
          </div>

          {/* Content */}
          <div>
            <h3 className="text-2xl font-bold text-gray-900 mb-1">{card?.value}</h3>
            <p className="text-sm text-gray-600">{card?.title}</p>
          </div>

          {/* Progress Bar for Collection Rate */}
          {card?.key === 'collection' && (
            <div className="mt-4">
              <div className="w-full bg-gray-200 rounded-full h-2">
                <div 
                  className="bg-secondary h-2 rounded-full transition-all duration-300"
                  style={{ width: `${Math.min(stats?.collectionRate, 100)}%` }}
                />
              </div>
            </div>
          )}

          {/* Additional Info */}
          {card?.key === 'revenue' && (
            <div className="mt-3 pt-3 border-t border-gray-100">
              <p className="text-xs text-gray-500">
                Objetivo: {formatCurrency(stats?.revenueTarget)}
              </p>
              <div className="w-full bg-gray-200 rounded-full h-1 mt-1">
                <div 
                  className="bg-accent h-1 rounded-full transition-all duration-300"
                  style={{ width: `${Math.min((stats?.monthlyRevenue / stats?.revenueTarget) * 100, 100)}%` }}
                />
              </div>
            </div>
          )}
        </div>
      ))}
    </div>
  );
};

export default VoucherStatsCards;