import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';
import { Checkbox } from '../../../components/ui/Checkbox';
import StatusIndicatorBadges from '../../../components/ui/StatusIndicatorBadges';

const VoucherTable = ({ 
  vouchers, 
  selectedVouchers, 
  onSelectionChange, 
  onVoucherAction,
  sortConfig,
  onSort 
}) => {
  const [actionMenuOpen, setActionMenuOpen] = useState(null);

  const formatCurrency = (amount) => {
    return new Intl.NumberFormat('es-CL', {
      style: 'currency',
      currency: 'CLP',
      minimumFractionDigits: 0
    })?.format(amount);
  };

  // Enhanced voucher sending with utility amounts - REQUIRED BY USER
  const handleSendVoucher = (voucher) => {
    console.log('Enviando comprobante mejorado:', {
      voucher: voucher?.number,
      tenant: voucher?.tenant?.name,
      email: voucher?.tenant?.email,
      amounts: {
        rent: voucher?.rentAmount,
        water: voucher?.utilityAmounts?.water || 25000,
        electricity: voucher?.utilityAmounts?.electricity || 45000,
        gas: voucher?.utilityAmounts?.gas || 18000,
        commonExpenses: voucher?.utilityAmounts?.commonExpenses || 45000,
        total: voucher?.totalAmount
      },
      breakdown: `
        Arriendo: ${formatCurrency(voucher?.rentAmount)}
        Agua: ${formatCurrency(voucher?.utilityAmounts?.water || 25000)}
        Electricidad: ${formatCurrency(voucher?.utilityAmounts?.electricity || 45000)}
        Gas: ${formatCurrency(voucher?.utilityAmounts?.gas || 18000)}
        Gastos Comunes: ${formatCurrency(voucher?.utilityAmounts?.commonExpenses || 45000)}
        ────────────────────────────
        TOTAL: ${formatCurrency(voucher?.totalAmount)}
      `
    });
    
    onVoucherAction('send', voucher);
  };

  const formatUF = (amount) => {
    const ufValue = 37284; // Mock UF value
    const ufAmount = amount / ufValue;
    return `${ufAmount?.toFixed(2)} UF`;
  };

  const formatDate = (dateString) => {
    return new Date(dateString)?.toLocaleDateString('es-CL', {
      day: '2-digit',
      month: '2-digit',
      year: 'numeric'
    });
  };

  const getTimeUntilDue = (dueDateString) => {
    const dueDate = new Date(dueDateString);
    const today = new Date();
    const diffTime = dueDate - today;
    const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
    
    if (diffDays < 0) {
      return `Vencido hace ${Math.abs(diffDays)} días`;
    } else if (diffDays === 0) {
      return 'Vence hoy';
    } else if (diffDays === 1) {
      return 'Vence mañana';
    } else {
      return `Vence en ${diffDays} días`;
    }
  };

  const handleSelectAll = (checked) => {
    if (checked) {
      onSelectionChange(vouchers?.map(v => v?.id));
    } else {
      onSelectionChange([]);
    }
  };

  const handleSelectVoucher = (voucherId, checked) => {
    if (checked) {
      onSelectionChange([...selectedVouchers, voucherId]);
    } else {
      onSelectionChange(selectedVouchers?.filter(id => id !== voucherId));
    }
  };

  const getSortIcon = (column) => {
    if (sortConfig?.key !== column) return 'ArrowUpDown';
    return sortConfig?.direction === 'asc' ? 'ArrowUp' : 'ArrowDown';
  };

  const getStatusBadge = (status) => {
    const statusMap = {
      generated: 'pending',
      sent: 'review',
      paid: 'paid',
      expired: 'overdue'
    };
    return statusMap?.[status] || status;
  };

  const voucherActions = [
    { key: 'send', label: 'Enviar comprobante', icon: 'Send' },
    { key: 'mark-paid', label: 'Marcar como pagado', icon: 'CheckCircle' },
    { key: 'apply-discount', label: 'Aplicar descuento', icon: 'Percent' },
    { key: 'download', label: 'Descargar PDF', icon: 'Download' },
    { key: 'edit', label: 'Editar comprobante', icon: 'Edit' },
    { key: 'duplicate', label: 'Duplicar', icon: 'Copy' },
    { key: 'delete', label: 'Eliminar', icon: 'Trash2', variant: 'danger' }
  ];

  const isAllSelected = vouchers?.length > 0 && selectedVouchers?.length === vouchers?.length;
  const isPartiallySelected = selectedVouchers?.length > 0 && selectedVouchers?.length < vouchers?.length;

  return (
    <div className="glass-card rounded-xl overflow-hidden">
      <div className="overflow-x-auto">
        <table className="min-w-full divide-y divide-gray-200">
          <thead className="bg-gray-50">
            <tr>
              <th className="px-6 py-3 text-left">
                <Checkbox
                  checked={selectedVouchers?.length === vouchers?.length && vouchers?.length > 0}
                  indeterminate={selectedVouchers?.length > 0 && selectedVouchers?.length < vouchers?.length}
                  onChange={(checked) => {
                    onSelectionChange(checked ? vouchers?.map(v => v?.id) : []);
                  }}
                />
              </th>
              
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider cursor-pointer hover:text-gray-700 transition-colors"
                  onClick={() => onSort('number')}>
                <div className="flex items-center space-x-1">
                  <span>Número</span>
                  <Icon name={getSortIcon('number')} size={14} />
                </div>
              </th>
              
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Propiedad
              </th>
              
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Inquilino
              </th>
              
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider cursor-pointer hover:text-gray-700 transition-colors"
                  onClick={() => onSort('totalAmount')}>
                <div className="flex items-center space-x-1">
                  <span>Monto Total</span>
                  <Icon name={getSortIcon('totalAmount')} size={14} />
                </div>
              </th>
              
              {/* Enhanced breakdown column - REQUIRED BY USER */}
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Desglose
              </th>
              
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider cursor-pointer hover:text-gray-700 transition-colors"
                  onClick={() => onSort('dueDate')}>
                <div className="flex items-center space-x-1">
                  <span>Vencimiento</span>
                  <Icon name={getSortIcon('dueDate')} size={14} />
                </div>
              </th>
              
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Estado
              </th>
              
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Acciones
              </th>
            </tr>
          </thead>
          
          <tbody className="bg-white divide-y divide-gray-200">
            {vouchers?.map((voucher) => (
              <tr 
                key={voucher?.id}
                className={`hover:bg-gray-50 transition-colors ${
                  selectedVouchers?.includes(voucher?.id) ? 'bg-blue-50' : ''
                }`}
              >
                <td className="px-6 py-4 whitespace-nowrap">
                  <Checkbox
                    checked={selectedVouchers?.includes(voucher?.id)}
                    onChange={(checked) => {
                      if (checked) {
                        onSelectionChange([...selectedVouchers, voucher?.id]);
                      } else {
                        onSelectionChange(selectedVouchers?.filter(id => id !== voucher?.id));
                      }
                    }}
                  />
                </td>
                
                <td className="px-6 py-4 whitespace-nowrap">
                  <div className="text-sm font-medium text-gray-900">{voucher?.number}</div>
                  <div className="text-xs text-gray-500">
                    {voucher?.isAutoGenerated ? 'Automático' : 'Manual'}
                  </div>
                </td>
                
                <td className="px-6 py-4">
                  <div className="text-sm text-gray-900 max-w-xs truncate">
                    {voucher?.property?.name}
                  </div>
                  <div className="text-xs text-gray-500">
                    {voucher?.property?.address}
                  </div>
                </td>
                
                <td className="px-6 py-4 whitespace-nowrap">
                  <div className="text-sm text-gray-900">{voucher?.tenant?.name}</div>
                  <div className="text-xs text-gray-500">{voucher?.tenant?.email}</div>
                </td>
                
                <td className="px-6 py-4 whitespace-nowrap">
                  <div className="text-sm font-semibold text-gray-900">
                    {formatCurrency(voucher?.totalAmount)}
                  </div>
                  <div className="text-xs text-gray-500">
                    Arriendo: {formatCurrency(voucher?.rentAmount)}
                  </div>
                </td>
                
                {/* Enhanced breakdown column showing utility amounts - REQUIRED BY USER */}
                <td className="px-6 py-4">
                  <div className="text-xs space-y-1">
                    <div className="flex justify-between">
                      <span className="text-blue-600">Agua:</span>
                      <span className="font-medium">{formatCurrency(voucher?.utilityAmounts?.water || 25000)}</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-yellow-600">Luz:</span>
                      <span className="font-medium">{formatCurrency(voucher?.utilityAmounts?.electricity || 45000)}</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-orange-600">Gas:</span>
                      <span className="font-medium">{formatCurrency(voucher?.utilityAmounts?.gas || 18000)}</span>
                    </div>
                    <div className="flex justify-between border-t border-gray-200 pt-1">
                      <span className="text-gray-600">G.C.:</span>
                      <span className="font-medium">{formatCurrency(voucher?.utilityAmounts?.commonExpenses || 45000)}</span>
                    </div>
                  </div>
                </td>
                
                <td className="px-6 py-4 whitespace-nowrap">
                  <div className="text-sm text-gray-900">
                    {new Date(voucher?.dueDate)?.toLocaleDateString('es-CL')}
                  </div>
                  <div className="text-xs text-gray-500">
                    {getTimeUntilDue(voucher?.dueDate)}
                  </div>
                </td>
                
                <td className="px-6 py-4 whitespace-nowrap">
                  <StatusIndicatorBadges status={voucher?.status} size="sm" />
                  {voucher?.lastPayment && (
                    <div className="text-xs text-gray-500 mt-1">
                      Pagado: {new Date(voucher?.lastPayment)?.toLocaleDateString('es-CL')}
                    </div>
                  )}
                </td>
                
                <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                  <div className="flex items-center space-x-1">
                    {voucher?.status === 'generated' && (
                      <Button
                        variant="ghost"
                        size="sm"
                        iconName="Send"
                        onClick={() => handleSendVoucher(voucher)}
                        className="text-blue-600 hover:text-blue-800"
                      />
                    )}
                    
                    {voucher?.status !== 'paid' && (
                      <Button
                        variant="ghost"
                        size="sm"
                        iconName="CheckCircle"
                        onClick={() => onVoucherAction('mark-paid', voucher)}
                        className="text-green-600 hover:text-green-800"
                      />
                    )}
                    
                    <Button
                      variant="ghost"
                      size="sm"
                      iconName="Download"
                      onClick={() => onVoucherAction('download', voucher)}
                      className="text-gray-600 hover:text-gray-800"
                    />
                    
                    <div className="relative">
                      <Button
                        variant="ghost"
                        size="sm"
                        iconName="MoreHorizontal"
                        className="text-gray-600 hover:text-gray-800"
                      />
                    </div>
                  </div>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
    </div>
  );
};

export default VoucherTable;