import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import Icon from '../../components/AppIcon';
import Button from '../../components/ui/Button';
import Input from '../../components/ui/Input';
import Header from '../../components/ui/Header';
import Sidebar from '../../components/ui/Sidebar';
import NavigationBreadcrumbs from '../../components/ui/NavigationBreadcrumbs';
import QuickActionFloatingButton from '../../components/ui/QuickActionFloatingButton';

// Import page components
import VoucherFiltersPanel from './components/VoucherFiltersPanel';
import VoucherTable from './components/VoucherTable';
import BulkActionsBar from './components/BulkActionsBar';
import VoucherGenerationModal from './components/VoucherGenerationModal';
import VoucherStatsCards from './components/VoucherStatsCards';
import VoucherMobileCard from './components/VoucherMobileCard';

const VouchersManagement = () => {
  const navigate = useNavigate();
  const [isSidebarOpen, setIsSidebarOpen] = useState(false);
  const [isFiltersCollapsed, setIsFiltersCollapsed] = useState(false);
  const [isMobile, setIsMobile] = useState(false);
  const [selectedVouchers, setSelectedVouchers] = useState([]);
  const [showGenerationModal, setShowGenerationModal] = useState(false);
  const [searchTerm, setSearchTerm] = useState('');
  const [sortConfig, setSortConfig] = useState({ key: 'dueDate', direction: 'desc' });

  const [filters, setFilters] = useState({
    search: '',
    status: '',
    property: '',
    month: '',
    minAmount: '',
    maxAmount: '',
    startDate: '',
    endDate: ''
  });

  // Enhanced mock voucher data with utility amounts
  const mockVouchers = [
    {
      id: 'voucher-001',
      number: 'COMP-2024-001',
      property: {
        id: 'prop-001',
        name: 'Departamento Las Condes #301',
        address: 'Av. Apoquindo 3000, Las Condes'
      },
      tenant: {
        id: 'tenant-001',
        name: 'María González Pérez',
        email: 'maria.gonzalez@email.com'
      },
      rentAmount: 850000,
      utilityAmounts: {
        water: 25000,
        electricity: 45000,
        gas: 18000,
        commonExpenses: 45000
      },
      totalAmount: 983000, // rent + utilities
      generatedDate: '2024-08-01',
      dueDate: '2024-08-05',
      status: 'paid',
      isAutoGenerated: true,
      lastPayment: '2024-08-03',
      notes: 'Pago realizado mediante transferencia bancaria'
    },
    {
      id: 'voucher-002',
      number: 'COMP-2024-002',
      property: {
        id: 'prop-002',
        name: 'Casa Providencia #45',
        address: 'Manuel Montt 1245, Providencia'
      },
      tenant: {
        id: 'tenant-002',
        name: 'Carlos Rodríguez Silva',
        email: 'carlos.rodriguez@email.com'
      },
      rentAmount: 1200000,
      utilityAmounts: {
        water: 35000,
        electricity: 75000,
        gas: 28000,
        commonExpenses: 60000
      },
      totalAmount: 1398000,
      generatedDate: '2024-08-01',
      dueDate: '2024-08-05',
      status: 'sent',
      isAutoGenerated: true
    },
    {
      id: 'voucher-003',
      number: 'COMP-2024-003',
      property: {
        id: 'prop-003',
        name: 'Oficina Ñuñoa #12',
        address: 'Irarrázaval 2890, Ñuñoa'
      },
      tenant: {
        id: 'tenant-003',
        name: 'Ana Martínez López',
        email: 'ana.martinez@email.com'
      },
      rentAmount: 950000,
      utilityAmounts: {
        water: 40000,
        electricity: 55000,
        gas: 30000,
        commonExpenses: 50000
      },
      totalAmount: 1445000,
      generatedDate: '2024-08-01',
      dueDate: '2024-08-05',
      status: 'expired',
      isAutoGenerated: true
    },
    {
      id: 'voucher-004',
      number: 'COMP-2024-004',
      property: {
        id: 'prop-004',
        name: 'Local Comercial Vitacura #8',
        address: 'Av. Vitacura 5678, Vitacura'
      },
      tenant: {
        id: 'tenant-004',
        name: 'Roberto Fernández Castro',
        email: 'roberto.fernandez@email.com'
      },
      rentAmount: 2500000,
      utilityAmounts: {
        water: 50000,
        electricity: 70000,
        gas: 40000,
        commonExpenses: 60000
      },
      totalAmount: 3600000,
      generatedDate: '2024-08-01',
      dueDate: '2024-08-05',
      status: 'generated',
      isAutoGenerated: false
    },
    {
      id: 'voucher-005',
      number: 'COMP-2024-005',
      property: {
        id: 'prop-005',
        name: 'Departamento Santiago Centro #205',
        address: 'Estado 1234, Santiago Centro'
      },
      tenant: {
        id: 'tenant-005',
        name: 'Patricia Morales Díaz',
        email: 'patricia.morales@email.com'
      },
      rentAmount: 750000,
      utilityAmounts: {
        water: 30000,
        electricity: 45000,
        gas: 25000,
        commonExpenses: 35000
      },
      totalAmount: 1175000,
      generatedDate: '2024-08-01',
      dueDate: '2024-08-05',
      status: 'paid',
      isAutoGenerated: true,
      lastPayment: '2024-08-04'
    }
  ];

  // Mock stats data
  const mockStats = {
    total: 45,
    pending: 18,
    paid: 22,
    overdue: 5,
    monthlyRevenue: 38500000,
    revenueTarget: 42000000,
    collectionRate: 85,
    totalChange: 12,
    pendingChange: -8,
    paidChange: 15,
    overdueChange: -3,
    revenueChange: 8,
    collectionChange: 5
  };

  useEffect(() => {
    const checkMobile = () => {
      const mobile = window.innerWidth < 1024;
      setIsMobile(mobile);
      if (!mobile) {
        setIsSidebarOpen(false);
      }
    };

    checkMobile();
    window.addEventListener('resize', checkMobile);
    return () => window.removeEventListener('resize', checkMobile);
  }, []);

  const handleMenuToggle = () => {
    setIsSidebarOpen(!isSidebarOpen);
  };

  const handleFiltersChange = (newFilters) => {
    setFilters(newFilters);
  };

  const handleClearFilters = () => {
    setFilters({
      search: '',
      status: '',
      property: '',
      month: '',
      minAmount: '',
      maxAmount: '',
      startDate: '',
      endDate: ''
    });
  };

  const handleSort = (key) => {
    setSortConfig(prev => ({
      key,
      direction: prev?.key === key && prev?.direction === 'asc' ? 'desc' : 'asc'
    }));
  };

  const handleVoucherAction = (action, voucher) => {
    console.log(`Action: ${action}`, voucher);
    
    switch (action) {
      case 'send': 
        // Enhanced send with utility amounts
        console.log('Sending voucher with utility amounts:', {
          voucher: voucher?.number,
          totalAmount: voucher?.totalAmount,
          breakdown: {
            rent: voucher?.rentAmount,
            utilities: voucher?.utilityAmounts
          }
        });
        break;
      case 'mark-paid': console.log('Marking as paid:', voucher?.number);
        break;
      case 'apply-discount': console.log('Applying discount to:', voucher?.number);
        break;
      case 'download':
        console.log('Downloading PDF for:', voucher?.number);
        break;
      case 'edit': console.log('Editing voucher:', voucher?.number);
        break;
      case 'duplicate': console.log('Duplicating voucher:', voucher?.number);
        break;
      case 'delete':
        console.log('Deleting voucher:', voucher?.number);
        break;
      default:
        console.log('Unknown action:', action);
    }
  };

  const handleBulkAction = (action) => {
    console.log(`Bulk action: ${action}`, selectedVouchers);
    
    switch (action) {
      case 'send-all': console.log('Sending all selected vouchers');
        break;
      case 'mark-paid-all': console.log('Marking all as paid');
        break;
      case 'apply-discount-all': console.log('Applying discount to all');
        break;
      case 'generate-report': navigate('/reports-analytics');
        break;
      case 'export-pdf':
        console.log('Exporting PDFs');
        break;
      case 'delete-all':
        console.log('Deleting all selected');
        break;
      default:
        console.log('Unknown bulk action:', action);
    }
  };

  const handleGeneration = (config) => {
    console.log('Generating vouchers with config:', config);
    // Simulate voucher generation
    setTimeout(() => {
      console.log('Vouchers generated successfully');
    }, 2000);
  };

  const filteredVouchers = mockVouchers?.filter(voucher => {
    const matchesSearch = !filters?.search || 
      voucher?.number?.toLowerCase()?.includes(filters?.search?.toLowerCase()) ||
      voucher?.tenant?.name?.toLowerCase()?.includes(filters?.search?.toLowerCase()) ||
      voucher?.property?.name?.toLowerCase()?.includes(filters?.search?.toLowerCase());
    
    const matchesStatus = !filters?.status || voucher?.status === filters?.status;
    const matchesProperty = !filters?.property || voucher?.property?.id === filters?.property;
    
    return matchesSearch && matchesStatus && matchesProperty;
  });

  const sortedVouchers = [...filteredVouchers]?.sort((a, b) => {
    const aValue = a?.[sortConfig?.key];
    const bValue = b?.[sortConfig?.key];
    
    if (sortConfig?.direction === 'asc') {
      return aValue > bValue ? 1 : -1;
    } else {
      return aValue < bValue ? 1 : -1;
    }
  });

  return (
    <div className="min-h-screen bg-background">
      <Header onMenuToggle={handleMenuToggle} isMenuOpen={isSidebarOpen} />
      <Sidebar isOpen={isSidebarOpen} onClose={() => setIsSidebarOpen(false)} />
      <main className={`transition-all duration-300 ${isSidebarOpen && isMobile ? 'lg:ml-60' : 'lg:ml-60'}`}>
        <div className="pt-16">
          <div className="flex">
            {/* Filters Panel */}
            <VoucherFiltersPanel
              filters={filters}
              onFiltersChange={handleFiltersChange}
              onClearFilters={handleClearFilters}
              isCollapsed={isFiltersCollapsed}
              onToggleCollapse={() => setIsFiltersCollapsed(!isFiltersCollapsed)}
            />

            {/* Main Content */}
            <div className="flex-1 p-6">
              {/* Breadcrumbs */}
              <NavigationBreadcrumbs />

              {/* Page Header */}
              <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between mb-8">
                <div className="flex items-center space-x-4 mb-4 lg:mb-0">
                  <div className="w-12 h-12 bg-primary text-white rounded-lg flex items-center justify-center">
                    <Icon name="FileText" size={24} />
                  </div>
                  <div>
                    <h1 className="text-2xl font-bold text-gray-900">Gestión de Comprobantes</h1>
                    <p className="text-gray-600">Administra y controla todos los comprobantes de arriendo</p>
                  </div>
                  <Button
                    variant="ghost"
                    size="icon"
                    onClick={() => setIsFiltersCollapsed(!isFiltersCollapsed)}
                    className="lg:hidden"
                  >
                    <Icon name="Filter" size={20} />
                  </Button>
                </div>

                <div className="flex items-center space-x-3">
                  <Button
                    variant="outline"
                    iconName="Download"
                    iconPosition="left"
                    onClick={() => console.log('Exporting data')}
                  >
                    Exportar
                  </Button>
                  <Button
                    variant="secondary"
                    iconName="Plus"
                    iconPosition="left"
                    onClick={() => console.log('Manual voucher creation')}
                  >
                    Crear Comprobante
                  </Button>
                  <Button
                    iconName="Zap"
                    iconPosition="left"
                    onClick={() => setShowGenerationModal(true)}
                  >
                    Generar Comprobantes
                  </Button>
                </div>
              </div>

              {/* Stats Cards */}
              <VoucherStatsCards stats={mockStats} />

              {/* Search Bar */}
              <div className="mb-6">
                <div className="flex items-center space-x-4">
                  <div className="flex-1">
                    <Input
                      type="search"
                      placeholder="Buscar por número, inquilino o propiedad..."
                      value={searchTerm}
                      onChange={(e) => setSearchTerm(e?.target?.value)}
                      className="w-full"
                    />
                  </div>
                  <div className="flex items-center space-x-2 text-sm text-gray-600">
                    <span>{sortedVouchers?.length} comprobantes</span>
                    {selectedVouchers?.length > 0 && (
                      <span className="text-primary font-medium">
                        ({selectedVouchers?.length} seleccionados)
                      </span>
                    )}
                  </div>
                </div>
              </div>

              {/* Desktop Table View */}
              <div className="hidden lg:block">
                <VoucherTable
                  vouchers={sortedVouchers}
                  selectedVouchers={selectedVouchers}
                  onSelectionChange={setSelectedVouchers}
                  onVoucherAction={handleVoucherAction}
                  sortConfig={sortConfig}
                  onSort={handleSort}
                />
              </div>

              {/* Mobile Card View */}
              <div className="lg:hidden space-y-4">
                {sortedVouchers?.map((voucher) => (
                  <VoucherMobileCard
                    key={voucher?.id}
                    voucher={voucher}
                    isSelected={selectedVouchers?.includes(voucher?.id)}
                    onSelect={(id, checked) => {
                      if (checked) {
                        setSelectedVouchers([...selectedVouchers, id]);
                      } else {
                        setSelectedVouchers(selectedVouchers?.filter(vId => vId !== id));
                      }
                    }}
                    onAction={handleVoucherAction}
                  />
                ))}
              </div>

              {/* Pagination */}
              {sortedVouchers?.length > 0 && (
                <div className="mt-8 flex items-center justify-between">
                  <p className="text-sm text-gray-600">
                    Mostrando {sortedVouchers?.length} de {mockVouchers?.length} comprobantes
                  </p>
                  <div className="flex items-center space-x-2">
                    <Button variant="outline" size="sm" disabled>
                      <Icon name="ChevronLeft" size={16} />
                    </Button>
                    <Button variant="outline" size="sm" className="bg-primary text-white">
                      1
                    </Button>
                    <Button variant="outline" size="sm" disabled>
                      <Icon name="ChevronRight" size={16} />
                    </Button>
                  </div>
                </div>
              )}
            </div>
          </div>
        </div>
      </main>
      {/* Bulk Actions Bar */}
      <BulkActionsBar
        selectedCount={selectedVouchers?.length}
        onBulkAction={handleBulkAction}
        onClearSelection={() => setSelectedVouchers([])}
      />
      {/* Generation Modal */}
      <VoucherGenerationModal
        isOpen={showGenerationModal}
        onClose={() => setShowGenerationModal(false)}
        onGenerate={handleGeneration}
      />
      {/* Quick Action FAB */}
      <QuickActionFloatingButton />
    </div>
  );
};

export default VouchersManagement;