// Modern Voucher PDF Generation Utilities
// This module handles the creation of professional voucher PDFs for both tenants and owners

const formatCurrency = (amount) => {
  return new Intl.NumberFormat('es-CL', {
    style: 'currency',
    currency: 'CLP',
    minimumFractionDigits: 0
  })?.format(amount || 0);
};

const formatDate = (dateString) => {
  if (!dateString) return '-';
  return new Date(dateString)?.toLocaleDateString('es-CL', {
    year: 'numeric',
    month: 'long',
    day: 'numeric'
  });
};

const generateVoucherHTML = (voucherType, payment, propertyData, userData) => {
  const currentDate = new Date()?.toLocaleDateString('es-CL');
  const isOwner = voucherType === 'owner';
  
  // Calculate totals
  const rentAmount = payment?.amount || 0;
  const gasAmount = payment?.gasAmount || 0;
  const commonExpenses = payment?.commonExpenses || 0;
  const waterAmount = payment?.waterAmount || 0;
  const electricityAmount = payment?.electricityAmount || 0;
  const internetAmount = payment?.internetAmount || 0;
  
  const subtotal = rentAmount + gasAmount + commonExpenses + waterAmount + electricityAmount + internetAmount;
  
  // Discounts calculation
  const tenantDiscounts = payment?.tenantDiscounts?.filter(d => d?.amount > 0) || [];
  const ownerDiscounts = payment?.ownerDiscounts?.filter(d => d?.amount > 0) || [];
  const otherDiscounts = payment?.otherDiscounts?.filter(d => d?.amount > 0) || [];
  
  const tenantDiscountTotal = tenantDiscounts?.reduce((sum, d) => sum + (d?.amount || 0), 0);
  const ownerDiscountTotal = ownerDiscounts?.reduce((sum, d) => sum + (d?.amount || 0), 0);
  const otherDiscountTotal = otherDiscounts?.reduce((sum, d) => sum + (d?.amount || 0), 0);
  
  const totalDiscounts = isOwner 
    ? tenantDiscountTotal + ownerDiscountTotal + otherDiscountTotal
    : tenantDiscountTotal;
    
  const finalAmount = subtotal - totalDiscounts;
  
  // Owner-specific calculations
  const adminCommission = isOwner ? Math.round(finalAmount * 0.08) : 0; // 8% commission
  const ownerNetAmount = isOwner ? finalAmount - adminCommission : finalAmount;

  return `
    <!DOCTYPE html>
    <html lang="es">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Comprobante de Pago - ${payment?.voucherNumber}</title>
        <style>
            * {
                margin: 0;
                padding: 0;
                box-sizing: border-box;
            }
            
            body {
                font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
                line-height: 1.6;
                color: #333;
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                min-height: 100vh;
                padding: 20px;
            }
            
            .voucher-container {
                max-width: 800px;
                margin: 0 auto;
                background: white;
                border-radius: 16px;
                box-shadow: 0 20px 40px rgba(0,0,0,0.1);
                overflow: hidden;
                position: relative;
            }
            
            .voucher-header {
                background: ${isOwner ? 'linear-gradient(135deg, #22c55e 0%, #16a34a 100%)' : 'linear-gradient(135deg, #3b82f6 0%, #2563eb 100%)'};
                color: white;
                padding: 30px;
                position: relative;
                overflow: hidden;
            }
            
            .voucher-header::before {
                content: '';
                position: absolute;
                top: -50%;
                right: -50%;
                width: 200%;
                height: 200%;
                background: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 100 100"><circle cx="50" cy="50" r="2" fill="rgba(255,255,255,0.1)"/></svg>') repeat;
                animation: float 20s infinite linear;
            }
            
            @keyframes float {
                0% { transform: translate(-50%, -50%) rotate(0deg); }
                100% { transform: translate(-50%, -50%) rotate(360deg); }
            }
            
            .header-content {
                position: relative;
                z-index: 2;
            }
            
            .company-name {
                font-size: 28px;
                font-weight: 700;
                margin-bottom: 8px;
                text-shadow: 2px 2px 4px rgba(0,0,0,0.3);
            }
            
            .voucher-type {
                font-size: 16px;
                opacity: 0.9;
                font-weight: 500;
            }
            
            .voucher-number {
                position: absolute;
                top: 30px;
                right: 30px;
                font-size: 14px;
                font-weight: 600;
                background: rgba(255,255,255,0.2);
                padding: 8px 16px;
                border-radius: 20px;
                backdrop-filter: blur(10px);
            }
            
            .voucher-body {
                padding: 40px;
            }
            
            .info-grid {
                display: grid;
                grid-template-columns: 1fr 1fr;
                gap: 30px;
                margin-bottom: 40px;
            }
            
            .info-section {
                background: #f8fafc;
                padding: 20px;
                border-radius: 12px;
                border-left: 4px solid ${isOwner ? '#22c55e' : '#3b82f6'};
            }
            
            .info-title {
                font-size: 14px;
                font-weight: 600;
                color: #64748b;
                margin-bottom: 12px;
                text-transform: uppercase;
                letter-spacing: 0.5px;
            }
            
            .info-content {
                color: #334155;
                font-weight: 500;
            }
            
            .amount-section {
                background: linear-gradient(135deg, #f1f5f9 0%, #e2e8f0 100%);
                border-radius: 16px;
                padding: 30px;
                margin: 30px 0;
                border: 2px solid ${isOwner ? '#22c55e' : '#3b82f6'};
            }
            
            .amount-title {
                font-size: 18px;
                font-weight: 700;
                color: #1e293b;
                margin-bottom: 20px;
                text-align: center;
            }
            
            .amount-row {
                display: flex;
                justify-content: space-between;
                align-items: center;
                padding: 12px 0;
                border-bottom: 1px solid #cbd5e1;
            }
            
            .amount-row:last-child {
                border-bottom: none;
                font-weight: 700;
                font-size: 18px;
                color: ${isOwner ? '#22c55e' : '#3b82f6'};
                border-top: 2px solid #cbd5e1;
                padding-top: 16px;
                margin-top: 8px;
            }
            
            .amount-label {
                color: #475569;
                font-weight: 500;
            }
            
            .amount-value {
                color: #1e293b;
                font-weight: 600;
            }
            
            .discount-section {
                background: #fef3cd;
                border: 1px solid #fbbf24;
                border-radius: 12px;
                padding: 20px;
                margin: 20px 0;
            }
            
            .discount-title {
                font-size: 16px;
                font-weight: 600;
                color: #92400e;
                margin-bottom: 12px;
            }
            
            .discount-item {
                display: flex;
                justify-content: space-between;
                padding: 8px 0;
                color: #78350f;
            }
            
            .owner-section {
                background: linear-gradient(135deg, #d1fae5 0%, #a7f3d0 100%);
                border: 1px solid #22c55e;
                border-radius: 12px;
                padding: 20px;
                margin: 20px 0;
            }
            
            .owner-title {
                font-size: 16px;
                font-weight: 600;
                color: #15803d;
                margin-bottom: 12px;
            }
            
            .status-badge {
                display: inline-block;
                padding: 6px 12px;
                border-radius: 20px;
                font-size: 12px;
                font-weight: 600;
                text-transform: uppercase;
                letter-spacing: 0.5px;
            }
            
            .status-paid {
                background: #dcfce7;
                color: #166534;
                border: 1px solid #22c55e;
            }
            
            .status-pending {
                background: #fef3c7;
                color: #92400e;
                border: 1px solid #f59e0b;
            }
            
            .status-overdue {
                background: #fee2e2;
                color: #991b1b;
                border: 1px solid #ef4444;
            }
            
            .notes-section {
                background: #f1f5f9;
                border-radius: 12px;
                padding: 20px;
                margin: 20px 0;
                border-left: 4px solid #64748b;
            }
            
            .footer {
                background: #1e293b;
                color: white;
                padding: 20px 40px;
                text-align: center;
                font-size: 12px;
                opacity: 0.8;
            }
            
            .watermark {
                position: absolute;
                bottom: 20px;
                right: 20px;
                opacity: 0.1;
                font-size: 64px;
                font-weight: 900;
                color: ${isOwner ? '#22c55e' : '#3b82f6'};
                transform: rotate(-45deg);
                pointer-events: none;
            }
            
            @media print {
                body {
                    background: white;
                    padding: 0;
                }
                
                .voucher-container {
                    box-shadow: none;
                    border-radius: 0;
                }
            }
        </style>
    </head>
    <body>
        <div class="voucher-container">
            <div class="voucher-header">
                <div class="header-content">
                    <div class="company-name">Sistema de Gestión Inmobiliaria</div>
                    <div class="voucher-type">
                        Comprobante de Pago ${isOwner ? 'para Propietario' : 'para Arrendatario'}
                    </div>
                </div>
                <div class="voucher-number">${payment?.voucherNumber || 'N/A'}</div>
            </div>
            
            <div class="voucher-body">
                <div class="info-grid">
                    <div class="info-section">
                        <div class="info-title">Información de la Propiedad</div>
                        <div class="info-content">
                            <strong>Dirección:</strong> ${propertyData?.address || 'No especificada'}<br>
                            <strong>Tipo:</strong> ${propertyData?.type || 'No especificado'}<br>
                            <strong>Código:</strong> ${propertyData?.code || 'N/A'}
                        </div>
                    </div>
                    
                    <div class="info-section">
                        <div class="info-title">${isOwner ? 'Información del Propietario' : 'Información del Arrendatario'}</div>
                        <div class="info-content">
                            <strong>Nombre:</strong> ${userData?.name || 'No especificado'}<br>
                            <strong>Email:</strong> ${userData?.email || 'No especificado'}<br>
                            <strong>Teléfono:</strong> ${userData?.phone || 'No especificado'}
                        </div>
                    </div>
                </div>
                
                <div class="info-grid">
                    <div class="info-section">
                        <div class="info-title">Detalles del Período</div>
                        <div class="info-content">
                            <strong>Período:</strong> ${payment?.month || 'No especificado'}<br>
                            <strong>Fecha Vencimiento:</strong> ${formatDate(payment?.dueDate)}<br>
                            <strong>Fecha de Pago:</strong> ${formatDate(payment?.paidDate) || 'Pendiente'}
                        </div>
                    </div>
                    
                    <div class="info-section">
                        <div class="info-title">Estado del Pago</div>
                        <div class="info-content">
                            <span class="status-badge status-${payment?.status}">
                                ${payment?.status === 'paid' ? 'Pagado' : 
                                  payment?.status === 'pending' ? 'Pendiente' : 
                                  payment?.status === 'overdue' ? 'Vencido' : 'Parcial'}
                            </span><br><br>
                            <strong>Método:</strong> ${payment?.paymentMethod === 'bank-transfer' ? 'Transferencia Bancaria' : 
                                                      payment?.paymentMethod === 'cash' ? 'Efectivo' : 
                                                      payment?.paymentMethod === 'check' ? 'Cheque' : 'Depósito'}
                        </div>
                    </div>
                </div>
                
                <div class="amount-section">
                    <div class="amount-title">Detalle de Montos</div>
                    
                    <div class="amount-row">
                        <span class="amount-label">Arriendo Mensual</span>
                        <span class="amount-value">${formatCurrency(rentAmount)}</span>
                    </div>
                    
                    <div class="amount-row">
                        <span class="amount-label">Gas</span>
                        <span class="amount-value">${formatCurrency(gasAmount)}</span>
                    </div>
                    
                    <div class="amount-row">
                        <span class="amount-label">Gastos Comunes</span>
                        <span class="amount-value">${formatCurrency(commonExpenses)}</span>
                    </div>
                    
                    <div class="amount-row">
                        <span class="amount-label">Agua</span>
                        <span class="amount-value">${formatCurrency(waterAmount)}</span>
                    </div>
                    
                    <div class="amount-row">
                        <span class="amount-label">Electricidad</span>
                        <span class="amount-value">${formatCurrency(electricityAmount)}</span>
                    </div>
                    
                    <div class="amount-row">
                        <span class="amount-label">Internet</span>
                        <span class="amount-value">${formatCurrency(internetAmount)}</span>
                    </div>
                    
                    <div class="amount-row">
                        <span class="amount-label">Subtotal</span>
                        <span class="amount-value">${formatCurrency(subtotal)}</span>
                    </div>
                </div>
                
                ${(tenantDiscounts?.length > 0 || (isOwner && (ownerDiscounts?.length > 0 || otherDiscounts?.length > 0))) ? `
                <div class="discount-section">
                    <div class="discount-title">Descuentos Aplicados</div>
                    ${tenantDiscounts?.map(discount => `
                        <div class="discount-item">
                            <span>${discount?.description || 'Descuento arrendatario'}</span>
                            <span>-${formatCurrency(discount?.amount)}</span>
                        </div>
                    `)?.join('')}
                    ${isOwner ? ownerDiscounts?.map(discount => `
                        <div class="discount-item">
                            <span>${discount?.description || 'Descuento propietario'}</span>
                            <span>-${formatCurrency(discount?.amount)}</span>
                        </div>
                    `)?.join('') : ''}
                    ${isOwner ? otherDiscounts?.map(discount => `
                        <div class="discount-item">
                            <span>${discount?.description || 'Otro descuento'}</span>
                            <span>-${formatCurrency(discount?.amount)}</span>
                        </div>
                    `)?.join('') : ''}
                </div>
                ` : ''}
                
                ${isOwner ? `
                <div class="owner-section">
                    <div class="owner-title">Cálculos para Propietario</div>
                    <div class="amount-row">
                        <span class="amount-label">Monto Bruto</span>
                        <span class="amount-value">${formatCurrency(finalAmount)}</span>
                    </div>
                    <div class="amount-row">
                        <span class="amount-label">Comisión Administración (8%)</span>
                        <span class="amount-value">-${formatCurrency(adminCommission)}</span>
                    </div>
                </div>
                ` : ''}
                
                <div class="amount-section">
                    <div class="amount-row">
                        <span class="amount-label">
                            ${isOwner ? 'MONTO NETO A RECIBIR' : 'TOTAL A PAGAR'}
                        </span>
                        <span class="amount-value">${formatCurrency(isOwner ? ownerNetAmount : finalAmount)}</span>
                    </div>
                </div>
                
                ${payment?.notes ? `
                <div class="notes-section">
                    <div class="info-title">Notas Adicionales</div>
                    <div class="info-content">${payment?.notes}</div>
                </div>
                ` : ''}
            </div>
            
            <div class="footer">
                Comprobante generado automáticamente el ${currentDate} | 
                ${isOwner ? 'Documento para uso interno del propietario' : 'Documento para arrendatario'} | 
                Sistema de Gestión Inmobiliaria
            </div>
            
            <div class="watermark">${isOwner ? 'OWNER' : 'TENANT'}</div>
        </div>
    </body>
    </html>
  `;
};

export const generateTenantVoucherPDF = async (payment, propertyData, tenantData) => {
  try {
    const html = generateVoucherHTML('tenant', payment, propertyData, tenantData);
    
    // Create a new window for the voucher
    const printWindow = window.open('', '_blank');
    if (!printWindow) {
      throw new Error('No se pudo abrir la ventana de impresión. Verifique que los pop-ups estén habilitados.');
    }
    
    printWindow?.document?.write(html);
    printWindow?.document?.close();
    
    // Wait for the content to load
    printWindow.onload = () => {
      setTimeout(() => {
        printWindow?.focus();
        printWindow?.print();
        
        // Close window after printing (optional)
        setTimeout(() => {
          printWindow?.close();
        }, 1000);
      }, 500);
    };
    
    return true;
  } catch (error) {
    console.error('Error generating tenant voucher:', error);
    throw error;
  }
};

export const generateOwnerVoucherPDF = async (payment, propertyData, ownerData) => {
  try {
    const html = generateVoucherHTML('owner', payment, propertyData, ownerData);
    
    // Create a new window for the voucher
    const printWindow = window.open('', '_blank');
    if (!printWindow) {
      throw new Error('No se pudo abrir la ventana de impresión. Verifique que los pop-ups estén habilitados.');
    }
    
    printWindow?.document?.write(html);
    printWindow?.document?.close();
    
    // Wait for the content to load
    printWindow.onload = () => {
      setTimeout(() => {
        printWindow?.focus();
        printWindow?.print();
        
        // Close window after printing (optional)
        setTimeout(() => {
          printWindow?.close();
        }, 1000);
      }, 500);
    };
    
    return true;
  } catch (error) {
    console.error('Error generating owner voucher:', error);
    throw error;
  }
};

// Preview functions for testing without printing
export const previewTenantVoucher = (payment, propertyData, tenantData) => {
  const html = generateVoucherHTML('tenant', payment, propertyData, tenantData);
  const newWindow = window.open('', '_blank');
  newWindow?.document?.write(html);
  newWindow?.document?.close();
};

export const previewOwnerVoucher = (payment, propertyData, ownerData) => {
  const html = generateVoucherHTML('owner', payment, propertyData, ownerData);
  const newWindow = window.open('', '_blank');
  newWindow?.document?.write(html);
  newWindow?.document?.close();
};